﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Jetson TK1 用 サポートパッド一覧
 * @detail  本ヘッダーファイルは Jetson TK1 向けにサポートしているパッドの一覧です。
 *          本ファイルと公開ヘッダーである、gpio_PadMap.h へ対応したいパッドを追記することで
 *          GPIO ライブラリで対応するパッドの追加、および外部の定義と内部の番号の紐付の変更が可能です。
 */

#pragma once

#include <nn/nn_Abort.h>

#include <nn/gpio/gpio_PadMap.h>

namespace nn{
namespace gpio{
namespace driver{
namespace detail{

/**
* @brief JetsonTK1 向けの GPIO パッド番号計算用マクロ
*
* @details JetsonTK1 向けの GPIO パッド番号計算用マクロです。
*/
#define NN_GPIO_GET_PAD_NUM(port, bit) ((port) * 8 + (bit))


/**
* @brief JetsonTK1 向けの GPIO ポート一覧
*
* @details JetsonTK1 向けの GPIO ポート一覧です。GPIO 番号計算用マクロを使用し、GPIO パッド番号を算出するために使用します。
*/
enum GpioPadPort
{
    GpioPadPort_PA = 0,
    GpioPadPort_PB,
    GpioPadPort_PC,
    GpioPadPort_PD,

    GpioPadPort_PE = 4,
    GpioPadPort_PF,
    GpioPadPort_PG,
    GpioPadPort_PH,

    GpioPadPort_PI = 8,
    GpioPadPort_PJ,
    GpioPadPort_PK,
    GpioPadPort_PL,

    GpioPadPort_PM = 12,
    GpioPadPort_PN,
    GpioPadPort_PO,
    GpioPadPort_PP,

    GpioPadPort_PQ = 16,
    GpioPadPort_PR,
    GpioPadPort_PS,
    GpioPadPort_PT,

    GpioPadPort_PU = 20,
    GpioPadPort_PV,
    GpioPadPort_PW,
    GpioPadPort_PX,

    GpioPadPort_PY = 24,
    GpioPadPort_PZ,
    GpioPadPort_PAA,
    GpioPadPort_PBB,

    GpioPadPort_PCC = 28,
    GpioPadPort_PDD,
    GpioPadPort_PEE,
    GpioPadPort_PFF,

    GpioPadPort_Num,
};


// サポートしている内部で使用するレジスタアクセス用 Gpio 番号の列挙体
// サポートするパッドが増える場合にはここにレジスタへのアクセス用の番号を追加してください。
enum InternalGpioPadNumber
{
    InternalGpioPadNumber_GPIO_PR0  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PR,  0),
    InternalGpioPadNumber_GPIO_PR2  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PR,  2),
    InternalGpioPadNumber_GPIO_PU0  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PU,  0),
    InternalGpioPadNumber_GPIO_PU1  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PU,  1),
    InternalGpioPadNumber_GPIO_PU2  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PU,  2),
    InternalGpioPadNumber_GPIO_PU3  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PU,  3),
    InternalGpioPadNumber_GPIO_PU4  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PU,  4),
    InternalGpioPadNumber_GPIO_PU5  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PU,  5),
    InternalGpioPadNumber_GPIO_PU6  = NN_GPIO_GET_PAD_NUM(GpioPadPort_PU,  6),
    InternalGpioPadNumber_None      = - 1,
};

// 外部の定義と内部のレジスタへのアクセス用番号の組み合わせの構造体の定義
struct PadMapCombination
{
    GpioPadName              publicPadName;
    InternalGpioPadNumber    internalPadName;
};

// 外部の定義と内部のレジスタへのアクセス用番号への変換一覧
// サポートするパッドの物理的なピン番号が変更になった場合や新しいパッドをサポートすることになった場合は
// ここに追加してください。
const PadMapCombination PadMapCombinationList[] =
{
    // 公開ヘッダーの定義                  内部のレジスタアクセス用番号
    { GpioPadName_AudioPowerEnable,        InternalGpioPadNumber_GPIO_PR2},
    { GpioPadName_EnablePowerToTheSdCard,  InternalGpioPadNumber_GPIO_PR0},
    { GpioPadName_BluetoothPowerEnable,    InternalGpioPadNumber_None},
};

// 公開しているパッドの数
const int NumberOfSupportPublicPads = sizeof(PadMapCombinationList) / sizeof(PadMapCombinationList[0]);

} // detail
} // driver
} // gpio
} // nn
