﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/svc/svc_HardwareParamsSelect.h>

#include <nn/gpio/detail/gpio_Log.h>
#include <nn/gpio/driver/gpio_Lib.h>
#include <nn/gpio/driver/gpio_PadAccessor.h>
#include <nn/gpio/gpio_PadMap.h>

#include "gpio_DdUtil.h"
#include "gpio_Driver-soc.tegra.h"
#include "gpio_Interrupt-soc.tegra.h"
#include "gpio_EventHolder-soc.tegra.h"


#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/spl/spl_Api.h>
#include "gpio_InitialConfig-hardware.icosa.h"
#include "gpio_InitialConfig-hardware.copper.h"
#include "gpio_InitialConfig-hardware.iowa.h"
#include "gpio_InitialConfig-hardware.hoag.h"
#include <nne/wec/wec.h>
#include "gpio_WakePinSetting-hardware.nx.h"
#include "gpio_InitialWakePinSetting-hardware.icosa.h"
#include "gpio_InitialWakePinSetting-hardware.copper.h"
#include "gpio_InitialWakePinSetting-hardware.iowa.h"
#include "gpio_InitialWakePinSetting-hardware.hoag.h"
#include "gpio_DisableWakePinSettingWithoutPmic-hardware.nx.h"
#endif

namespace nn{
namespace gpio{
namespace driver{
namespace detail{

namespace
{

// 公開 API の PadName を内部で使用するレジスタへのアクセス用番号に変換する関数
nn::gpio::driver::detail::InternalGpioPadNumber ConvertPadNumberToInternalPadNumber(nn::gpio::GpioPadName pad) NN_NOEXCEPT
{
    for(int i = 0; i<nn::gpio::driver::detail::NumberOfSupportPublicPads; i++)
    {
        if(nn::gpio::driver::detail::PadMapCombinationList[i].publicPadName == pad)
        {
            if(nn::gpio::driver::detail::PadMapCombinationList[i].internalPadName == nn::gpio::driver::detail::InternalGpioPadNumber_None)
            {
                // 対応するパッド番号に対するモジュールがボードに乗っていないので、ASSERT する。
                NN_SDK_ASSERT(false, "This board does not have PadName (%d)\n", pad);
            }
            else
            {
                return nn::gpio::driver::detail::PadMapCombinationList[i].internalPadName;
            }
        }
    }

    NN_ABORT("Unsupport Pad Name(%d) is selected\n", pad);
}

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
nn::spl::HardwareType GetHardwareType() NN_NOEXCEPT
{
    nn::Bit64 config;
    nn::spl::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GetConfig(&config, nn::spl::ConfigItem_HardwareType));
    nn::spl::Finalize();
    return static_cast<nn::spl::HardwareType>(config);
}
#endif

}

void Driver::Initialize() NN_NOEXCEPT
{
    // GPIO コントローラーの割り込みをセットアップし、割り込み監視用のスレッドを立ち上げる
    auto& interruptNotifier = detail::InterruptNotifier::GetInstance();
    interruptNotifier.Initialize();

    // Sleep　中の GPIO 設定はこの時点ではすべて触らない設定に
    for(int i = 0; i < GpioPadPort_Num; i++)
    {
        m_GpioValuesForSleep[i].IsForceSet = 0;
    }
#if defined NN_BUILD_CONFIG_HARDWARE_NX
    nne::wec::Initialize();
#endif
    m_InitializeCount++;
}

void Driver::Finalize() NN_NOEXCEPT
{
    // Initialize された分、Finalize が呼ばれたら、全セッションをクローズし、割り込み管理用スレッドも破棄する。
    NN_SDK_ASSERT(!(m_InitializeCount == 0), "initialize されていません。");

    auto& interruptNotifier = detail::InterruptNotifier::GetInstance();
    std::lock_guard<nn::os::Mutex> lock(*interruptNotifier.GetInterruptEventMutexPointer());

    m_InitializeCount--;
    if(m_InitializeCount == 0)
    {
        interruptNotifier.Finalize();

        for (auto&& e : m_OpenSessionList)
        {
            auto& eventHolder = e.eventHolder;

            // Bound 済みのものが残っていた場合
            if (eventHolder.IsBoundEvent())
            {
                // EventHolder を紐付済みリストから外す
                auto& interruptNotifier = detail::InterruptNotifier::GetInstance();
                interruptNotifier.DeleteEventHolderFromBoundList(&eventHolder);

                // 割り込みを disable にする。
                nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_ENB, static_cast<InternalGpioPadNumber>(e.padNumber), m_GpioVirtualAddress);
                int bitPosition = ConvertInternalGpioPadNumberToBitPosition(static_cast<InternalGpioPadNumber>(e.padNumber));
                SetBitForTegraMaskedWrite(0, bitPosition, accessAddress);
                DummyRead(accessAddress);

                // 渡されたイベントを破棄する。
                eventHolder.DestroyEvent();

                // イベントの登録を外す。
                eventHolder.DetachEvent();
            }
        }

    }
}

// OpenSession と OpenSessionForDev 共通の処理
void Driver::OpenSessionImpl(GpioPadSession* pOutSession, InternalGpioPadNumber padNumber) NN_NOEXCEPT
{
    // すでにそのパッド番号に対するセッションがオープンされているかチェック
    for (auto&& e : m_OpenSessionList)
    {
        if (e.padNumber == padNumber)
        {
            NN_SDK_ASSERT("すでに入力されたパッドに対するセッションがオープンしています");
        }
    }

    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pOutSession->_impl).impl;

    // コンストラクタを呼ぶために placement new する
    auto pSession = new (pSessionImpl) GpioPadSessionImpl(padNumber);

    // pOutSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    nn::Bit32* accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_CNF, padNumber, m_GpioVirtualAddress);
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(padNumber);

    // GPIO_CNF へ Per-Pin Mask Write で値を書きこみ、GPIO としてセットする。
    //(Tegra_K1_TRM_DP06905001v02p.pdf p.278)
    SetBitForTegraMaskedWrite( 1, bitPosition, accessAddress);
    DummyRead(accessAddress);

    // List に Session を追加する
    m_OpenSessionList.push_back(*pSession);
}


void Driver::OpenSession(GpioPadSession* pOutSession, GpioPadName pad) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSession);

    // GPIO の名前から内部用の番号へ変換する
    InternalGpioPadNumber internalPadNum = ConvertPadNumberToInternalPadNumber(pad);
    OpenSessionImpl(pOutSession, internalPadNum);
}


void Driver::OpenSessionDev(GpioPadSession* pOutSession, int padNumber) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSession);
    OpenSessionImpl(pOutSession, static_cast<InternalGpioPadNumber>(padNumber));
}



void Driver::CloseSession(GpioPadSession* pSession) NN_NOEXCEPT
{
    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pSession->_impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(*pSessionImpl), "セッションがオープンされていません。 PadNumber = %d\n", pSessionImpl->padNumber);

    // Unbind されていないのであれば、Unbind する
    if (pSessionImpl->eventHolder.IsBoundEvent())
    {
        UnbindInterrupt(pSession);
    }

    // Session をリストから外す
    // TODO : mutex でガード
    m_OpenSessionList.erase(m_OpenSessionList.iterator_to(*pSessionImpl));

    // OpenSession で placement new しているので、ここで明示的にデストラクタを呼ぶ
    pSessionImpl->~GpioPadSessionImpl();
}


void Driver::SetDirection(const GpioPadSession& session, Direction direction) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // ディレクションを変える機能を持っているか確認
    if(direction == Direction_Input)
    {
        NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_DataInput, "この GPIO パッドの向きを変えることはできません。");
    }
    else
    {
        NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_DataOutput, "この GPIO パッドの向きを変えることはできません。");
    }

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // NX 以外の時に Pinmux を設定する
#ifndef NN_BUILD_CONFIG_HARDWARE_NX
    m_PinmuxAccessor.SetGpioPinmux(internalPadNum, direction);
#endif

    // pOutSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_OE, internalPadNum, m_GpioVirtualAddress);

    // GPIO_OE を output の時は DRIVENに、input の時は TRISTATE にする
    // (Tegra_K1_TRM_DP06905001v02p.pdf p.268 の回路図参照)
    SetBitForTegraMaskedWrite(direction, bitPosition, accessAddress);
    DummyRead(accessAddress);
}

Direction Driver::GetDirection(const GpioPadSession& session ) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_OE, internalPadNum, m_GpioVirtualAddress);

    if(GetBit<Bit32>(accessAddress, bitPosition))
    {
        return Direction_Output;
    }
    else
    {
        return Direction_Input;
    }
}

void Driver::SetInterruptMode(const GpioPadSession& session, InterruptMode mode ) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // 割り込みモード設定機能を持っているか確認
    NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_InterruptMode, "この GPIO パッドは割り込みモードの設定は出来ません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換(ここは Read-Modify-Write)
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_LVL, internalPadNum, m_GpioVirtualAddress);

    // 各割り込みモードに対応した Bit 列をシフトして設定
    switch(mode)
    {
    case InterruptMode_LowLevel:
        WriteMasked32( accessAddress,
                       (InternalInterruptMode_LowLevel << bitPosition),
                       (InternalInterruptMode_BitMask << bitPosition));
        break;

    case InterruptMode_HighLevel:
        WriteMasked32( accessAddress,
                       (InternalInterruptMode_HighLevel << bitPosition),
                       (InternalInterruptMode_BitMask << bitPosition));
        break;

    case InterruptMode_RisingEdge:
        WriteMasked32( accessAddress,
                       (InternalInterruptMode_RisingEdge << bitPosition),
                       (InternalInterruptMode_BitMask << bitPosition));
        break;

    case InterruptMode_FallingEdge:
        WriteMasked32( accessAddress,
                       (InternalInterruptMode_FallingEdge << bitPosition),
                       (InternalInterruptMode_BitMask << bitPosition));
        break;

    case InterruptMode_AnyEdge:
        WriteMasked32( accessAddress,
                       (InternalInterruptMode_AnyEdge << bitPosition),
                       (InternalInterruptMode_BitMask << bitPosition));
        break;

    default:
        NN_SDK_ASSERT("GPIO ライブラリが対応していない割り込み方法が指定されています");
        break;
    }
    DummyRead(accessAddress);
}

InterruptMode Driver::GetInterruptMode(const GpioPadSession& session) NN_NOEXCEPT
{

    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // 割り込みモード設定機能を持っているか確認
    NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_InterruptMode, "この GPIO パッドは割り込みモードの設定は出来ません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_LVL, internalPadNum, m_GpioVirtualAddress);

    // レジスタに書かれた InterruptMode を bitPosition 分シフトした値で InterruptMode を判別
    nn::Bit32 shiftedInterruptModeRegister = *accessAddress >> bitPosition;

    switch(shiftedInterruptModeRegister & InternalInterruptMode_BitMask)
    {
    case InternalInterruptMode_LowLevel:
        return InterruptMode_LowLevel;

    case InternalInterruptMode_HighLevel:
        return InterruptMode_HighLevel;

    case InternalInterruptMode_RisingEdge:
        return InterruptMode_RisingEdge;

    case InternalInterruptMode_FallingEdge:
        return InterruptMode_FallingEdge;

    case InternalInterruptMode_AnyEdge:
        return InterruptMode_AnyEdge;

    default: NN_UNEXPECTED_DEFAULT;
    }

    // ここにはこない想定
    NN_ABORT("GPIO ライブラリが対応していない割り込み方法が指定されています");
}

void Driver::SetInterruptEnableToRegister(const GpioPadSession& session, bool enable ) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_ENB, internalPadNum, m_GpioVirtualAddress);

    // GPIO_INT_ENB を叩いて割り込みを設定する
    if(enable)
    {
        SetBitForTegraMaskedWrite(1, bitPosition, accessAddress);

    }
    else
    {
        SetBitForTegraMaskedWrite(0, bitPosition, accessAddress);
    }
    DummyRead(accessAddress);
}

void Driver::SetInterruptEnable(GpioPadSession* pSession, bool enable ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(*detail::InterruptNotifier::GetInstance().GetInterruptEventMutexPointer());

    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pSession->_impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(*pSessionImpl), "セッションがオープンされていません。PadNumber = %d\n", pSessionImpl->padNumber);

    // 割り込み機能を持っているか確認
    NN_SDK_ASSERT(pSessionImpl->padFunction & AccessType_InterruptEnable, "この GPIO パッドは割り込みを機能を使用出来ません。");

    // 内部の割り込み管理フラグを変更する
    pSessionImpl->eventHolder.SetInternalInterruptFlag(enable);

    // 内部の割り込みフラグが enable かつ Bind 済みならレジスタを叩いて割り込みを enable にする
    if (enable && pSessionImpl->eventHolder.IsBoundEvent())
    {
        SetInterruptEnableToRegister(*pSession, true);
    }
    else
    {
        SetInterruptEnableToRegister(*pSession, false);
    }
}

bool Driver::GetInterruptEnable(const GpioPadSession& session) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // 割り込み機能を持っているか確認
    NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_InterruptEnable, "この GPIO パッドは割り込みを機能を使用出来ません。");

    // 内部の割り込み管理フラグを返す
    return sessionImpl.eventHolder.IsInternalInterruptEnable();
}

InterruptStatus Driver::GetInterruptStatus(const GpioPadSession& session) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // 割り込み機能を持っているか確認
    NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_InterruptEnable, "この GPIO パッドは割り込みを機能を使用出来ません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_STA, internalPadNum, m_GpioVirtualAddress);

    // GPIO_INT_STA を叩いて割り込みが発生しているかどうかを返す。
    if(GetBit<Bit32>(accessAddress, bitPosition))
    {
        return InterruptStatus_Active;
    }
    else
    {
        return InterruptStatus_Inactive;
    }
}

void Driver::ClearInterruptStatus(const GpioPadSession& session) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // 割り込み機能を持っているか確認
    NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_InterruptEnable, "この GPIO パッドは割り込みを機能を使用出来ません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_CLR, internalPadNum, m_GpioVirtualAddress);

    // GPIO_INT_CLR を叩いて割り込みをクリアする
    SetBit<Bit32>(accessAddress, 1, bitPosition);
    DummyRead(accessAddress);
}


GpioValue Driver::GetValue(const GpioPadSession& session) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // Input 機能を持っているか確認
    NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_DataInput, "この GPIO パッドは入力に使用することはできません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // InternalGpioPadNumber を アクセスするアドレス・ビットへ変換
    int bitPosition      = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_IN, internalPadNum, m_GpioVirtualAddress);

    if(GetBit<Bit32>(accessAddress, bitPosition))
    {
        return GpioValue_High;
    }
    else
    {
        return GpioValue_Low;
    }
}


void Driver::SetValue(const GpioPadSession& session, GpioValue value ) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // Output 機能を持っているか確認
    NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_DataOutput, "この GPIO パッドは出力に使用することはできません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // pOutSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int          bitPosition   = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_OUT, internalPadNum, m_GpioVirtualAddress);

    // GPIO_OUT へ Per-Pin Mask Write で値を書きこみ、出力を変える
    SetBitForTegraMaskedWrite(value, bitPosition, accessAddress);
    DummyRead(accessAddress);
}

void Driver::SetValueForSleepState(const GpioPadSession& session, GpioValue value ) NN_NOEXCEPT
{
    auto& sessionImpl = nn::util::Get<GpioPadSessionImplPadded>(session._impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(sessionImpl), "セッションがオープンされていません。PadNumber = %d\n", sessionImpl.padNumber);

    // Output 機能を持っているか確認
    NN_SDK_ASSERT(sessionImpl.padFunction & AccessType_DataOutput, "この GPIO パッドは出力に使用することはできません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(sessionImpl.padNumber);

    // pOutSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int          bitPosition   = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    int          portNumber    = ConvertInternalGpioPadNumberToPortNumber(internalPadNum);

    // 値を保存しておく
    m_GpioValuesForSleep[portNumber].IsForceSet |= 1 << bitPosition;
    m_GpioValuesForSleep[portNumber].out        &= ~(1 << bitPosition);
    m_GpioValuesForSleep[portNumber].out        |= (value == GpioValue_High) ? (1 << bitPosition) : 0;
}


void Driver::SetDebounceEnabled(GpioPadSession* pSession, bool isEnable) NN_NOEXCEPT
{
    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pSession->_impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(*pSessionImpl), "セッションがオープンされていません。PadNumber = %d\n", pSessionImpl->padNumber);

    // チャタリング防止機能を持っているか確認
    NN_SDK_ASSERT(pSessionImpl->padFunction & AccessType_DebounceFunction, "この GPIO パッドはチャタリング防止機能を使用出来ません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(pSessionImpl->padNumber);

    // pSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int          bitPosition   = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_DB_CTRL, internalPadNum, m_GpioVirtualAddress);

    SetBitForTegraMaskedWrite(isEnable ? 1 : 0, bitPosition, accessAddress);
    DummyRead(accessAddress);
}

bool Driver::GetDebounceEnabled(GpioPadSession* pSession) NN_NOEXCEPT
{
    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pSession->_impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(*pSessionImpl), "セッションがオープンされていません。PadNumber = %d\n", pSessionImpl->padNumber);

    // チャタリング防止機能を持っているか確認
    NN_SDK_ASSERT(pSessionImpl->padFunction & AccessType_DebounceFunction, "この GPIO パッドはチャタリング防止機能を使用出来ません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(pSessionImpl->padNumber);

    // pSession 内の GPIO 番号を アクセスするアドレス・ビットへ変換
    int          bitPosition   = ConvertInternalGpioPadNumberToBitPosition(internalPadNum);
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_DB_CTRL, internalPadNum, m_GpioVirtualAddress);

    return GetBit<Bit32>(accessAddress, bitPosition);
}

void Driver::SetDebounceTime(GpioPadSession* pSession, int msecTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(0 <= msecTime && msecTime <= 128);

    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pSession->_impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(*pSessionImpl), "セッションがオープンされていません。PadNumber = %d\n", pSessionImpl->padNumber);

    // チャタリング防止機能を持っているか確認
    NN_SDK_ASSERT(pSessionImpl->padFunction & AccessType_DebounceFunction, "この GPIO パッドはチャタリング防止機能を使用出来ません。");

#if defined NN_BUILD_CONFIG_HARDWARE_NX
    // チャタリング防止時間の設定は特定ピンだけが設定可能
    NN_SDK_ASSERT(pSessionImpl->padNumber == 38  || // ExtconDetS
                  pSessionImpl->padNumber == 62  || // ExtconDetU
                  pSessionImpl->padNumber == 174 || // CodecHpDetIrq
                  pSessionImpl->padNumber == 190 || // ButtonVolUp
                  pSessionImpl->padNumber == 191 || // ButtonVolDn
                  pSessionImpl->padNumber == 201 || // SdCd
                  pSessionImpl->padNumber == 203,   // SdWp for testing
        "This pad (%d) cannot change decounde time. Please contact the person in charge of gpio library\n", pSessionImpl->padNumber
    );
#endif

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(pSessionImpl->padNumber);

    // pSession 内の GPIO 番号を アクセスするアドレスへ変換
    nn::Bit32*   accessAddress       = GetGpioRegAccessAddress(GpioRegisterType_GPIO_DB_CNT, internalPadNum, m_GpioVirtualAddress);

    // Debounce 時間を書き込む(このピンが属するポートすべてに影響する)
    WriteMasked32(accessAddress, msecTime, 0xff );
    DummyRead(accessAddress);
}

int Driver::GetDebounceTime(GpioPadSession* pSession) NN_NOEXCEPT
{
    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pSession->_impl).impl;

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(*pSessionImpl), "セッションがオープンされていません。PadNumber = %d\n", pSessionImpl->padNumber);

    // チャタリング防止機能を持っているか確認
    NN_SDK_ASSERT(pSessionImpl->padFunction & AccessType_DebounceFunction, "この GPIO パッドはチャタリング防止機能を使用出来ません。");

    // 内部で使用する padNumber へキャスト
    InternalGpioPadNumber internalPadNum = static_cast<InternalGpioPadNumber>(pSessionImpl->padNumber);

    // pSession 内の GPIO 番号を アクセスするアドレスへ変換
    nn::Bit32*   accessAddress       = GetGpioRegAccessAddress(GpioRegisterType_GPIO_DB_CNT, internalPadNum, m_GpioVirtualAddress);

    return *accessAddress & 0xff;
}


Result Driver::BindInterrupt(nn::os::SystemEventType* pEvent, GpioPadSession* pSession) NN_NOEXCEPT
{
    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pSession->_impl).impl;

    // Initialize 済みか確認
    NN_SDK_ASSERT(m_InitializeCount > 0,"ライブラリが Initialize されていません。");

    // セッションがオープンしているかどうか確認
    NN_SDK_ASSERT(IsSessionOpened(*pSessionImpl), "セッションがオープンされていません。PadNumber = %d\n", pSessionImpl->padNumber);

    std::lock_guard<nn::os::Mutex> lock(*detail::InterruptNotifier::GetInstance().GetInterruptEventMutexPointer());

    // すでにバインド済みなら登録せずにエラーを返す。
    if (pSessionImpl->eventHolder.IsBoundEvent())
    {
        return nn::gpio::ResultAlreadyBound();
    }

    // SystemEvent を初期化する。
    // DFC で使う場合で最適化したい場合は、プロセス内通信として初期化した方がよい。
    nn::os::CreateSystemEvent(pEvent, nn::os::EventClearMode_ManualClear, true);

    // イベントをセッション内に登録
    pSessionImpl->eventHolder.AttachEvent(pEvent);

    // InterruptNotifier で管理している紐付済みセッションに登録する
    auto& interruptNotifier = detail::InterruptNotifier::GetInstance();
    nn::Result result = interruptNotifier.AddEventHolderToBoundList(&pSessionImpl->eventHolder);
    if(!result.IsSuccess())
    {
        return result;
    }

    // 指定のパッドに Event が Bind 済みかつ、内部の割り込みが enable ならレジスタを叩いて実際の割り込みを enable にする。
    if (pSessionImpl->eventHolder.IsInternalInterruptEnable())
    {
        SetInterruptEnableToRegister(*pSession, true);
    }
    else
    {
        SetInterruptEnableToRegister(*pSession, false);
    }

    return nn::ResultSuccess();
}


void Driver::UnbindInterrupt(GpioPadSession* pSession) NN_NOEXCEPT
{
    auto* pSessionImpl = &nn::util::Get<GpioPadSessionImplPadded>(pSession->_impl).impl;

    // Initialize 済みか確認
    NN_SDK_ASSERT(m_InitializeCount > 0,"ライブラリが Initialize されていません。");

    // Bind 済みかチェック
    NN_SDK_ASSERT(pSessionImpl->eventHolder.IsBoundEvent());

    std::lock_guard<nn::os::Mutex> lock(*detail::InterruptNotifier::GetInstance().GetInterruptEventMutexPointer());

    // EventHolder を紐付済みリストから外す
    auto& interruptNotifier = detail::InterruptNotifier::GetInstance();
    interruptNotifier.DeleteEventHolderFromBoundList(&pSessionImpl->eventHolder);

    // 割り込みを disable にする。
    nn::Bit32*   accessAddress = GetGpioRegAccessAddress(GpioRegisterType_GPIO_INT_ENB, static_cast<InternalGpioPadNumber>(pSessionImpl->padNumber), m_GpioVirtualAddress);
    int bitPosition = ConvertInternalGpioPadNumberToBitPosition(static_cast<InternalGpioPadNumber>(pSessionImpl->padNumber));
    SetBitForTegraMaskedWrite(0, bitPosition, accessAddress);
    DummyRead(accessAddress);

    // 渡されたイベントを破棄する。
    pSessionImpl->eventHolder.DestroyEvent();

    // イベントの登録を外す。
    pSessionImpl->eventHolder.DetachEvent();
}

bool Driver::IsWakeEventActive(GpioPadName name) NN_NOEXCEPT
{
    return m_WakeBitFlag[name];
}

WakeBitFlag Driver::GetWakeEventActiveFlagSet() NN_NOEXCEPT
{
    return m_WakeBitFlag;
}

void Driver::SetWakeEventActiveFlagSetForDebug(GpioPadName name, bool isEnabled) NN_NOEXCEPT
{
    m_WakeBitFlagForDebug[name] = isEnabled;
}

void Driver::Suspend() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE("Store All Gpio Pin Register\n");

    // レジスタの値をメモリへ保存
    int port = 0;
    for (auto& i : m_EscapedRegisterValue)
    {
        i.conf   = ReadBitsFromPort<nn::Bit16>(GpioRegisterType_GPIO_CNF,     port, m_GpioVirtualAddress);
        i.oe     = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_OE,      port, m_GpioVirtualAddress);
        i.out    = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_OUT,     port, m_GpioVirtualAddress);
        i.intEnb = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_INT_ENB, port, m_GpioVirtualAddress);
        i.intLvl = ReadBitsFromPort<nn::Bit32>(GpioRegisterType_GPIO_INT_LVL, port, m_GpioVirtualAddress);
        i.dbCtrl = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_DB_CTRL, port, m_GpioVirtualAddress);
        i.dbCnt  = ReadBitsFromPort<nn::Bit8>( GpioRegisterType_GPIO_DB_CNT,  port, m_GpioVirtualAddress);

        /*
        NN_DETAIL_GPIO_INFO("---------------port = %d--------------------\n", port);
        NN_DETAIL_GPIO_INFO("RegValue conf = %x\n", ReadBitsFromPort<nn::Bit16>(GpioRegisterType_GPIO_CNF, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue oe   = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_OE, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue out  = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_OUT, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue intEnb = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_INT_ENB, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue intLvl = %x\n", ReadBitsFromPort<nn::Bit32>(GpioRegisterType_GPIO_INT_LVL, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue dbCtrl = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_DB_CTRL, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue dbCnt = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_DB_CNT, port, m_GpioVirtualAddress));
        */

        // 保存し終えたピンはすべて割り込みを disable に
        nn::Bit8 disable = 0x00;
        WriteBitsFromPort<nn::Bit8>( disable, GpioRegisterType_GPIO_INT_ENB, port, m_GpioVirtualAddress);

        // IsForceSet が enable のものは出力する値を設定する
        if(m_GpioValuesForSleep[port].IsForceSet)
        {
            i.out = (i.out & ~m_GpioValuesForSleep[port].IsForceSet) | (m_GpioValuesForSleep[port].out & m_GpioValuesForSleep[port].IsForceSet);
            WriteBitsFromPort<nn::Bit8>(i.out, GpioRegisterType_GPIO_OUT, port, m_GpioVirtualAddress);
            NN_DETAIL_GPIO_TRACE("Force Set GPIO value. port = %d, value = %x\n", port, m_GpioValuesForSleep[port].out);
        }

        // 書き込み確定のための DummyRead
        DummyRead(GetGpioRegAccessAddress(GpioRegisterType_GPIO_CNF, static_cast<InternalGpioPadNumber>(port * 8), m_GpioVirtualAddress));

        port++;
    }
}

// Suspend と SuspendLow の役割分担は http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=166038312 を参照
void Driver::SuspendLow() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    // 要因を取り始める前に古い要因はクリアしておく
    nne::wec::ClearWakeEvents();

    // 充電 IC の HiZ モードの切り替えをセキュアモニタでおこなう際に割り込みが入ってしまうためセキュアモニタでスリープ直前に wake 要因を取得できるようにする（SIGLO-38305）
    // // これ以降に発生した wake 要因を取得できるようにする。呼ばない場合は WakeDetectMode_HwLpsEntry として要因が取得される
    // NN_DETAIL_GPIO_TRACE("Start wake pin detect\n");
    // nne::wec::WecLpsEntryPrepare(nne::wec::WakeDetectMode_WecLpsEntry);
#endif
}

void Driver::Resume() NN_NOEXCEPT
{
    NN_DETAIL_GPIO_TRACE("Restore All Gpio Interrupt Pins\n");

    int port = 0;

    for (auto& i : m_EscapedRegisterValue)
    {
        WriteBitsFromPort<nn::Bit8>(i.intEnb, GpioRegisterType_GPIO_INT_ENB, port, m_GpioVirtualAddress);

        //NN_DETAIL_GPIO_INFO("RegValue intEnb = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_INT_ENB, port, m_GpioVirtualAddress));

        port++;
    }
}

// Resume と ResumeLow の役割分担は http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=166038312 を参照
void Driver::ResumeLow() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    nne::wec::WakeEvent wakeEventList[nne::wec::WakeEvent_Num];
    int activeWakeEventCount;

    // ここまでに取れた wake 要因を保存する
    nne::wec::WecLpsExitRestore();

    // WakeEvent の Active 状態を読みだす
    nne::wec::GetActiveWakeEvents(wakeEventList, &activeWakeEventCount, nne::wec::WakeEvent_Num);

    // BitFlag に設定する
    m_WakeBitFlag.Reset();
    for (int i = 0; i < activeWakeEventCount; i++)
    {
        for (int j = 0; j < NumberOfSupportPublicPads; j++)
        {
            if (wakeEventList[i] == PadMapCombinationList[j].wakeEvent)
            {
                NN_DETAIL_GPIO_TRACE("Active Wake Event (wec : %d, gpio : %d)\n", wakeEventList[i], PadMapCombinationList[j].publicPadName);
                m_WakeBitFlag[PadMapCombinationList[j].publicPadName] = true;
            }
        }
    }

    // Debug 用の設定との or を取る
    m_WakeBitFlag |= m_WakeBitFlagForDebug;
#endif

    NN_DETAIL_GPIO_TRACE("Restore All Gpio Pin Register\n");

    // レジスタの値をメモリから復旧
    int port = 0;
    for (auto& i : m_EscapedRegisterValue)
    {
        WriteBitsFromPort<nn::Bit16>(i.conf, GpioRegisterType_GPIO_CNF, port, m_GpioVirtualAddress);
        WriteBitsFromPort<nn::Bit8>(i.oe, GpioRegisterType_GPIO_OE, port, m_GpioVirtualAddress);
        WriteBitsFromPort<nn::Bit8>(i.out, GpioRegisterType_GPIO_OUT, port, m_GpioVirtualAddress);
        WriteBitsFromPort<nn::Bit32>(i.intLvl, GpioRegisterType_GPIO_INT_LVL, port, m_GpioVirtualAddress);
        WriteBitsFromPort<nn::Bit8>(i.dbCtrl, GpioRegisterType_GPIO_DB_CTRL, port, m_GpioVirtualAddress);
        WriteBitsFromPort<nn::Bit8>(i.dbCnt, GpioRegisterType_GPIO_DB_CNT, port, m_GpioVirtualAddress);

        // XXX: Pinmux 書き込み前に InterruptEnable を書き戻すと意図してない割り込みが発生するため、別関数で対応する。
        //WriteBitsFromPort<nn::Bit8>( i.intEnb, GpioRegisterType_GPIO_INT_ENB, port, m_GpioVirtualAddress);

        /*
        NN_DETAIL_GPIO_INFO("---------------port = %d--------------------\n", port);
        NN_DETAIL_GPIO_INFO("RegValue conf = %x\n", ReadBitsFromPort<nn::Bit16>(GpioRegisterType_GPIO_CNF, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue oe   = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_OE, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue out  = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_OUT, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue intEnb = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_INT_ENB, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue intLvl = %x\n", ReadBitsFromPort<nn::Bit32>(GpioRegisterType_GPIO_INT_LVL, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue dbCtrl = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_DB_CTRL, port, m_GpioVirtualAddress));
        NN_DETAIL_GPIO_INFO("RegValue dbCnt = %x\n", ReadBitsFromPort<nn::Bit8>(GpioRegisterType_GPIO_DB_CNT, port, m_GpioVirtualAddress));
        */

        port++;
    }

    // 書き込み確定のための DummyRead
    DummyRead(GetGpioRegAccessAddress(GpioRegisterType_GPIO_CNF, static_cast<InternalGpioPadNumber>(0), m_GpioVirtualAddress));
}


void Driver::SetInitialGpioConfig() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    int                configCount;
    GpioInitialConfig* pConfigList;

    switch (GetHardwareType())
    {
    case nn::spl::HardwareType_Copper:
        pConfigList = detail::initGpioConfigListForCopper;
        configCount = sizeof(detail::initGpioConfigListForCopper) / sizeof(detail::initGpioConfigListForCopper[0]);
        NN_DETAIL_GPIO_TRACE("Set Gpio Initial Config For Copper\n");
        break;

    case nn::spl::HardwareType_Iowa:
        pConfigList = detail::initGpioConfigListForIowa;
        configCount = sizeof(detail::initGpioConfigListForIowa) / sizeof(detail::initGpioConfigListForIowa[0]);
        NN_DETAIL_GPIO_TRACE("Set Gpio Initial Config For Iowa\n");
        break;

    case nn::spl::HardwareType_Hoag:
        pConfigList = detail::initGpioConfigListForHoag;
        configCount = sizeof(detail::initGpioConfigListForHoag) / sizeof(detail::initGpioConfigListForHoag[0]);
        NN_DETAIL_GPIO_TRACE("Set Gpio Initial Config For Hoag\n");
        break;

    case nn::spl::HardwareType_Icosa:
    default: // TORIAEZU
        pConfigList = detail::initGpioConfigListForIcosa;
        configCount = sizeof(detail::initGpioConfigListForIcosa) / sizeof(detail::initGpioConfigListForIcosa[0]);
        NN_DETAIL_GPIO_TRACE("Set Gpio Initial Config\n");
        break;
    }

    for (int i = 0; i < configCount; i++)
    {
        GpioPadSession session;
        OpenSession(&session, pConfigList[i].name);
        SetDirection(session, pConfigList[i].direction);
        if (pConfigList[i].direction == Direction_Output)
        {
            SetValue(session, pConfigList[i].value);
        }
        CloseSession(&session);
    }
#else
    NN_DETAIL_GPIO_TRACE("Not set Gpio Initial Config\n");
#endif
}

void Driver::SetInitialWakePinConfig() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    int            configCount;
    WakePinConfig* pConfigList;

    nne::wec::Initialize();

    switch (GetHardwareType())
    {
        case nn::spl::HardwareType_Copper:
            pConfigList = detail::initWakePinConfigListForCopper;
            configCount = sizeof(detail::initWakePinConfigListForCopper) / sizeof(detail::initWakePinConfigListForCopper[0]);
            break;

        case nn::spl::HardwareType_Iowa:
            pConfigList = detail::initWakePinConfigListForIowa;
            configCount = sizeof(detail::initWakePinConfigListForIowa) / sizeof(detail::initWakePinConfigListForIowa[0]);
            break;

        case nn::spl::HardwareType_Hoag:
            pConfigList = detail::initWakePinConfigListForHoag;
            configCount = sizeof(detail::initWakePinConfigListForHoag) / sizeof(detail::initWakePinConfigListForHoag[0]);
            break;

        case nn::spl::HardwareType_Icosa:
        default: // TORIAEZU
            pConfigList = detail::initWakePinConfigListForIcosa;
            configCount = sizeof(detail::initWakePinConfigListForIcosa) / sizeof(detail::initWakePinConfigListForIcosa[0]);
            break;
    }

    for(int i = 0; i < configCount; i++)
    {
        SetWakeEventLevel(pConfigList[i].wakeEventId, pConfigList[i].level);
        SetWakeEventEnabled(pConfigList[i].wakeEventId, pConfigList[i].isWakeEventEnable);
    }

    // ソフトウェア的な Finalize は用意されていない
#else
    NN_DETAIL_GPIO_TRACE("Not set Wakepin Initial Config\n");
#endif
}


void Driver::SetWakePinDebugMode(WakePinDebugMode mode) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    switch (mode)
    {
    case WakePinDebugMode_AutoImmediateWake :
        // WakeEvent_SDMMC3_DAT1 ピンは L 設定で常に wake event が来ている状態になる (CRCT-333)
        nne::wec::SetWakeEventEnabled(nne::wec::WakeEvent_SDMMC3_DAT1, true);
        nne::wec::SetWakeEventLevel(nne::wec::WakeEvent_SDMMC3_DAT1, nne::wec::WakeEventLevel_Low);
        break;

    case WakePinDebugMode_NoWake :
        // wake pin を disable にする設定を書き込む
        for (int i = 0; i < sizeof(detail::wakePinDisableWithoutPmicConfigList) / sizeof(detail::wakePinDisableWithoutPmicConfigList[0]); i++)
        {
            SetWakeEventLevel(detail::wakePinDisableWithoutPmicConfigList[i].wakeEventId, detail::wakePinDisableWithoutPmicConfigList[i].level);
            SetWakeEventEnabled(detail::wakePinDisableWithoutPmicConfigList[i].wakeEventId, detail::wakePinDisableWithoutPmicConfigList[i].isWakeEventEnable);
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }


#else
    NN_DETAIL_GPIO_TRACE("No wake pin debug settings\n");
#endif
}


} // detail
} // driver
} // gpio
} // nn
