﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/account/account_Api.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SaveDataExtension.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/fs/fs_Utility.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/util/util_BitUtil.h>

#if defined(NN_BUILD_CONFIG_COMPILER_GCC)
    #define NN_FS_PRAGMA_PUSH_WARNING_MAYBE_UNINITIALIZED \
        NN_PRAGMA_PUSH_WARNINGS \
        _Pragma("GCC diagnostic ignored  \"-Wmaybe-uninitialized\"")

    #define NN_FS_PRAGMA_POP_WARNING_MAYBE_UNINITIALIZED NN_PRAGMA_POP_WARNINGS
#else
    #define NN_FS_PRAGMA_PUSH_WARNING_MAYBE_UNINITIALIZED
    #define NN_FS_PRAGMA_POP_WARNING_MAYBE_UNINITIALIZED
#endif // defined(NN_BUILD_CONFIG_COMPILER_GCC)

namespace nn { namespace fs {

namespace {
    // 前提とする fat のクラスタサイズ
    const int64_t BlockSize = 16 * 1024;

    // 最悪で 1 ファイルごとに 1 ディレクトリエントリを占有する
    const int64_t MarginSizePerFile = BlockSize;

    int64_t RoundUpOccupationSize(int64_t size)
    {
        return nn::util::align_up(size, BlockSize);
    }

    int64_t CalculateSaveDataExtensionContextFileSize(int64_t availableSize, int64_t journalSize)
    {
        // SaveDataFileSystemDriver::QueryExpandLogSize() 相当
        return RoundUpOccupationSize( 1024 * 1024 + ( availableSize + journalSize ) / 1024 );
    }

    //! @brief 対象のセーブのサイズが期待値に満たない場合は拡張を行い、容量不足の場合は拡張のために必要とする容量を返す
    Result ExtendSaveDataIfNeeded(int64_t* pOutRequiredSize, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId, int64_t requiredAvailableSize, int64_t requiredJournalSize)
    {
        NN_SDK_ASSERT(pOutRequiredSize != nullptr);

        int64_t availableSize;
        NN_RESULT_DO(GetSaveDataAvailableSize(&availableSize, saveDataSpaceId, saveDataId));

        int64_t journalSize;
        NN_RESULT_DO(GetSaveDataJournalSize(&journalSize, saveDataSpaceId, saveDataId));

        if (availableSize < requiredAvailableSize || journalSize < requiredJournalSize)
        {
            // 拡張が発生した場合、拡張後のサイズが 1 MiB の倍数の値であることを要求する。
            // 判定はデータ保存領域・ジャーナリング領域のそれぞれで行う。
            if (availableSize < requiredAvailableSize)
            {
                NN_RESULT_THROW_UNLESS(util::is_aligned(requiredAvailableSize, fs::SaveDataExtensionUnitSize), fs::ResultExtensionSizeInvalid());
            }
            if (journalSize < requiredJournalSize)
            {
                NN_RESULT_THROW_UNLESS(util::is_aligned(requiredJournalSize, fs::SaveDataExtensionUnitSize), fs::ResultExtensionSizeInvalid());
            }

            // いずれか一方を拡張する場合、もう一方はサイズを維持する
            int64_t extendedAvailableSize = std::max(requiredAvailableSize, availableSize);
            int64_t extendedJournalSize = std::max(requiredJournalSize, journalSize);

            NN_RESULT_TRY(ExtendSaveData(saveDataSpaceId, saveDataId, extendedAvailableSize, extendedJournalSize))
                NN_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
                {
                    int64_t currentSaveDataTotalSize;
                    NN_RESULT_DO(QuerySaveDataTotalSize(&currentSaveDataTotalSize, availableSize, journalSize));

                    int64_t newSaveDataTotalSize;
                    NN_RESULT_DO(QuerySaveDataTotalSize(&newSaveDataTotalSize, extendedAvailableSize, extendedJournalSize));

                    *pOutRequiredSize +=
                        (RoundUpOccupationSize(newSaveDataTotalSize) - RoundUpOccupationSize(currentSaveDataTotalSize))
                        + CalculateSaveDataExtensionContextFileSize(extendedAvailableSize, extendedJournalSize)
                        + MarginSizePerFile * 2;
                    NN_RESULT_THROW(fs::ResultUsableSpaceNotEnough());
                }
            NN_RESULT_END_TRY
        }

        *pOutRequiredSize = 0;
        NN_RESULT_SUCCESS;
    }

    template <typename CreateFunc>
    Result CreateSaveData(int64_t* pInOutRequiredSize, CreateFunc createSaveData, int64_t availableSize, int64_t journalSize, int64_t additionalOccupationSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pInOutRequiredSize != nullptr);

        NN_RESULT_TRY(createSaveData())
            NN_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
            {
                int64_t saveDataTotalSize;
                NN_RESULT_DO(QuerySaveDataTotalSize(&saveDataTotalSize, availableSize, journalSize));
                *pInOutRequiredSize += RoundUpOccupationSize(saveDataTotalSize) + additionalOccupationSize;
            }
            NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
            {
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }

    //! @brief predFindSave を満たすセーブデータが存在しなければ createSaveData() で作成する。既存なら必要に応じて availableSize/journalSize に拡張する。
    //!        容量不足で操作が失敗した場合は必要な空き容量を *pInOutRequiredSize に加算しつつ、 ResultSuccess を返す。
    //!        additionalOccupationSize には新規作成時に追加で必要なサイズを指定する。
    //! @pre *pInOutRequiredSize が未初期化でない
    template <typename CreateFunc>
    Result EnsureAndExtendSaveData(int64_t* pInOutRequiredSize, SaveDataFilter& filter, CreateFunc createSaveData, int64_t availableSize, int64_t journalSize, int64_t additionalOccupationSize)
    {
        NN_SDK_ASSERT(pInOutRequiredSize != nullptr);

        nn::fs::SaveDataInfo info;
        NN_RESULT_TRY(FindSaveDataWithFilter(&info, SaveDataSpaceId::User, filter))
            NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(CreateSaveData(pInOutRequiredSize, createSaveData, availableSize, journalSize, additionalOccupationSize));
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        int64_t requiredSize = 0;
        NN_RESULT_TRY(ExtendSaveDataIfNeeded(&requiredSize, SaveDataSpaceId::User, info.saveDataId, availableSize, journalSize))
            NN_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
            {
                *pInOutRequiredSize += requiredSize;
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result CheckSaveDataType(fs::SaveDataType saveDataType, const account::Uid& user) NN_NOEXCEPT
    {
        switch (saveDataType)
        {
        case fs::SaveDataType::Account:
            if (!static_cast<bool>(user))
            {
                NN_RESULT_THROW(fs::ResultInvalidArgument());
            }
            NN_RESULT_SUCCESS;
#if 0
        case fs::SaveDataType::Device:
            if (static_cast<bool>(user))
            {
                NN_RESULT_THROW(fs::ResultInvalidArgument());
            }
            NN_RESULT_SUCCESS;
#endif
        default:
            NN_RESULT_THROW(fs::ResultInvalidArgument());
        }
    }

    Result CheckExtensionSizeUnderMax(fs::SaveDataType saveDataType, const nn::ns::ApplicationControlProperty& applicationControlProperty, int64_t requiredAvailableSize, int64_t requiredJournalSize) NN_NOEXCEPT
    {
        const auto& property = applicationControlProperty;
        switch (saveDataType)
        {
        case fs::SaveDataType::Account:
            if (requiredAvailableSize > property.userAccountSaveDataSizeMax ||
                requiredJournalSize > property.userAccountSaveDataJournalSizeMax)
            {
                NN_RESULT_THROW(fs::ResultExtensionSizeTooLarge());
            }
            NN_RESULT_SUCCESS;
        case fs::SaveDataType::Device:
            if (requiredAvailableSize > property.deviceSaveDataSizeMax ||
                requiredJournalSize > property.deviceSaveDataJournalSizeMax)
            {
                NN_RESULT_THROW(fs::ResultExtensionSizeTooLarge());
            }
            NN_RESULT_SUCCESS;
        default:
            NN_RESULT_THROW(fs::ResultInvalidArgument());
        }
    }

    Result EnsureApplicationBcatDeliveryCacheStorageImpl(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(pOutRequiredSize != nullptr, fs::ResultInvalidArgument());

        const auto& property = applicationControlProperty;
        int64_t requiredSize = 0;

        if (property.bcatDeliveryCacheStorageSize > 0)
        {
            SaveDataFilter filter = SaveDataFilter::Make(applicationId.value, SaveDataType::Bcat, util::nullopt, util::nullopt, util::nullopt);

            NN_RESULT_DO(EnsureAndExtendSaveData(
                &requiredSize,
                filter,
                [&]() {
                    return CreateBcatSaveData(applicationId, property.bcatDeliveryCacheStorageSize);
                },
                property.bcatDeliveryCacheStorageSize,
                detail::BcatSaveDataJournalSize,
                MarginSizePerFile
            ));
        }

        if (requiredSize == 0)
        {
            *pOutRequiredSize = 0;
            NN_RESULT_SUCCESS;
        }
        else
        {
            *pOutRequiredSize = requiredSize;
            NN_RESULT_THROW(ResultUsableSpaceNotEnough());
        }
    }
}

    // キャッシュストレージの保存先ストレージ
    Result GetCacheStorageTargetMedia(CacheStorageTargetMedia* outValue, ncm::ApplicationId applicationId)
    {
        auto isCacheStorageExisting = [=](bool* outIsExisting, SaveDataSpaceId spaceId) NN_NOEXCEPT -> Result
        {
            bool exists = true;
            SaveDataFilter filter = SaveDataFilter::Make(applicationId.value, SaveDataType::Cache, util::nullopt, util::nullopt, util::nullopt);
            nn::fs::SaveDataInfo info;
            NN_RESULT_TRY(FindSaveDataWithFilter(&info, spaceId, filter))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                    exists = false;
                }
            NN_RESULT_END_TRY

            *outIsExisting = exists;
            NN_RESULT_SUCCESS;
        };

        // SD にアクセス可能なら SD にアプリ ID が同一のキャッシュストレージがあるかどうかを確認する
        if (fs::IsSdCardAccessible())
        {
            // SD にキャッシュストレージがあるかどうか確認
            bool isExistInSd = false;
            NN_RESULT_DO(isCacheStorageExisting(&isExistInSd, SaveDataSpaceId::SdUser));

            // SD に存在する場合
            if (isExistInSd)
            {
                *outValue = CacheStorageTargetMedia_Sd;
                NN_RESULT_SUCCESS;
            }
        }

        // NAND にアプリ ID が同一のキャッシュストレージがあるかどうか確認
        bool isExistInNand = false;
        NN_RESULT_DO(isCacheStorageExisting(&isExistInNand, SaveDataSpaceId::User));

        // NAND には存在するが SD に存在しない場合、保存先ストレージは NAND
        if (isExistInNand)
        {
            *outValue = CacheStorageTargetMedia_Nand;
        }
        // どっちにもない場合は Any
        else
        {
            *outValue = CacheStorageTargetMedia_Any;
        }

        NN_RESULT_SUCCESS;
    }

    Result TryCreateCacheStorage(int64_t* pOutRequiredSize, SaveDataSpaceId spaceId, ncm::ApplicationId applicationId, Bit64 saveDataOwnerId, uint16_t index, int64_t cacheStorageSize, int64_t cacheStorageJournalSize, bool isExpand)
    {
        int64_t requiredSize = 0;
        nn::fs::SaveDataInfo info;
        SaveDataFilter filter = SaveDataFilter::Make(applicationId.value, SaveDataType::Cache, util::nullopt, util::nullopt, index);
        // 同一インデックスのキャッシュストレージが存在するかを確認
        bool exists = true;
        NN_FS_RESULT_TRY(FindSaveDataWithFilter(&info, spaceId, filter))
            NN_FS_RESULT_CATCH(fs::ResultTargetNotFound)
            {
                exists = false;
            }
        NN_FS_RESULT_END_TRY

        // 同一インデックスのキャッシュストレージが存在した
        if (exists)
        {
            if(!isExpand)
            {
                NN_FS_RESULT_DO(fs::ResultAlreadyExists());
            }
            // SD を Extend
            NN_FS_RESULT_TRY(ExtendSaveDataIfNeeded(&requiredSize, spaceId, info.saveDataId, cacheStorageSize, cacheStorageJournalSize))
                NN_FS_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
                {
                }
                NN_FS_RESULT_CATCH(fs::ResultSaveDataExtending)
                {
                    NN_FS_RESULT_DO(fs::ResultSaveDataCorrupted());
                }
            NN_FS_RESULT_END_TRY
        }
        // 同一インデックスのキャッシュストレージが存在しなかった
        else
        {
            // NotEnoughSpace は Success で返ってくる
            NN_FS_RESULT_DO(CreateSaveData(&requiredSize, [&]() {
                return CreateCacheStorage(applicationId, spaceId, saveDataOwnerId, index, cacheStorageSize, cacheStorageJournalSize, 0);
                },
                cacheStorageSize,
                cacheStorageJournalSize,
                MarginSizePerFile));
        }
        *pOutRequiredSize = requiredSize;
        NN_RESULT_SUCCESS;
    }

    // index 付き CacheStorage 作成
    Result EnsureApplicationCacheStorage(int64_t* pOutRequiredSize, CacheStorageTargetMedia* pOutCacheStorageTargetMedia, ncm::ApplicationId applicationId, Bit64 saveDataOwnerId, uint16_t index, int64_t cacheStorageSize, int64_t cacheStorageJournalSize, bool isExpand) NN_NOEXCEPT
    {
        NN_FS_RESULT_THROW_UNLESS(pOutCacheStorageTargetMedia != nullptr, fs::ResultInvalidArgument());

        *pOutCacheStorageTargetMedia = CacheStorageTargetMedia_Sd;
        int64_t requiredSize = 0;
        CacheStorageTargetMedia targetMedia;
        NN_FS_RESULT_DO(GetCacheStorageTargetMedia(&targetMedia, applicationId));

        // SD に保存
        if(targetMedia == CacheStorageTargetMedia_Sd)
        {
            // 結果によらずターゲットストレージは SD 確定
            *pOutCacheStorageTargetMedia = CacheStorageTargetMedia_Sd;
            NN_FS_RESULT_DO(TryCreateCacheStorage(&requiredSize, SaveDataSpaceId::SdUser, applicationId, saveDataOwnerId, index, cacheStorageSize, cacheStorageJournalSize, isExpand));
        }
        // NAND に保存
        else if(targetMedia == CacheStorageTargetMedia_Nand)
        {
            // 結果によらずターゲットストレージは NAND 確定
            *pOutCacheStorageTargetMedia = CacheStorageTargetMedia_Nand;
            NN_FS_RESULT_DO(TryCreateCacheStorage(&requiredSize, SaveDataSpaceId::User, applicationId, saveDataOwnerId, index, cacheStorageSize, cacheStorageJournalSize, isExpand));
        }
        // 新規作成の場合
        else
        {
            bool isCreateAtSd = fs::IsSdCardAccessible();
            // ns が SD マウント済みなら SD に 配置しようとする
            if(isCreateAtSd)
            {
                *pOutCacheStorageTargetMedia = CacheStorageTargetMedia_Sd;
                // NotEnoughSpace は Success で返ってくる
                NN_FS_RESULT_DO(CreateSaveData(&requiredSize, [&]() {
                    return CreateCacheStorage(applicationId, SaveDataSpaceId::SdUser, saveDataOwnerId, index, cacheStorageSize, cacheStorageJournalSize, 0);
                    },
                    cacheStorageSize,
                    cacheStorageJournalSize,
                    MarginSizePerFile));
                // ここが 0 でなければ NotEnoughSpace なので NAND に配置しようとする
                if(requiredSize != 0)
                {
                    isCreateAtSd = false;
                }
            }
            // SD 未マウント or SD に作ろうとしたけど容量不足の場合、 NAND に配置しようとする
            if(!isCreateAtSd)
            {
                requiredSize = 0;
                *pOutCacheStorageTargetMedia = CacheStorageTargetMedia_Nand;
                NN_FS_RESULT_DO(CreateSaveData(&requiredSize, [&]() {
                    return CreateCacheStorage(applicationId, SaveDataSpaceId::User, saveDataOwnerId, index, cacheStorageSize, cacheStorageJournalSize, 0);
                    },
                    cacheStorageSize,
                    cacheStorageJournalSize,
                    MarginSizePerFile));
                if(requiredSize != 0)
                {
                    // SD の有無にかかわらず、容量不足だった場合は Any で返す
                    *pOutCacheStorageTargetMedia = CacheStorageTargetMedia_Any;
                }
            }
        }

        if (requiredSize == 0)
        {
            *pOutRequiredSize = 0;
            NN_RESULT_SUCCESS;
        }
        else
        {
            *pOutRequiredSize = requiredSize;
            NN_FS_RESULT_THROW(ResultUsableSpaceNotEnough());
        }
    }// NOLINT(impl/function_size)

    // obsolete : 本 API はまもなく削除されます。
    Result EnsureApplicationCacheStorage(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty) NN_NOEXCEPT
    {
        CacheStorageTargetMedia media;
        NN_RESULT_DO(EnsureApplicationCacheStorage(pOutRequiredSize, &media, applicationId, applicationControlProperty));
        NN_RESULT_SUCCESS;
    }

    // 起動時に Ocean からキャッシュストレージ作成
    Result EnsureApplicationCacheStorage(int64_t* pOutRequiredSize, CacheStorageTargetMedia* pOutCacheStorageTargetMedia, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty) NN_NOEXCEPT
    {
        // 起動に作成する場合、キャッシュストレージサイズが 0 以下なら実行しない
        if(applicationControlProperty.cacheStorageSize > 0)
        {
            NN_RESULT_DO(EnsureApplicationCacheStorage(pOutRequiredSize, pOutCacheStorageTargetMedia, applicationId, applicationControlProperty.saveDataOwnerId, 0, applicationControlProperty.cacheStorageSize, applicationControlProperty.cacheStorageJournalSize, true));
        }
        NN_RESULT_SUCCESS;
    }

    // アプリからインデックス付きキャッシュストレージを作成
    Result CreateApplicationCacheStorage(int64_t* pOutRequiredSize, CacheStorageTargetMedia* pOutCacheStorageTargetMedia, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty, uint16_t index, int64_t cacheStorageSize, int64_t cacheStorageJournalSize) NN_NOEXCEPT
    {
        if(applicationControlProperty.cacheStorageIndexMax < index)
        {
            return fs::ResultCacheStorageIndexTooLarge();
        }
        if(applicationControlProperty.cacheStorageDataAndJournalSizeMax < cacheStorageSize + cacheStorageJournalSize)
        {
            return fs::ResultCacheStorageSizeTooLarge();
        }
        NN_RESULT_DO(EnsureApplicationCacheStorage(pOutRequiredSize, pOutCacheStorageTargetMedia, applicationId, applicationControlProperty.saveDataOwnerId, index, cacheStorageSize, cacheStorageJournalSize, false));
        NN_RESULT_SUCCESS;
    }

    //! @brief 一時ストレージを削除する
    Result CleanUpTemporaryStorage() NN_NOEXCEPT
    {
        // TORIAEZU: 自動 abort 無し
        while (NN_STATIC_CONDITION(true))
        {
            nn::fs::SaveDataInfo info;
            SaveDataFilter filter = SaveDataFilter::Make(util::nullopt, SaveDataType::Temporary, util::nullopt, util::nullopt, util::nullopt);

            NN_FS_RESULT_TRY(FindSaveDataWithFilter(&info, SaveDataSpaceId::Temporary, filter))
                NN_FS_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                    NN_RESULT_SUCCESS;
                }
            NN_FS_RESULT_END_TRY

            NN_FS_RESULT_DO(DeleteSaveData(SaveDataSpaceId::Temporary, info.saveDataId));
        }

        NN_RESULT_SUCCESS;
    }

    Result EnsureApplicationBcatDeliveryCacheStorage(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty) NN_NOEXCEPT
    {
        NN_FS_RESULT_THROW_UNLESS(pOutRequiredSize != nullptr, fs::ResultInvalidArgument());
        NN_FS_RESULT_DO(EnsureApplicationBcatDeliveryCacheStorageImpl(pOutRequiredSize, applicationId, applicationControlProperty));
        NN_RESULT_SUCCESS;
    }

    Result EnsureApplicationSaveData(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty, const account::Uid& user) NN_NOEXCEPT
    {
        NN_FS_RESULT_THROW_UNLESS(pOutRequiredSize != nullptr, fs::ResultInvalidArgument());

        const auto& property = applicationControlProperty;

        int64_t requiredSize = 0;

        // ユーザアカウントセーブ
        if (user != nn::account::InvalidUid && property.userAccountSaveDataSize > 0)
        {
            UserId userId = {{user._data[0], user._data[1]}};
            SaveDataFilter filter = SaveDataFilter::Make(applicationId.value, SaveDataType::Account, userId, util::nullopt, util::nullopt);

            NN_FS_RESULT_DO(EnsureAndExtendSaveData(
                &requiredSize,
                filter,
                [&]() {
                    return CreateSaveData(
                        applicationId,
                        fs::ConvertAccountUidToFsUserId(user),
                        property.saveDataOwnerId,
                        property.userAccountSaveDataSize,
                        property.userAccountSaveDataJournalSize,
                        0
                    );
                },
                property.userAccountSaveDataSize,
                property.userAccountSaveDataJournalSize,
                RoundUpOccupationSize(detail::ThumbnailFileSize) + MarginSizePerFile * 2
            ));
        }

        // 本体セーブ
        if (property.deviceSaveDataSize > 0)
        {
            SaveDataFilter filter = SaveDataFilter::Make(applicationId.value, SaveDataType::Device, util::nullopt, util::nullopt, util::nullopt);

            NN_FS_RESULT_DO(EnsureAndExtendSaveData(
                &requiredSize,
                filter,
                [&]() {
                    return CreateDeviceSaveData(applicationId, property.saveDataOwnerId, property.deviceSaveDataSize, property.deviceSaveDataJournalSize, 0);
                },
                property.deviceSaveDataSize,
                property.deviceSaveDataJournalSize,
                MarginSizePerFile
            ));
        }

        // bcat セーブ
        {
            int64_t requiredSizeForBcat = 0;
            NN_FS_RESULT_TRY(EnsureApplicationBcatDeliveryCacheStorageImpl(&requiredSizeForBcat, applicationId, property))
                NN_FS_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
                {
                    requiredSize += requiredSizeForBcat;
                }
            NN_FS_RESULT_END_TRY
        }

        // 一時ストレージ: 最後に作成する
        if (property.temporaryStorageSize > 0)
        {
            auto CalculateRequiredSizeForTemporaryStorage = [&property](int64_t* pRequiredSize) -> Result
            {
                int64_t saveDataTotalSize;
                // TORIAEZU: QuerySaveDataTotalSize でサイズ計算代用
                NN_FS_RESULT_DO(QuerySaveDataTotalSize(&saveDataTotalSize, property.temporaryStorageSize, 0));
                *pRequiredSize = RoundUpOccupationSize(saveDataTotalSize) + MarginSizePerFile;
                NN_RESULT_SUCCESS;
            };

            if (requiredSize == 0)
            {
                NN_FS_RESULT_TRY(CreateTemporaryStorage(applicationId, property.saveDataOwnerId, property.temporaryStorageSize, 0))
                    NN_FS_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
                    {
                        int64_t temporaryStorageSize;
                        NN_FS_RESULT_DO(CalculateRequiredSizeForTemporaryStorage(&temporaryStorageSize));
                        requiredSize += temporaryStorageSize;
                    }
                    NN_FS_RESULT_CATCH(fs::ResultPathAlreadyExists)
                    {
                    }
                NN_FS_RESULT_END_TRY
            }
            else
            {
                // 他のセーブデータの作成で容量不足になっていた場合は一時ストレージも作成できなかった体で返す
                nn::fs::SaveDataInfo info;
                SaveDataFilter filter = SaveDataFilter::Make(applicationId.value, SaveDataType::Temporary, util::nullopt, util::nullopt, util::nullopt);
                NN_FS_RESULT_TRY(FindSaveDataWithFilter(&info, SaveDataSpaceId::Temporary, filter));
                    NN_FS_RESULT_CATCH(fs::ResultTargetNotFound)
                    {
                        int64_t temporaryStorageSize;
                        NN_FS_RESULT_DO(CalculateRequiredSizeForTemporaryStorage(&temporaryStorageSize));
                        requiredSize += temporaryStorageSize;
                    }
                NN_FS_RESULT_END_TRY
            }
        }

        if( requiredSize == 0 )
        {
            *pOutRequiredSize = 0;
            NN_RESULT_SUCCESS;
        }
        else
        {
            *pOutRequiredSize = requiredSize;
            NN_FS_RESULT_THROW(ResultUsableSpaceNotEnough());
        }

    } // NOLINT(impl/function_size)

    Result ExtendApplicationSaveData(int64_t* pOutRequiredSize, ncm::ApplicationId applicationId, const nn::ns::ApplicationControlProperty& applicationControlProperty, nn::fs::SaveDataType saveDataType, const account::Uid& user, int64_t saveDataSize, int64_t saveDataJournalSize) NN_NOEXCEPT
    {
        NN_FS_RESULT_THROW_UNLESS(pOutRequiredSize != nullptr, fs::ResultInvalidArgument());

        NN_FS_RESULT_DO(CheckSaveDataType(saveDataType, user));

        nn::fs::SaveDataInfo info;

        UserId userId = {{user._data[0], user._data[1]}};
        SaveDataFilter filter = SaveDataFilter::Make(applicationId.value, saveDataType, userId, util::nullopt, util::nullopt);

        NN_FS_RESULT_DO(FindSaveDataWithFilter(&info, SaveDataSpaceId::User, filter));

        const auto& property = applicationControlProperty;
        NN_FS_RESULT_DO(CheckExtensionSizeUnderMax(saveDataType, property, saveDataSize, saveDataJournalSize));

        int64_t requiredSize = 0;
        // info が nullopt でなければ値は設定されているはずだが、
        // gcc のビルドで info.saveDataId が未初期化と出る。その警告を抑制する。
        NN_FS_PRAGMA_PUSH_WARNING_MAYBE_UNINITIALIZED
        NN_FS_RESULT_TRY(ExtendSaveDataIfNeeded(&requiredSize, SaveDataSpaceId::User, info.saveDataId, saveDataSize, saveDataJournalSize))
            NN_FS_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
            {
                *pOutRequiredSize = requiredSize;
                NN_FS_RESULT_RETHROW;
            }
        NN_FS_RESULT_END_TRY
        NN_FS_PRAGMA_POP_WARNING_MAYBE_UNINITIALIZED

        NN_RESULT_SUCCESS;
    }

    Result GetApplicationSaveDataSize(int64_t* pOutSize, int64_t* pOutJournalSize, ncm::ApplicationId applicationId, nn::fs::SaveDataType saveDataType, const account::Uid& user) NN_NOEXCEPT
    {
        NN_FS_RESULT_THROW_UNLESS(pOutSize != nullptr, fs::ResultInvalidArgument());
        NN_FS_RESULT_THROW_UNLESS(pOutJournalSize != nullptr, fs::ResultInvalidArgument());

        NN_FS_RESULT_DO(CheckSaveDataType(saveDataType, user));

        nn::fs::SaveDataInfo info;

        UserId userId = {{user._data[0], user._data[1]}};
        SaveDataFilter filter = SaveDataFilter::Make(applicationId.value, saveDataType, userId, util::nullopt, util::nullopt);

        NN_FS_RESULT_DO(FindSaveDataWithFilter(&info, SaveDataSpaceId::User, filter));

        int64_t saveDataSize = 0;
        int64_t saveDataJournalSize = 0;
        NN_FS_RESULT_DO(GetSaveDataAvailableSize(&saveDataSize, info.saveDataId));
        NN_FS_RESULT_DO(GetSaveDataJournalSize(&saveDataJournalSize, info.saveDataId));

        *pOutSize = saveDataSize;
        *pOutJournalSize = saveDataJournalSize;

        NN_RESULT_SUCCESS;
    }

}}
