﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/aoc/aoc_SystemApi.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_Data.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/fs_AddOnContent.h>
#include <nn/fs/fs_AddOnContentPrivate.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_DataId.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_Result.h>
#include <nn/aoc/aoc_ResultPrivate.h>

namespace {

    // TORIAEZU: PrepareAddOnContent - Mount 間を排他
    nn::os::Mutex g_AocLrMutex(false);

    nn::ncm::DataId GetDataId(nn::aoc::AddOnContentIndex targetIndex) NN_NOEXCEPT
    {
        auto targetBase = nn::aoc::GetAddOnContentBaseId();
        nn::ncm::DataId dataId = {targetBase + targetIndex};
        return dataId;
    }

    nn::ncm::DataId GetDataId(
        nn::ApplicationId targetApplication,
        nn::aoc::AddOnContentIndex targetIndex
    )
    {
        auto targetBase = nn::aoc::GetAddOnContentBaseId(targetApplication);
        nn::ncm::DataId dataId = {targetBase + targetIndex};
        return dataId;
    }

    nn::Result ConvertAddOnContentResult(nn::Result result)
    {
        if (
            nn::ns::ResultApplicationRecordNotFound::Includes(result)        ||
            nn::ns::ResultNoContentMetaInApplicationRecord::Includes(result) ||
            nn::aoc::ResultNotFoundAddOnContent::Includes(result)
            )
        {
            return nn::fs::ResultTargetNotFound();
        }
        else
        {
            return result;
        }
    }

    nn::Result PrepareAddOnContent(nn::aoc::AddOnContentIndex targetIndex) NN_NOEXCEPT
    {
        return ConvertAddOnContentResult(nn::aoc::PrepareAddOnContent(targetIndex));
    }
    nn::Result PrepareAddOnContent(nn::aoc::AddOnContentIndex targetIndex, nn::ApplicationId targetApplication) NN_NOEXCEPT
    {
        return ConvertAddOnContentResult(nn::aoc::PrepareAddOnContent(targetIndex, targetApplication));
    }
}

namespace nn { namespace fs {

Result QueryMountAddOnContentCacheSize(
    size_t* pOutValue,
    nn::aoc::AddOnContentIndex targetIndex
) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        std::lock_guard<os::Mutex> scopedLock(g_AocLrMutex);
        NN_RESULT_DO(PrepareAddOnContent(targetIndex));
        nn::ncm::DataId dataId = GetDataId(targetIndex);
        NN_RESULT_DO(detail::QueryMountDataCacheSize(pOutValue, dataId, nn::ncm::StorageId::None));
        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX NN_DETAIL_FS_ACCESS_LOG_FORMAT_QUERY_SIZE, targetIndex, detail::ReferenceQuerySizeOutValue(pOutValue)));
    NN_RESULT_SUCCESS;
}

Result QueryMountAddOnContentCacheSize(
    size_t* pOutValue,
    nn::ApplicationId targetApplication,
    nn::aoc::AddOnContentIndex targetIndex
) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        std::lock_guard<os::Mutex> scopedLock(g_AocLrMutex);
        NN_RESULT_DO(PrepareAddOnContent(targetIndex, targetApplication));
        nn::ncm::DataId dataId = GetDataId(targetApplication, targetIndex);
        NN_RESULT_DO(detail::QueryMountDataCacheSize(pOutValue, dataId, nn::ncm::StorageId::None));
        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX NN_DETAIL_FS_ACCESS_LOG_FORMAT_QUERY_SIZE, targetIndex, detail::ReferenceQuerySizeOutValue(pOutValue)));
    NN_RESULT_SUCCESS;
}

Result MountAddOnContent(
    const char* name,
    nn::aoc::AddOnContentIndex targetIndex,
    void* pFileSystemCacheBuffer,
    size_t fileSystemCacheBufferSize
) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        std::lock_guard<os::Mutex> scopedLock(g_AocLrMutex);
        NN_RESULT_DO(PrepareAddOnContent(targetIndex));
        nn::ncm::DataId dataId = GetDataId(targetIndex);
        NN_RESULT_DO(detail::MountData(name, dataId, nn::ncm::StorageId::None, pFileSystemCacheBuffer, fileSystemCacheBufferSize, true, true));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTADDONCONTENT(name, targetIndex)));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}


Result MountAddOnContent(
    const char* name,
    nn::ApplicationId targetApplication,
    nn::aoc::AddOnContentIndex targetIndex,
    void* pFileSystemCacheBuffer,
    size_t fileSystemCacheBufferSize
) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        std::lock_guard<os::Mutex> scopedLock(g_AocLrMutex);
        NN_RESULT_DO(PrepareAddOnContent(targetIndex, targetApplication));
        nn::ncm::DataId dataId = GetDataId(targetApplication, targetIndex);
        NN_RESULT_DO(detail::MountData(name, dataId, nn::ncm::StorageId::None, pFileSystemCacheBuffer, fileSystemCacheBufferSize, true, true));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTADDONCONTENT(name, targetIndex)));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

}}
