﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/json/friends_JsonEventHandlerForReader.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

JsonEventHandlerForReader::JsonEventHandlerForReader() NN_NOEXCEPT :
    m_Handler(nullptr),
    m_Cancelable(nullptr),
    m_OverflowDetector(nullptr)
{
}

void JsonEventHandlerForReader::SetHandler(JsonEventHandler* handler) NN_NOEXCEPT
{
    m_Handler = handler;
}

void JsonEventHandlerForReader::SetCancelable(const detail::service::util::Cancelable* cancelable) NN_NOEXCEPT
{
    m_Cancelable = cancelable;
}

void JsonEventHandlerForReader::SetOverflowDetector(const std::atomic<bool>* overflowDetector) NN_NOEXCEPT
{
    m_OverflowDetector = overflowDetector;
}

bool JsonEventHandlerForReader::StartObject() NN_NOEXCEPT
{
    if (IsCanceled() || !m_JsonPath.TryPushObject())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->OnStartObject(m_JsonPath);
    }

    return true;
}

bool JsonEventHandlerForReader::EndObject(RAPIDJSON_NAMESPACE::SizeType numObjects) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.Pop();

    bool isContinue = true;

    if (m_Handler)
    {
        isContinue = m_Handler->OnEndObject(m_JsonPath, static_cast<size_t>(numObjects));
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::StartArray() NN_NOEXCEPT
{
    if (IsCanceled() || !m_JsonPath.TryPushArray())
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->OnStartArray(m_JsonPath);
    }

    return true;
}

bool JsonEventHandlerForReader::EndArray(RAPIDJSON_NAMESPACE::SizeType numElements) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.Pop();

    bool isContinue = true;

    if (m_Handler)
    {
        isContinue = m_Handler->OnEndArray(m_JsonPath, static_cast<size_t>(numElements));
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::Key(const char* key, RAPIDJSON_NAMESPACE::SizeType length, bool) NN_NOEXCEPT
{
    if (IsCanceled() || !m_JsonPath.TryPushKey(key, static_cast<size_t>(length), IsOverflowed()))
    {
        return false;
    }

    if (m_Handler)
    {
        return m_Handler->OnKey(m_JsonPath, key);
    }

    return true;
}

bool JsonEventHandlerForReader::Null() NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.UpdateCurrent();

    bool isContinue = true;

    if (m_Handler)
    {
        isContinue = m_Handler->OnNull(m_JsonPath, nullptr);
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::Bool(bool value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.UpdateCurrent();

    bool isContinue = true;

    if (m_Handler)
    {
        isContinue = m_Handler->OnBoolean(m_JsonPath, value);
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::Int(int32_t value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.UpdateCurrent();

    bool isContinue = true;

    if (m_Handler)
    {
        AnyInteger integer;

        integer.s = value;
        integer.isSigned = true;

        isContinue = m_Handler->OnInteger(m_JsonPath, integer);
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::Uint(uint32_t value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.UpdateCurrent();

    bool isContinue = true;

    if (m_Handler)
    {
        AnyInteger integer;

        integer.u = value;
        integer.isSigned = false;

        isContinue = m_Handler->OnInteger(m_JsonPath, integer);
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::Int64(int64_t value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.UpdateCurrent();

    bool isContinue = true;

    if (m_Handler)
    {
        AnyInteger integer;

        integer.s = value;
        integer.isSigned = true;

        isContinue = m_Handler->OnInteger(m_JsonPath, integer);
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::Uint64(uint64_t value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.UpdateCurrent();

    bool isContinue = true;

    if (m_Handler)
    {
        AnyInteger integer;

        integer.u = value;
        integer.isSigned = false;

        isContinue = m_Handler->OnInteger(m_JsonPath, integer);
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::Double(double value) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.UpdateCurrent();

    bool isContinue = true;

    if (m_Handler)
    {
        isContinue = m_Handler->OnDouble(m_JsonPath, value);
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::String(const char* value, RAPIDJSON_NAMESPACE::SizeType length, bool) NN_NOEXCEPT
{
    if (IsCanceled())
    {
        return false;
    }
    m_JsonPath.UpdateCurrent();

    bool isContinue = true;

    if (m_Handler)
    {
        isContinue = m_Handler->OnString(m_JsonPath, value, static_cast<size_t>(length), IsOverflowed());
    }
    m_JsonPath.PopIfKey();

    return isContinue;
}

bool JsonEventHandlerForReader::IsCanceled() const NN_NOEXCEPT
{
    if (m_Cancelable)
    {
        return m_Cancelable->IsCanceled();
    }

    return false;
}

bool JsonEventHandlerForReader::IsOverflowed() const NN_NOEXCEPT
{
    if (m_OverflowDetector)
    {
        return *m_OverflowDetector;
    }

    return false;
}

}}}}}
