﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskSendFacedFriendRequest.h>
#include <nn/friends/detail/service/core/friends_FacedFriendRequestManager.h>
#include <nn/friends/detail/service/core/friends_WorkBuffer.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

SendFacedFriendRequestTask::SendFacedFriendRequestTask() NN_NOEXCEPT :
    m_IsAccountIdSpecified(false),
    m_IsForeground(false),
    m_LastSendResult(nn::ResultSuccess())
{
}

nn::Result SendFacedFriendRequestTask::SetParameter(bool isForeground) NN_NOEXCEPT
{
    m_IsForeground = isForeground;

    NN_RESULT_SUCCESS;
}

nn::Result SendFacedFriendRequestTask::SetParameter(nn::account::NetworkServiceAccountId accountId, bool isForeground) NN_NOEXCEPT
{
    m_AccountId = accountId;
    m_IsAccountIdSpecified = true;

    m_IsForeground = isForeground;

    NN_RESULT_SUCCESS;
}

nn::Result SendFacedFriendRequestTask::Main() NN_NOEXCEPT
{
    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepSend());

    NN_RESULT_SUCCESS;
}

nn::Result SendFacedFriendRequestTask::StepSend() NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    FacedFriendRequestResource* resources = reinterpret_cast<FacedFriendRequestResource*>(WorkBuffer::GetInstance().Get());
    int count;

    if (m_IsAccountIdSpecified)
    {
        NN_RESULT_DO(FacedFriendRequestManager::GetInstance().GetFacedFriendRequest(resources, GetUid(), m_AccountId));
        count = 1;
    }
    else
    {
        NN_RESULT_DO(FacedFriendRequestManager::GetInstance().GetFacedFriendRequestList(&count,
            resources, GetUid(), 0, FacedFriendRequestCountMax));
    }

    for (int i = 0; i < count; i++)
    {
        NN_RESULT_DO(StepSendImpl(accountId, resources[i]));
    }

    NN_RESULT_DO(FacedFriendRequestManager::GetInstance().UpdateFacedFriendRequestList(GetUid(), resources, count));

    if (m_IsForeground)
    {
        NN_RESULT_DO(m_LastSendResult);
    }

    NN_RESULT_SUCCESS;
}

nn::Result SendFacedFriendRequestTask::StepSendImpl(nn::account::NetworkServiceAccountId accountId,
    FacedFriendRequestResource& resource) NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] SendFacedFriendRequestTask::StepSend(%016llx) ...\n", resource.accountId.id);

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url), "https://%s/2.0.0/friend_requests", WebApi::GetFqdn());

    char postField[1024] = {};
    NN_RESULT_DO(CreateSendPostField(accountId, resource, postField, sizeof (postField)));

    NN_RESULT_DO(stream.Open("POST", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.SetPostField(postField, false));

    detail::service::json::JsonEventHandler handler;

    nn::Result result = nn::ResultSuccess();

    NN_RESULT_TRY(WebApi::Call(handler, stream, GetUid(), GetCancelable()))
        NN_RESULT_CATCH(ResultServerError400SenderFriendCapacityIsFull)
        {
            result = ResultMyFriendListLimitReached();
        }
        NN_RESULT_CATCH(ResultServerError400ReceiverFriendCapacityIsFull)
        {
            result = ResultFriendListLimitReached();
        }
        NN_RESULT_CATCH(ResultServerError403OperationIsNotPermitted)
        {
            result = ResultFriendRequestReceptionNotAllowed();
        }
        NN_RESULT_CATCH(ResultServerError409ResourceDuplicated)
        {
            result = ResultAlreadyRequested();
        }
        NN_RESULT_CATCH_ALL
        {
            result = NN_RESULT_CURRENT_RESULT;
        }
    NN_RESULT_END_TRY;

    if (result.IsSuccess() ||
        ResultNetworkServiceAccountNotExists::Includes(result) ||
        ResultAlreadyRequested::Includes(result) ||
        ResultAlreadyFriend::Includes(result))
    {
        resource.requestStatus = RequestStatus_Pending;
    }

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultMyFriendListLimitReached)
        {
            resource.requestStatus = RequestStatus_UnsentForMyFriendListLimitReached;
        }
        NN_RESULT_CATCH(ResultFriendListLimitReached)
        {
            resource.requestStatus = RequestStatus_UnsentForFriendListLimitReached;
        }
        NN_RESULT_CATCH(ResultFriendRequestReceptionNotAllowed)
        {
            resource.requestStatus = RequestStatus_UnsentForFriendRequestReceptionNotAllowed;
        }
        NN_RESULT_CATCH(ResultBlockedByMe)
        {
            resource.requestStatus = RequestStatus_UnsentForBlockedByMe;
        }
        NN_RESULT_CATCH_ALL
        {
            if (resource.requestStatus == RequestStatus_Unsent)
            {
                resource.requestStatus = RequestStatus_UnsentForNetworkFailureOccurred;
            }
        }
    NN_RESULT_END_TRY;

    m_LastSendResult = result;

    NN_RESULT_SUCCESS;
}

nn::Result SendFacedFriendRequestTask::CreateSendPostField(nn::account::NetworkServiceAccountId accountId,
    const FacedFriendRequestResource& resource, char* buffer, size_t size) NN_NOEXCEPT
{
    nn::friends::detail::service::json::FixedSizeAllocator allocator;
    nn::friends::detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    nn::friends::detail::service::json::JsonWriter writer(stream, &allocator, 16);

    writer.StartObject();
    {
        writer.Key("channels");
        writer.StartArray();
        {
            writer.String(ParameterConverter::ConvertRequestType(RequestType_Faced));
        }
        writer.EndArray();

        char idString[17];

        nn::util::SNPrintf(idString, sizeof (idString), "%016llx", accountId.id);
        writer.Key("senderId");
        writer.String(idString);

        nn::util::SNPrintf(idString, sizeof (idString), "%016llx", resource.accountId.id);
        writer.Key("receiverId");
        writer.String(idString);

        writer.Key("extras");
        writer.StartObject();
        {
            if (resource.appInfo.appId != nn::ApplicationId::GetInvalidId())
            {
                writer.Key("sender");
                writer.StartObject();
                {
                    writer.Key("route:sender");
                    writer.String("me");

                    writer.Key("route:name");
                    writer.String(resource.inAppScreenName.name);

                    writer.Key("route:name:language");
                    writer.String(resource.inAppScreenName.language.string);
                }
                writer.EndObject();

                writer.Key("senderAndReceiver");
                writer.StartObject();
                {
                    nn::util::SNPrintf(idString, sizeof (idString), "%016llx", resource.appInfo.appId.value);
                    writer.Key("route:appInfo:appId");
                    writer.String(idString);

                    nn::util::SNPrintf(idString, sizeof (idString), "%016llx", resource.appInfo.presenceGroupId);
                    writer.Key("route:appInfo:presenceGroupId");
                    writer.String(idString);

                    writer.Key("route:name");
                    writer.String(resource.myInAppScreenName.name);

                    writer.Key("route:name:language");
                    writer.String(resource.myInAppScreenName.language.string);
                }
                writer.EndObject();
            }
            else
            {
                writer.Key("sender");
                writer.StartObject();
                {
                    writer.Key("route:sender");
                    writer.String("me");
                }
                writer.EndObject();
            }
        }
        writer.EndObject();
    }
    writer.EndObject();

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

}}}}}
