﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskGetFriendRequestList.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerFriendRequestResourceList.h>
#include <nn/friends/detail/service/core/friends_FacedFriendRequestManager.h>
#include <nn/friends/detail/service/core/friends_FriendRequestCountManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

GetFriendRequestListTask::GetFriendRequestListTask() NN_NOEXCEPT :
    m_IsParameterSet(false)
{
}

nn::Result GetFriendRequestListTask::SetParameter(int* outCount, FriendRequestImpl* outRequests,
    int offset, int count, RequestListType listType) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(listType == RequestListType_Faced ||
        listType == RequestListType_Sent || listType == RequestListType_Received, ResultInvalidArgument());

    m_OutCount = outCount;
    m_OutRequests = outRequests;

    m_Offset = offset;
    m_Count = count;
    m_ListType = listType;

    m_IsParameterSet = true;

    NN_RESULT_SUCCESS;
}

nn::Result GetFriendRequestListTask::Main() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsParameterSet);

    if (m_ListType == RequestListType_Faced)
    {
        NN_RESULT_DO(StepGetFacedFriendRequestList());
    }
    else if (m_ListType == RequestListType_Sent)
    {
        NN_RESULT_DO(StepGetSentFriendRequestList());
    }
    else if (m_ListType == RequestListType_Received)
    {
        NN_RESULT_DO(StepGetReceivedFriendRequestList());
    }
    else
    {
        NN_ABORT("");
    }

    NN_RESULT_SUCCESS;
}

nn::Result GetFriendRequestListTask::StepGetFacedFriendRequestList() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] GetFriendRequestListTask::StepGetFacedFriendRequestList ...\n");

    return FacedFriendRequestManager::GetInstance().GetFacedFriendRequestList(m_OutCount,
        m_OutRequests, GetUid(), m_Offset, m_Count);
}

nn::Result GetFriendRequestListTask::StepGetSentFriendRequestList() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] GetFriendRequestListTask::StepGetSentFriendRequestList ...\n");

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[512] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/2.0.0/users/%016llx/friend_requests/outbox?offset=%d&count=%d&sort=createdAt:desc&filter.state.$eq=PENDING",
        WebApi::GetFqdn(), accountId.id, m_Offset, m_Count);

    NN_RESULT_DO(stream.Open("GET", url));

    FriendRequestResourceListHandler handler;
    handler.Initialize(m_OutCount, m_OutRequests, GetUid(), m_ListType, m_Count);

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_SUCCESS;
}

nn::Result GetFriendRequestListTask::StepGetReceivedFriendRequestList() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] GetFriendRequestListTask::StepGetReceivedFriendRequestList ...\n");

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[512] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/2.0.0/users/%016llx/friend_requests/inbox?offset=%d&count=%d&sort=createdAt:desc&filter.state.$eq=PENDING",
        WebApi::GetFqdn(), accountId.id, m_Offset, m_Count);

    NN_RESULT_DO(stream.Open("GET", url));

    FriendRequestResourceListHandler handler;
    handler.Initialize(m_OutCount, m_OutRequests, GetUid(), m_ListType, m_Count);

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_DO(FriendRequestCountManager::GetInstance().SetCount(GetUid(), handler.GetUnreadCount(), handler.GetReadCount()));

    NN_RESULT_SUCCESS;
}

}}}}}
