﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_UserSettingManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* FileName = "setting.cache";
}

UserSettingManager::UserSettingManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_CurrentUid(nn::account::InvalidUid)
{
}

nn::Result UserSettingManager::GetPlayLogList(int* outCount, PlayLog* outPlayLogs,
    const nn::account::Uid& uid, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outPlayLogs);
    NN_SDK_REQUIRES(count > 0);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outPlayLogs, 0, sizeof (PlayLog) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = 0; i < NN_ARRAY_SIZE(m_Record.profileExtra.data.playLog); i++)
    {
        if (m_Record.profileExtra.data.playLog[i].appInfo.appId == nn::ApplicationId::GetInvalidId())
        {
            break;
        }
        if (count-- == 0)
        {
            break;
        }

        outPlayLogs[actualCount++] = m_Record.profileExtra.data.playLog[i];
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result UserSettingManager::GetPlayLogPermission(ExtraPermission* outPermission, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPermission);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    *outPermission = static_cast<ExtraPermission>(m_Record.playLogPermission);

    NN_RESULT_SUCCESS;
}

nn::Result UserSettingManager::HasFriendCode(bool* out, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    *out = (m_Record.friendCode.value[0] != '\0');

    NN_RESULT_SUCCESS;
}

nn::Result UserSettingManager::Read(UserSettingImpl* outSetting, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSetting);

    std::memset(outSetting, 0, sizeof (UserSettingImpl));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    outSetting->data.uid = uid;
    outSetting->data.presencePermission = m_Record.presencePermission;
    outSetting->data.playLogPermission = m_Record.playLogPermission;
    outSetting->data.friendRequestReception = m_Record.friendRequestReception;
    outSetting->data.friendCode = m_Record.friendCode;
    outSetting->data.friendCodeNextIssuableTime = m_Record.friendCodeNextIssuableTime;
    outSetting->data.profileExtra = m_Record.profileExtra;

    NN_RESULT_SUCCESS;
}

nn::Result UserSettingManager::Write(const nn::account::Uid& uid, const UserResource& resource) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    m_Record.presencePermission = resource.presencePermission;
    m_Record.playLogPermission = resource.playLogPermission;
    m_Record.friendRequestReception = resource.friendRequestReception;
    m_Record.friendCode = resource.friendCode;
    m_Record.friendCodeNextIssuableTime = resource.friendCodeNextIssuableTime;
    m_Record.profileExtra = resource.profileExtra;

    NormalizePlayLogPermission();

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    // 新規にデータを作成する際、ストレージの容量不足にならないよう、不要なファイルを削除する。
    NN_RESULT_DO(AccountStorageManager::GetInstance().DeleteUnmanagedNetworkServiceAccountDirectory());
    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    {
        NN_RESULT_TRY(nn::fs::DeleteFile(path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (m_Record)));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &m_Record, sizeof (m_Record),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    NN_RESULT_SUCCESS;
}

void UserSettingManager::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_CurrentUid = nn::account::InvalidUid;
}

nn::Result UserSettingManager::Load(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadImpl(uid))
        NN_RESULT_CATCH_ALL
        {
            std::memset(&m_Record, 0, sizeof (m_Record));
        }
    NN_RESULT_END_TRY;

    m_CurrentUid = uid;

    NormalizePlayLogPermission();

    NN_RESULT_SUCCESS;
}

nn::Result UserSettingManager::LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT
{
    if (uid == m_CurrentUid)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, &m_Record, sizeof (m_Record)));

    NN_RESULT_SUCCESS;
}

void UserSettingManager::NormalizePlayLogPermission() NN_NOEXCEPT
{
    if (m_Record.playLogPermission == ExtraPermission_Unconfigured)
    {
        // リーガル要求により、子どもアカウントの場合はフレンドまでの公開範囲とする。
        if (Account::IsChild(m_CurrentUid))
        {
            m_Record.playLogPermission = ExtraPermission_Friends;
        }
        else
        {
            m_Record.playLogPermission = ExtraPermission_Everyone;
        }
    }
}

}}}}}
