﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_PlayHistoryManager.h>
#include <nn/friends/detail/service/core/friends_UuidManager.h>
#include <nn/crypto.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* HistoryFileName = "history.bin";
    const char* HistoryStatsFileName = "history.stats.bin";

    const size_t HashKeyCount = 8;
    const size_t HashKeySize = 32;

    // HMAC-SHA256 用鍵
    const Bit8 g_HashKey[HashKeyCount][HashKeySize] =
    {
        {
            0x5C, 0x00, 0x93, 0x9F, 0x37, 0x6E, 0x46, 0xB5, 0x9F, 0xCB, 0xD3, 0xA3, 0x51, 0x62, 0x75, 0x5F,
            0x3B, 0x77, 0x72, 0x85, 0x1B, 0xF1, 0x47, 0x12, 0xAF, 0x33, 0x88, 0x12, 0x03, 0xA0, 0x05, 0x5D
        },
        {
            0xD5, 0xB8, 0xCD, 0x15, 0x04, 0xF0, 0x49, 0xA1, 0x96, 0xCD, 0x23, 0x1C, 0x94, 0x93, 0xED, 0xEB,
            0xDF, 0xE2, 0x08, 0x38, 0x2D, 0xEA, 0x43, 0xE3, 0x97, 0x3A, 0x80, 0x48, 0xA1, 0xBD, 0x8D, 0x08
        },
        {
            0x82, 0x38, 0xCC, 0x93, 0x79, 0x85, 0x44, 0xC1, 0x88, 0x24, 0x0F, 0xBA, 0xC0, 0x5C, 0x2F, 0xC4,
            0xB9, 0x13, 0xF9, 0x16, 0x3B, 0x8E, 0x4F, 0x1D, 0xB2, 0x1A, 0x3B, 0x77, 0xF4, 0x05, 0x0D, 0xB7
        },
        {
            0x7C, 0xAB, 0xF7, 0x5B, 0x04, 0xB6, 0x42, 0x96, 0x9A, 0xB7, 0x08, 0xB1, 0xF5, 0x9F, 0x08, 0x05,
            0x7C, 0xCF, 0x1B, 0x3E, 0xA4, 0x7C, 0x4E, 0xAC, 0xB1, 0xB5, 0x1C, 0xF0, 0xD0, 0x64, 0x7E, 0x58
        },
        {
            0xA6, 0x89, 0x5B, 0x47, 0xD8, 0xCE, 0x44, 0x5F, 0xB7, 0xD0, 0x2A, 0x1E, 0x92, 0x30, 0x38, 0x3F,
            0x3C, 0x97, 0xB8, 0xC7, 0x74, 0x66, 0x44, 0x19, 0x9D, 0x56, 0xEF, 0x89, 0x9F, 0xD1, 0x65, 0x0C
        },
        {
            0x8F, 0xAD, 0x70, 0x0E, 0xD6, 0xAE, 0x48, 0x8B, 0xA8, 0x71, 0x82, 0x1B, 0x44, 0xCB, 0xDB, 0x1C,
            0x79, 0x6A, 0x66, 0x1F, 0xC3, 0xCE, 0x49, 0x80, 0xAA, 0x78, 0xD2, 0x33, 0x49, 0xF6, 0xB7, 0x49
        },
        {
            0xD2, 0x39, 0x20, 0x21, 0x43, 0xAC, 0x4F, 0xD0, 0x93, 0x27, 0xFF, 0xCD, 0x36, 0x47, 0x14, 0x1A,
            0xE7, 0x7A, 0x82, 0x7E, 0xD8, 0x45, 0x42, 0xCE, 0xAA, 0xDF, 0xB3, 0xB1, 0xFC, 0xAA, 0x00, 0x0F
        },
        {
            0xFB, 0xDC, 0x7C, 0x99, 0x92, 0xEC, 0x41, 0x59, 0x8C, 0x75, 0x8D, 0x67, 0x3D, 0xE5, 0xBF, 0xCE,
            0x97, 0x59, 0xC3, 0x5F, 0x08, 0xE4, 0x4B, 0x6E, 0x9E, 0x11, 0xE3, 0x67, 0x8C, 0x84, 0x7D, 0x1F
        }
    };
}

PlayHistoryManager::PlayHistoryManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_CurrentUid(nn::account::InvalidUid),
    m_Count(0),
    m_IsDirty(false),
    m_Statistics()
{
}

nn::Result PlayHistoryManager::GetPlayHistoryRegistrationKey(PlayHistoryRegistrationKey* outKey,
    const nn::account::Uid& uid, bool isLocalPlayed) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outKey);

    nn::account::NetworkServiceAccountId accountId = {};
    bool hasNetworkServiceAccountId = true;

    NN_RESULT_TRY(Account::GetNetworkServiceAccountId(&accountId, uid))
        NN_RESULT_CATCH_ALL
        {
            hasNetworkServiceAccountId = false;
        }
    NN_RESULT_END_TRY;

    PlayHistoryRegistrationKeyData data = {};

    // バージョンは 1 で固定。
    data.version = 1;
    // 1=NX とする。
    data.platform = 1;
    // 使用するキーを取得毎に変える。
    data.hashKeyIndex = static_cast<Bit8>(detail::service::util::GetRandom() % HashKeyCount);

    data.isLocalPlayed = isLocalPlayed;

    data.accountId = accountId;
    data.hasNetworkServiceAccountId = hasNetworkServiceAccountId;

    NN_RESULT_DO(UuidManager::GetInstance().Get(&data.uuid, uid));

    // 前半 32 バイトのハッシュをとる。
    nn::crypto::GenerateHmacSha256Mac(data.hash, sizeof (data.hash), &data, 32, &g_HashKey[data.hashKeyIndex], HashKeySize);

    std::memcpy(outKey, &data, sizeof (data));

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::GetPlayHistoryRegistrationKey(PlayHistoryRegistrationKey* outKey,
    nn::account::NetworkServiceAccountId accountId, bool isLocalPlayed) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outKey);

    PlayHistoryRegistrationKeyData data = {};

    // バージョンは 1 で固定。
    data.version = 1;
    // 1=NX とする。
    data.platform = 1;
    // 使用するキーを取得毎に変える。
    data.hashKeyIndex = static_cast<Bit8>(detail::service::util::GetRandom() % HashKeyCount);

    data.isLocalPlayed = isLocalPlayed;

    data.accountId = accountId;
    data.hasNetworkServiceAccountId = true;

    data.uuid = nn::util::GenerateUuid();

    // 前半 32 バイトのハッシュをとる。
    nn::crypto::GenerateHmacSha256Mac(data.hash, sizeof (data.hash), &data, 32, &g_HashKey[data.hashKeyIndex], HashKeySize);

    std::memcpy(outKey, &data, sizeof (data));

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::GetAccountIdFromPlayHistoryRegistrationKey(nn::account::NetworkServiceAccountId* outAccountId,
    const PlayHistoryRegistrationKey& key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAccountId);

    NN_RESULT_THROW_UNLESS(VerifyPlayHistoryRegistrationKey(key), ResultPlayHistoryRegistrationKeyBroken());

    const PlayHistoryRegistrationKeyData* data = reinterpret_cast<const PlayHistoryRegistrationKeyData*>(&key);

    NN_RESULT_THROW_UNLESS(data->hasNetworkServiceAccountId, ResultNetworkServiceAccountNotExistsGeneral());

    *outAccountId = data->accountId;

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::AddPlayHistory(const nn::account::Uid& uid, const PlayHistoryRegistrationKey& key,
    const ApplicationInfo& appInfo, const InAppScreenName& name, const InAppScreenName& myName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(VerifyPlayHistoryRegistrationKey(key), ResultPlayHistoryRegistrationKeyBroken());

    nn::util::Uuid uuid;
    NN_RESULT_DO(UuidManager::GetInstance().Get(&uuid, uid));

    const PlayHistoryRegistrationKeyData* data = reinterpret_cast<const PlayHistoryRegistrationKeyData*>(&key);

    NN_RESULT_THROW_UNLESS(uuid != data->uuid, ResultOwnPlayHistoryRegistrationKey());

    if (data->hasNetworkServiceAccountId)
    {
        nn::account::NetworkServiceAccountId ownAccountId = {};

        if (detail::service::core::Account::GetNetworkServiceAccountId(&ownAccountId, uid).IsSuccess())
        {
            NN_RESULT_THROW_UNLESS(data->accountId != ownAccountId, ResultOwnNetworkServiceAccountSpecified());
        }
    }

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int index = SearchSameUser(key);

    nn::time::PosixTime currentTime = {};

    // ネットワーク時刻が補正できていない環境でいっしょに遊んでいる場合があることと、
    // 本体上に表示されている時間を入れた方が検索しやすいという点を考慮し、ユーザー時計を採用する。
    nn::time::StandardUserSystemClock::GetCurrentTime(&currentTime);

    Record record = {};

    record.accountId = data->accountId;
    record.uuid = data->uuid;
    record.playRecord.appInfo = appInfo;
    record.playRecord.name = name;
    record.playRecord.myName = myName;
    record.playRecord.time = currentTime;
    record.hasNetworkServiceAccountId = data->hasNetworkServiceAccountId;
    record.isLocalPlayed = data->isLocalPlayed;

    if (index != -1)
    {
        m_Count = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, index);
    }

    m_Count = detail::service::util::ArrayAccessor::InsertToTopEntry(m_Records, m_Count, PlayHistoryCountMax, record);

    if (data->isLocalPlayed)
    {
        m_Statistics.totalLocalPlayCount++;
    }
    else
    {
        m_Statistics.totalOnlinePlayCount++;
    }

    m_IsDirty = true;

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::Save() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_CurrentUid || !m_IsDirty)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(m_CurrentUid);

    NN_RESULT_DO(SaveHistory());
    NN_RESULT_DO(SaveHistoryStats());

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());
    m_IsDirty = false;

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::Delete(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK_IF_EXISTS(uid);

    char path[64];
    AccountStorageManager::GetInstance().MakePath(path, sizeof (path), HistoryFileName);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    AccountStorageManager::GetInstance().MakePath(path, sizeof (path), HistoryStatsFileName);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    if (uid == m_CurrentUid)
    {
        m_Count = 0;
        std::memset(&m_Statistics, 0, sizeof (m_Statistics));
    }

    m_IsDirty = false;

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::GetPlayHistoryList(int* outCount, PlayHistoryImpl* outHistories,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outHistories);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outHistories, 0, sizeof (PlayHistoryImpl) * count);

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = offset; i < m_Count; i++)
    {
        if (count-- == 0)
        {
            break;
        }

        PlayHistoryImpl& impl = outHistories[actualCount++];

        impl.data.accountId = m_Records[i].accountId;
        impl.data.playRecord = m_Records[i].playRecord;
        impl.data.hasNetworkServiceAccountId = m_Records[i].hasNetworkServiceAccountId;
        impl.data.isLocalPlayed = m_Records[i].isLocalPlayed;
        impl.data.isValid = true;
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::GetPlayHistoryStatistics(PlayHistoryStatistics* outStatistics,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outStatistics);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    std::memset(outStatistics, 0, sizeof (PlayHistoryStatistics));

    NN_RESULT_DO(Load(uid));

    *outStatistics = m_Statistics;

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::Load(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadImpl(uid))
        NN_RESULT_CATCH_ALL
        {
            m_Count = 0;
            std::memset(&m_Statistics, 0, sizeof (m_Statistics));
        }
    NN_RESULT_END_TRY;

    m_CurrentUid = uid;

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT
{
    if (uid == m_CurrentUid)
    {
        NN_RESULT_SUCCESS;
    }

    if (m_IsDirty)
    {
        NN_RESULT_DO(Save());
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    NN_RESULT_DO(LoadHistory(uid));
    NN_RESULT_DO(LoadHistoryStats(uid));

    NN_RESULT_SUCCESS;
}

void PlayHistoryManager::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_CurrentUid = nn::account::InvalidUid;
}

nn::Result PlayHistoryManager::LoadHistory(const nn::account::Uid& uid) NN_NOEXCEPT
{
    char path[64];
    AccountStorageManager::GetInstance().MakePath(path, sizeof (path), HistoryFileName);

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    if ((size % sizeof (Record)) != 0)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. ((size % sizeof (Record)) != 0)\n", HistoryFileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    int count = static_cast<int>(size / sizeof (Record));

    if (count > PlayHistoryCountMax)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (count > PlayHistoryCountMax)\n", HistoryFileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (Record) * count));

    if (!VerifyRecord(count))
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (verification failed)\n", HistoryFileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    m_Count = count;

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::LoadHistoryStats(const nn::account::Uid& uid) NN_NOEXCEPT
{
    char path[64];
    AccountStorageManager::GetInstance().MakePath(path, sizeof (path), HistoryStatsFileName);

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, &m_Statistics, sizeof (m_Statistics)));

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::SaveHistory() NN_NOEXCEPT
{
    char path[64];
    AccountStorageManager::GetInstance().MakePath(path, sizeof (path), HistoryFileName);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    if (m_Count == 0)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (Record) * m_Count));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (Record) * m_Count,
        nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

nn::Result PlayHistoryManager::SaveHistoryStats() NN_NOEXCEPT
{
    char path[64];
    AccountStorageManager::GetInstance().MakePath(path, sizeof (path), HistoryStatsFileName);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (m_Statistics)));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &m_Statistics, sizeof (m_Statistics),
        nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

int PlayHistoryManager::SearchSameUser(const PlayHistoryRegistrationKey& key) NN_NOEXCEPT
{
    const PlayHistoryRegistrationKeyData* data =
        reinterpret_cast<const PlayHistoryRegistrationKeyData*>(&key);

    if (data->hasNetworkServiceAccountId)
    {
        for (int i = 0; i < m_Count; i++)
        {
            if (data->accountId == m_Records[i].accountId)
            {
                return i;
            }
        }

        // ネットワークサービスアカウントを持っていない時にいっしょに遊んだユーザーに記録された可能性があるので、
        // return せずに UUID による比較も行う。
    }

    for (int i = 0; i < m_Count; i++)
    {
        if (data->uuid == m_Records[i].uuid)
        {
            return i;
        }
    }

    return -1;
}

bool PlayHistoryManager::VerifyPlayHistoryRegistrationKey(const PlayHistoryRegistrationKey& key) NN_NOEXCEPT
{
    const PlayHistoryRegistrationKeyData* data =
        reinterpret_cast<const PlayHistoryRegistrationKeyData*>(&key);

    // バージョンは 1 で固定。
    if (data->version != 1)
    {
        return false;
    }

    // TORIAEZU: 1=NX
    if (data->platform != 1)
    {
        return false;
    }

    if (!(data->hashKeyIndex >= 0 && data->hashKeyIndex < HashKeyCount))
    {
        return false;
    }

    Bit8 hash[32];

    // 前半 32 バイトのハッシュをとる。
    nn::crypto::GenerateHmacSha256Mac(hash, sizeof (hash), data, 32,
        &g_HashKey[data->hashKeyIndex], HashKeySize);

    if (std::memcmp(hash, data->hash, sizeof (hash)) != 0)
    {
        return false;
    }

    return true;
}

bool PlayHistoryManager::VerifyRecord(int count) NN_NOEXCEPT
{
    for (int i = 0; i < count; i++)
    {
        if (!DataVerifier::VerifyString(m_Records[i].playRecord.name.name, sizeof (m_Records[i].playRecord.name.name)))
        {
            return false;
        }
        if (!DataVerifier::VerifyString(m_Records[i].playRecord.myName.name, sizeof (m_Records[i].playRecord.myName.name)))
        {
            return false;
        }
        if (!DataVerifier::VerifyPosixTime(m_Records[i].playRecord.time))
        {
            return false;
        }
    }

    return true;
}

}}}}}
