﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_FriendListManager.h>
#include <nn/friends/detail/service/core/friends_FriendPresenceManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* FileName = "friend.cache";
}

namespace
{
    bool IsSamePresenceGroupApplication(const FriendPresenceImpl& presence, nn::Bit64 presenceGroupId) NN_NOEXCEPT
    {
        return (presence.data.lastPlayedApp.appId != nn::ApplicationId::GetInvalidId() &&
            presence.data.lastPlayedApp.presenceGroupId == presenceGroupId);
    }

    bool IsSamePlayRecord(const PlayRecord& lhs, const PlayRecord& rhs) NN_NOEXCEPT
    {
        return (lhs.appInfo.appId == rhs.appInfo.appId && lhs.appInfo.presenceGroupId == rhs.appInfo.presenceGroupId &&
            nn::util::Strncmp(lhs.name.name, rhs.name.name, sizeof (lhs.name.name)) == 0 &&
            nn::util::Strncmp(lhs.name.language.string, rhs.name.language.string, sizeof (lhs.name.language.string)) == 0 &&
            nn::util::Strncmp(lhs.myName.name, rhs.myName.name, sizeof (lhs.myName.name)) == 0 &&
            nn::util::Strncmp(lhs.myName.language.string, rhs.myName.language.string, sizeof (lhs.myName.language.string)) == 0 &&
            lhs.time == rhs.time);
    }
}

FriendListManager::FriendListManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_CurrentUid(nn::account::InvalidUid),
    m_Count(0),
    m_IsDirty(false)
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_FriendCountCache); i++)
    {
        m_FriendCountCache[i].Initialize(nn::account::InvalidUid);
    }
}

nn::Result FriendListManager::GetFriendList(int* outCount, nn::account::NetworkServiceAccountId* outAccountIds,
    const nn::account::Uid& uid, int offset, int count, const detail::ipc::SizedFriendFilter& filter,
    const ApplicationInfo& appInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outAccountIds);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outAccountIds, 0, sizeof (nn::account::NetworkServiceAccountId) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int16_t sorted[FriendCountMax] = {};
    GetSortIndexes(sorted);

    int actualCount = 0;

    for (int i = 0; i < m_Count; i++)
    {
        if (!Filter(sorted[i], filter, appInfo))
        {
            continue;
        }
        if (offset > 0)
        {
            offset--;
            continue;
        }

        outAccountIds[actualCount++] = m_Records[sorted[i]].accountId;

        if (actualCount == count)
        {
            break;
        }
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::GetFriendList(int* outCount, FriendImpl* outFriends,
    const nn::account::Uid& uid, int offset, int count, const detail::ipc::SizedFriendFilter& filter,
    const ApplicationInfo& appInfo, bool checkAppFieldPermission) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outFriends);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outFriends, 0, sizeof (FriendImpl) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int16_t sorted[FriendCountMax] = {};
    GetSortIndexes(sorted);

    int actualCount = 0;

    for (int i = 0; i < m_Count; i++)
    {
        if (!Filter(sorted[i], filter, appInfo))
        {
            continue;
        }
        if (offset > 0)
        {
            offset--;
            continue;
        }

        FriendImpl& impl = outFriends[actualCount++];

        impl.data.uid = uid;
        impl.data.accountId = m_Records[sorted[i]].accountId;
        impl.data.nickname = m_Records[sorted[i]].nickname;

        FriendPresenceManager::GetInstance().GetPresence(&impl.data.presence, uid, m_Records[sorted[i]].accountId);

        impl.data.presence.data.isSamePresenceGroupApp = IsSamePresenceGroupApplication(impl.data.presence, appInfo.presenceGroupId);

        if (checkAppFieldPermission && !impl.data.presence.data.isSamePresenceGroupApp)
        {
            std::memset(impl.data.presence.data.appField, 0, sizeof (impl.data.presence.data.appField));
        }

        impl.data.favorite = m_Records[sorted[i]].isFavorite;
        impl.data.newly = m_Records[sorted[i]].isNewly;
        impl.data.isValid = true;

        if (actualCount == count)
        {
            break;
        }
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::GetFriendCount(int* outCount,
    const nn::account::Uid& uid, const detail::ipc::SizedFriendFilter& filter, const ApplicationInfo& appInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!filter.isFavoriteOnly && !filter.isSameAppPlayedOnly && !filter.isArbitraryAppPlayedOnly)
    {
        if (filter.presenceStatus == detail::ipc::SizedPresenceStatusFilter_None)
        {
            int index = SearchFriendCountCache(uid);

            if (index == -1)
            {
                *outCount = 0;
            }
            else
            {
                if (!m_FriendCountCache[index].isSet)
                {
                    NN_RESULT_DO(UpdateFriendCountCache(uid));
                }

                *outCount = m_FriendCountCache[index].total;
            }
        }
        else
        {
            // フレンドリストのデータにアクセスしない場合は、ファイル読み込み処理をスキップする。
            FriendPresenceManager::GetInstance().GetCount(outCount, uid, static_cast<PresenceStatusFilter>(filter.presenceStatus));
        }

        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = 0; i < m_Count; i++)
    {
        if (Filter(i, filter, appInfo))
        {
            actualCount++;
        }
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::GetNewlyFriendCount(int* outCount,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchFriendCountCache(uid);

    if (index == -1)
    {
        *outCount = 0;
    }
    else
    {
        if (!m_FriendCountCache[index].isSet)
        {
            NN_RESULT_DO(UpdateFriendCountCache(uid));
        }

        *outCount = m_FriendCountCache[index].newlyArrived;
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::GetFriendInfo(FriendImpl* outFriends,
    const nn::account::Uid& uid, const nn::account::NetworkServiceAccountId* accountIds, int count,
    const ApplicationInfo& appInfo, bool checkAppFieldPermission) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outFriends);
    NN_SDK_REQUIRES_NOT_NULL(accountIds);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outFriends, 0, sizeof (FriendImpl) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    for (int i = 0; i < count; i++)
    {
        int index = SearchRecord(accountIds[i]);

        if (index != -1)
        {
            FriendImpl& impl = outFriends[i];

            impl.data.uid = uid;
            impl.data.accountId = m_Records[index].accountId;
            impl.data.nickname = m_Records[index].nickname;

            FriendPresenceManager::GetInstance().GetPresence(&impl.data.presence, uid, m_Records[index].accountId);

            impl.data.presence.data.isSamePresenceGroupApp = IsSamePresenceGroupApplication(impl.data.presence, appInfo.presenceGroupId);

            if (checkAppFieldPermission && !impl.data.presence.data.isSamePresenceGroupApp)
            {
                std::memset(impl.data.presence.data.appField, 0, sizeof (impl.data.presence.data.appField));
            }

            impl.data.favorite = m_Records[index].isFavorite;
            impl.data.newly = m_Records[index].isNewly;
            impl.data.isValid = true;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::GetFriendDetailedInfo(FriendDetailedInfoImpl* outInfo,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInfo);

    std::memset(outInfo, 0, sizeof (FriendDetailedInfoImpl));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int index = SearchRecord(accountId);

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotFriend());

    FriendDetailedInfoImpl& impl = *outInfo;

    impl.data.uid = uid;
    impl.data.profileExtra.data.accountId = m_Records[index].accountId;
    impl.data.profileExtra.data.nickname = m_Records[index].nickname;
    impl.data.profileExtra.data.profileImageUrl = m_Records[index].profileImageUrl;

    std::memcpy(impl.data.profileExtra.data.playLog, m_Records[index].playLog, sizeof (PlayLog) * PlayLogCountMax);

    impl.data.lastPlayRecord = m_Records[index].lastPlayRecord;
    impl.data.requestType = m_Records[index].requestType;
    impl.data.requestRouteInfo = m_Records[index].requestRouteInfo;
    impl.data.requestRouteInfoExtra = m_Records[index].requestRouteInfoExtra;
    impl.data.isValid = true;

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::GetLocalUpdatedRecord(int* outCount, Record* outRecords,
    const nn::account::Uid& uid, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outRecords);

    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidArgument());

    std::memset(outRecords, 0, sizeof (Record) * count);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int actualCount = 0;

    for (int i = 0; i < m_Count; i++)
    {
        if (m_Records[i].isNewlyConfirmed || m_Records[i].isLastPlayRecordUpdated)
        {
            std::memcpy(&outRecords[actualCount++], &m_Records[i], sizeof (Record));
        }
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::GetFriendSetting(FriendSettingImpl* outSetting,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSetting);

    std::memset(outSetting, 0, sizeof (FriendSettingImpl));

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int index = SearchRecord(accountId);

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotFriend());

    FriendSettingImpl impl = {};

    impl.data.uid = uid;
    impl.data.accountId = m_Records[index].accountId;
    impl.data.favorite = m_Records[index].isFavorite;
    impl.data.newly = m_Records[index].isNewly;
    impl.data.onlineNotification = m_Records[index].isOnlineNotification;

    std::memcpy(outSetting, &impl, sizeof (impl));

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::DropNewly(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int index = SearchRecord(accountId);

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotFriend());

    if (m_Records[index].isNewly)
    {
        m_Records[index].isNewly = false;
        m_Records[index].isNewlyConfirmed = true;

        NN_RESULT_DO(Save(index));
        NN_RESULT_DO(UpdateFriendCountCache(uid));
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::DropNewlyAll(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    bool isDirty = false;

    for (int i = 0; i < m_Count; i++)
    {
        if (m_Records[i].isNewly)
        {
            m_Records[i].isNewly = false;
            m_Records[i].isNewlyConfirmed = true;
            isDirty = true;
        }
    }

    if (isDirty)
    {
        NN_RESULT_DO(SaveAll());
        NN_RESULT_DO(UpdateFriendCountCache(uid));
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::UpdateLastPlayRecord(const nn::account::Uid& uid,
    nn::account::NetworkServiceAccountId accountId, const PlayRecord& lastPlayRecord) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int index = SearchRecord(accountId);

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotFriend());

    if (!IsSamePlayRecord(lastPlayRecord, m_Records[index].lastPlayRecord))
    {
        m_Records[index].lastPlayRecord = lastPlayRecord;
        m_Records[index].isLastPlayRecordUpdated = true;

        // AddPlayHistory の処理を重くしないために、ストレージへの書き込みは遅延させる。
        m_IsDirty = true;
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::UpdateFriend(const nn::account::Uid& uid, const FriendResource& resource, bool isLocalChangeApplied) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int index = SearchRecord(resource.accountId);

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotFriend());

    Record& record = m_Records[index];

    if (isLocalChangeApplied)
    {
        record.isNewlyConfirmed = false;
        record.isLastPlayRecordUpdated = false;
    }

    record.accountId = resource.accountId;
    record.nickname = resource.nickname;
    record.profileImageUrl = resource.profileImageUrl;

    std::memcpy(record.playLog, resource.playLog, sizeof (PlayLog) * PlayLogCountMax);

    if (!record.isLastPlayRecordUpdated)
    {
        record.lastPlayRecord = resource.lastPlayRecord;
    }

    record.requestType = resource.requestType;
    record.isFavorite = resource.isFavorite;

    if (!record.isNewlyConfirmed)
    {
        record.isNewly = resource.isNewly;
    }

    record.isOnlineNotification = resource.isOnlineNotification;
    record.requestRouteInfo = resource.requestRouteInfo;
    record.requestRouteInfoExtra = resource.requestRouteInfoExtra;

    NN_RESULT_DO(Save(index));
    NN_RESULT_DO(UpdateFriendCountCache(uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::UpdateFriendList(const nn::account::Uid& uid, const FriendResource* resources, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(resources);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    bool isNewlyConfirmed[FriendCountMax] = {};

    for (int i = 0; i < count; i++)
    {
        for (int j = 0; j < m_Count; j++)
        {
            if (resources[i].accountId == m_Records[j].accountId)
            {
                // フレンドリストの同期によって新着フラグを落としている間にローカルの新着フラグが更新された場合、ローカルの更新フラグを維持する。
                if (resources[i].isNewly)
                {
                    isNewlyConfirmed[i] = m_Records[j].isNewlyConfirmed;
                }
                break;
            }
        }
    }

    for (int i = 0; i < count; i++)
    {
        Record& record = m_Records[i];

        record.isNewlyConfirmed = isNewlyConfirmed[i];
        record.isLastPlayRecordUpdated = false;

        record.accountId = resources[i].accountId;
        record.nickname = resources[i].nickname;
        record.profileImageUrl = resources[i].profileImageUrl;

        std::memcpy(record.playLog, resources[i].playLog, sizeof (PlayLog) * PlayLogCountMax);

        record.lastPlayRecord = resources[i].lastPlayRecord;
        record.requestType = resources[i].requestType;
        record.isFavorite = resources[i].isFavorite;

        if (record.isNewlyConfirmed)
        {
            record.isNewly = false;
        }
        else
        {
            record.isNewly = resources[i].isNewly;
        }

        record.isOnlineNotification = resources[i].isOnlineNotification;
        record.requestRouteInfo = resources[i].requestRouteInfo;
        record.requestRouteInfoExtra = resources[i].requestRouteInfoExtra;
    }

    m_Count = count;

    NN_RESULT_DO(SaveAll());
    NN_RESULT_DO(UpdateFriendCountCache(uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::DeleteFriend(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int index = SearchRecord(accountId);

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotFriend());

    m_Count = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, index);

    NN_RESULT_DO(SaveAll());
    NN_RESULT_DO(UpdateFriendCountCache(uid));

    NN_RESULT_SUCCESS;
}

void FriendListManager::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_IsDirty)
    {
        SaveAll();
    }

    m_CurrentUid = nn::account::InvalidUid;
}

void FriendListManager::NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = -1;

    for (int i = 0; i < NN_ARRAY_SIZE(m_FriendCountCache); i++)
    {
        if (m_FriendCountCache[i].uid == nn::account::InvalidUid)
        {
            index = i;
            break;
        }
    }

    if (index == -1)
    {
        return;
    }

    m_FriendCountCache[index].Initialize(uid);
}

void FriendListManager::NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = SearchFriendCountCache(uid);

    if (index != -1)
    {
        m_FriendCountCache[index].Initialize(nn::account::InvalidUid);
    }
}

void FriendListManager::NotifyNetworkServiceAccountAvailabilityChanged(const nn::account::Uid& uid, bool isAvailable) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!isAvailable)
    {
        int index = SearchFriendCountCache(uid);

        if (index != -1)
        {
            m_FriendCountCache[index].ResetCount();
        }
    }
}

nn::Result FriendListManager::Load(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadImpl(uid))
        NN_RESULT_CATCH_ALL
        {
            m_Count = 0;
        }
    NN_RESULT_END_TRY;

    m_CurrentUid = uid;

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT
{
    if (uid == m_CurrentUid)
    {
        NN_RESULT_SUCCESS;
    }

    if (m_IsDirty)
    {
        NN_RESULT_DO(SaveAll());
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    // NN_DETAIL_FRIENDS_INFO("[friends] friends_%016llx%016llx was mounted.\n", uid._data[0], uid._data[1]);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    // NN_DETAIL_FRIENDS_INFO("[friends] friends_%016llx%016llx/%s was opened.\n", uid._data[0], uid._data[1], FileName);

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    if ((size % sizeof (Record)) != 0)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. ((size % sizeof (Record)) != 0)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    int count = static_cast<int>(size / sizeof (Record));

    if (count > FriendCountMax)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (count > FriendCountMax)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (Record) * count));
    m_Count = count;

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::Save(int index) NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(m_CurrentUid);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    {
        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, sizeof (Record) * index, &m_Records[index], sizeof (Record),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::SaveAll() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(m_CurrentUid);

    // 新規にデータを作成する際、ストレージの容量不足にならないよう、不要なファイルを削除する。
    NN_RESULT_DO(AccountStorageManager::GetInstance().DeleteUnmanagedNetworkServiceAccountDirectory());
    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    if (m_Count == 0)
    {
        NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());
        m_IsDirty = false;

        NN_RESULT_SUCCESS;
    }

    {
        NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (Record) * m_Count));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (Record) * m_Count,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());
    m_IsDirty = false;

    NN_RESULT_SUCCESS;
}

nn::Result FriendListManager::UpdateFriendCountCache(const nn::account::Uid& uid) NN_NOEXCEPT
{
    int index = SearchFriendCountCache(uid);

    if (index == -1)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(Load(uid));

    int newlyArrivedCount = 0;

    for (int i = 0; i < m_Count; i++)
    {
        if (m_Records[i].isNewly)
        {
            newlyArrivedCount++;
        }
    }

    m_FriendCountCache[index].Set(m_Count, newlyArrivedCount);

    NN_RESULT_SUCCESS;
}

bool FriendListManager::Filter(int index, const detail::ipc::SizedFriendFilter& filter, const ApplicationInfo& appInfo) NN_NOEXCEPT
{
    FriendPresenceImpl presence;
    FriendPresenceManager::GetInstance().GetPresence(&presence, m_CurrentUid, m_Records[index].accountId);

    switch (filter.presenceStatus)
    {
    case detail::ipc::SizedPresenceStatusFilter_Online:
        {
            if (presence.data.status != PresenceStatus_Online)
            {
                return false;
            }
        }
        break;
    case detail::ipc::SizedPresenceStatusFilter_OnlinePlay:
        {
            if (presence.data.status != PresenceStatus_OnlinePlay)
            {
                return false;
            }
        }
        break;
    case detail::ipc::SizedPresenceStatusFilter_OnlineOrOnlinePlay:
        {
            if (!(presence.data.status == PresenceStatus_Online || presence.data.status == PresenceStatus_OnlinePlay))
            {
                return false;
            }
        }
        break;
    default:
        break;
    }

    if (filter.isFavoriteOnly && !m_Records[index].isFavorite)
    {
        return false;
    }

    if (filter.isSameAppPresenceOnly)
    {
        if (!IsSamePresenceGroupApplication(presence, appInfo.presenceGroupId))
        {
            return false;
        }
    }

    if (filter.isSameAppPlayedOnly)
    {
        if (!IsSamePresenceGroupApplication(presence, appInfo.presenceGroupId))
        {
            bool isFound = false;

            for (int i = 0; i < NN_ARRAY_SIZE(m_Records[index].playLog); i++)
            {
                if (m_Records[index].playLog[i].appInfo.appId == nn::ApplicationId::GetInvalidId())
                {
                    break;
                }
                if (m_Records[index].playLog[i].appInfo.presenceGroupId == appInfo.presenceGroupId)
                {
                    isFound = true;
                    break;
                }
            }
            if (!isFound)
            {
                return false;
            }
        }
    }

    if (filter.isArbitraryAppPlayedOnly)
    {
        if (!IsSamePresenceGroupApplication(presence, filter.presenceGroupId))
        {
            bool isFound = false;

            for (int i = 0; i < NN_ARRAY_SIZE(m_Records[index].playLog); i++)
            {
                if (m_Records[index].playLog[i].appInfo.appId == nn::ApplicationId::GetInvalidId())
                {
                    break;
                }
                if (m_Records[index].playLog[i].appInfo.presenceGroupId == filter.presenceGroupId)
                {
                    isFound = true;
                    break;
                }
            }
            if (!isFound)
            {
                return false;
            }
        }
    }

    return true;
}

int FriendListManager::SearchRecord(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (m_Records[i].accountId == accountId)
        {
            return i;
        }
    }

    return -1;
}

int FriendListManager::SearchFriendCountCache(const nn::account::Uid& uid) NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_FriendCountCache); i++)
    {
        if (m_FriendCountCache[i].uid == uid)
        {
            return i;
        }
    }

    return -1;
}

void FriendListManager::GetSortIndexes(int16_t* outIndexes) NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        outIndexes[i] = static_cast<int16_t>(i);
    }

    // 作成時刻の昇順でソートした後、お気に入りを最優先にする。
    std::sort(outIndexes, outIndexes + m_Count,
        [this](const int16_t& lhs, const int16_t& rhs)
        {
            return this->m_Records[lhs].requestRouteInfo.time > this->m_Records[rhs].requestRouteInfo.time;
        });
    std::sort(outIndexes, outIndexes + m_Count,
        [this](const int16_t& lhs, const int16_t& rhs)
        {
            return this->m_Records[lhs].isFavorite && !this->m_Records[rhs].isFavorite;
        });
}

}}}}}
