﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <nn/util/util_Endian.h>

#if defined(NN_FONTLL2_BUILD_FONTLL2)
namespace nn { namespace fontll2 { namespace detail {
#else
namespace nn { namespace fontll { namespace detail {
#endif

    //---------------------------------------------------------------------------
    //! @brief bfttf形式のフォントをデコードするクラスです。
    //---------------------------------------------------------------------------
    class BfttfDecoder
    {
    public:
        // プラットフォームごとに暗号化キーを変えます。
#if defined(NN_BUILD_CONFIG_SPEC_NX)
        static const uint32_t ScrambleKey = 0x49621806;
#elif defined(NN_BUILD_CONFIG_SPEC_GENERIC)
        static const uint32_t ScrambleKey = 0xa6018502;
#else
        static const uint32_t ScrambleKey = 0x8cf2dcd9;
#endif
        static const uint32_t ScrambleKeyGeneric = 0xa6018502;
        static const uint32_t ScrambleSignature = 0x7f9a0218;

        //! @brief デコードを行います。
        //!
        //! @param[in] pData    デコードするデータ。内部で変更される場合があります。
        //! @param[in] dataSize デコードするデータのサイズ。
        //! @return デコードしたデータ。NULLが返された場合は、デコードに失敗したことを示します。
        //!
        static void* Decode(void* pData, uint32_t dataSize)
        {
            if (dataSize < 8)
            {
                // データサイズが8より小さい場合は不正
                return NULL;
            }
            uint32_t* data = static_cast<uint32_t*>(pData);
            if (CorrectEndian(data[0]) == ScrambleSignature)
            {
                // デコード済み、何もせずに返す
                return reinterpret_cast<void*>(&data[2]);
            }
            else
            {
                uint32_t scrambleKey = ScrambleKey;
                // Generic 用の bfttf はどのプラットフォームでも読み込めるようにします。
                if ((CorrectEndian(data[0]) ^ ScrambleKeyGeneric) == ScrambleSignature)
                {
                    scrambleKey = ScrambleKeyGeneric;
                }
                // デコードしていない
                if ((CorrectEndian(data[0]) ^ scrambleKey) == ScrambleSignature)
                {
                    // 正しいbffntのデータ
                    // ttfのサイズを取得
                    uint32_t ttfSize = CorrectEndian(data[1]) ^ scrambleKey;
                    if (dataSize < ttfSize)
                    {
                        // ttfのサイズがデータサイズより大きい場合は不正
                        return NULL;
                    }
                    // スクランブル解除
                    uint32_t signature = ScrambleSignature;
                    data[0] = CorrectEndian(signature);
                    uint32_t loopEnd = (ttfSize + 3) / 4  + 2;
                    for (uint32_t i = 2; i < loopEnd; i++)
                    {
                        data[i] = CorrectEndian(CorrectEndian(data[i]) ^ scrambleKey);
                    }
                    return reinterpret_cast<void*>(&data[2]);
                }
                else
                {
                    // 正しいデータではない
                    return NULL;
                }
            }
        }
    private:
        // 記録されているデータのエンディアンを実行環境のエンディアンに直します。
        // データはビッグエンディアンで記録されています。
        template<typename T>
        static T CorrectEndian(const T& x)
        {
#if defined( NN_BUILD_CONFIG_ENDIAN_BIG )
            {
                return x;
            }
#else
            {
                T result = x;
                nn::util::SwapEndian(&result);
                return result;
            }
#endif
        }

    };


}}}
