﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

namespace nn     {
namespace fgm    {
namespace server {

template <class T>
class Cbuf
{

private:
    T *           m_Array;
    uint32_t      m_ArraySize;
    uint32_t      m_ReadIndex;
    uint32_t      m_WriteIndex;
    uint32_t      m_Lost;
    nn::os::Mutex m_Lock;

public:

    Cbuf()
    NN_NOEXCEPT :
        m_Array(nullptr),
        m_ArraySize(0),
        m_ReadIndex(0),
        m_WriteIndex(0),
        m_Lost(0),
        m_Lock(false)
    {

    }

    uint32_t CbufRemaining()
    NN_NOEXCEPT
    {
        return (m_WriteIndex + m_ArraySize - m_ReadIndex) % m_ArraySize;
    }

    uint32_t CbufLost()
    NN_NOEXCEPT
    {
        uint32_t lost = m_Lost;
        m_Lost = 0;
        return lost;
    }

    void CbufWrite(T * pEntryIn)
    NN_NOEXCEPT
    {
        uint32_t readIndex  = m_ReadIndex;
        uint32_t writeIndex = m_WriteIndex;
        uint32_t emptyCount = (readIndex + m_ArraySize - writeIndex - 1) % m_ArraySize;
        if (emptyCount)
        {
            m_Array[writeIndex] = *pEntryIn;
            m_WriteIndex = (++writeIndex) % m_ArraySize;
        }
        else
        {
            // overwrite oldest entry
            std::lock_guard<nn::os::Mutex> lock(m_Lock);
            m_Array[writeIndex] = *pEntryIn;
            m_ReadIndex  = (++readIndex)  % m_ArraySize;
            m_WriteIndex = (++writeIndex) % m_ArraySize;
            m_Lost++;
        }
    }

    uint32_t CbufRead(T * pEntryOut)
    NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Lock);
        uint32_t readIndex  = m_ReadIndex;
        uint32_t writeIndex = m_WriteIndex;
        uint32_t totalCount = (writeIndex + m_ArraySize - readIndex) % m_ArraySize;
        if (totalCount > 0)
        {
            *pEntryOut = m_Array[readIndex];
            m_ReadIndex  = (++readIndex) % m_ArraySize;
            return 1;
        }
        return 0;
    }

    nn::Result Initialize(T * pArrayIn, uint32_t arraySize)
    NN_NOEXCEPT
    {
        m_Array      = pArrayIn;
        m_ArraySize  = arraySize;
        m_ReadIndex  = 0;
        m_WriteIndex = 0;
        return ResultSuccess();
    }

    nn::Result Finalize()
    NN_NOEXCEPT
    {
        m_Array     = nullptr;
        m_ArraySize = 0;
        return ResultSuccess();
    }
};

}}}
