﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/fatal/detail/fatal_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/pl/pl_SharedFontApi.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "fatalsrv_Font.h"

namespace nn { namespace fatalsrv{

    Result TextRenderer::Initialize(void* workBuffer, size_t workBufferSize, int stride) NN_NOEXCEPT
    {
        m_pFontEngineBuffer = workBuffer;
        m_FontEngineBufferSize = workBufferSize;

        m_ScaleX = m_ScaleY = 1;

        auto sharedFont = GetSharedFont();
        m_pFont = sharedFont.pointer;

        m_Color.value = 0xffffffff;
        m_Stride = stride;
        NN_RESULT_SUCCESS;
    }

    void TextRenderer::SetDisplayBuffer(void* displayBuffer, int width, int height) NN_NOEXCEPT
    {
        m_pDisplayBuffer = displayBuffer;
        m_DisplayWidth = width;
        m_DIsplayHeight = height;
    }

    void TextRenderer::SetFontSize(int fontSize) NN_NOEXCEPT
    {
        m_FontSize = fontSize;
    }

    void TextRenderer::SetColor(const Color& color) NN_NOEXCEPT
    {
        m_Color = color;
    }

    void TextRenderer::SetNewLineHeight(int height) NN_NOEXCEPT
    {
        m_NewLineHeight = height;
    }

    void TextRenderer::SetScaleX(float scaleX) NN_NOEXCEPT
    {
        m_ScaleX = scaleX;
    }
    void TextRenderer::SetScaleY(float scaleY) NN_NOEXCEPT
    {
        m_ScaleY = scaleY;
    }

    uint32_t TextRenderer::PutString(int x, int y, const char* format, ...) NN_NOEXCEPT
    {
        const int MaxBufferSize = 512 * 2;
        char text[MaxBufferSize] = {};
        uint32_t utf32buffer[MaxBufferSize] = {};

        std::va_list vaList;
        va_start(vaList, format);
        nn::util::VSNPrintf(text, sizeof(text) - 1, format, vaList);
        va_end(vaList);

        int utf32BufferLength;
        auto encodingResult = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&utf32BufferLength, text);
        if (encodingResult != nn::util::CharacterEncodingResult_Success)
        {
            return 0;
        }

        nn::util::ConvertStringUtf8ToUtf32(utf32buffer, sizeof(utf32buffer) / sizeof(uint32_t), text);

        uint32_t posX = x;
        uint32_t posY = y;
        for (int i = 0; i < utf32BufferLength; ++i)
        {
            if (utf32buffer[i] == 0x0000000a)
            {
                posX = x;
                posY += m_NewLineHeight;
                continue;
            }
            posX += PutChar(posX, posY, utf32buffer[i]);
        }

        return posX;
    }

    uint32_t TextRenderer::PutChar(int x, int y, uint32_t charCode) NN_NOEXCEPT
    {
        auto fontSizeX = static_cast<int>(m_FontSize * m_ScaleX);
        auto fontSizeY = static_cast<int>(m_FontSize * m_ScaleY);

        // TODO: スキップするべき文字の表とか作る。
        // とりあえず 0x0d は普通にテキスト作ると入りがちなので。
        if (charCode == 0x0000000d)
        {
            return 0;
        }

        // INFO
        // フォントエンジンのキャッシュが時々おかしくなるので
        // 一文字毎に初期化しなおす
        // 速度が問われない場面なのでとりあえずこれで不具合を回避
        m_FontEngine.Initialize(m_pFontEngineBuffer, static_cast<uint32_t>(m_FontEngineBufferSize));
        NN_UTIL_SCOPE_EXIT{ m_FontEngine.Finalize(); };
        m_FontEngine.LoadFont(m_FontName, m_pFont, 0, sizeof(m_FontName));
        m_FontEngine.SetFont(m_FontName);

        float ascentRatio = 0;
        {
            nn::fontll::Metrics metrics;
            m_FontEngine.GetFontMetrics(&metrics);

            ascentRatio = static_cast<float>(metrics.os2WinAscent) / metrics.metricsResolution;
        }
        m_FontEngine.SetScale(fontSizeX << 16, 0, 0, fontSizeY << 16);

        auto map = m_FontEngine.AcquireGlyphmap(charCode, nn::fontll::FormatGrayMap8);
        if (map == NULL)
        {
            NN_DETAIL_FATAL_WARN("Warning: AcquireGlyphmap failed. char code = %08x\n", charCode);
            return 0;
        }

        int ascent = static_cast<int>(ascentRatio * fontSizeY + 0.0f * m_ScaleY);
        int offsetFromTop = std::max(ascent - map->hiY, 0);
        int renderOriginY = offsetFromTop;

        int dx = static_cast<int>(x * m_ScaleX);
        int dy = static_cast<int>(y * m_ScaleY);

        auto pBuffer = reinterpret_cast<PixelType*>(m_pDisplayBuffer);
        for (int i = 0; i < map->height; ++i)
        {
            for (int j = 0; j < map->idX + 1; ++j)
            {
                auto index = (i + dy + renderOriginY) * m_Stride + j + dx;
                Color color = {{ 0, 0, 0, 255 }};
                pBuffer[index] = GetPixelValue(color);
            }

            for (int j = 0; j < map->width; ++j)
            {
                auto v = map->bits[i * map->width + j];
                auto baseColor = m_Color;

                baseColor.bits.a = baseColor.bits.a = 255;
                baseColor.bits.g = baseColor.bits.g * v / 255;
                baseColor.bits.b = baseColor.bits.b * v / 255;
                baseColor.bits.r = baseColor.bits.r * v / 255;
                auto index = (i + dy + renderOriginY) * m_Stride + j + dx + map->loX;
                pBuffer[index] = GetPixelValue(baseColor);
            }
        }

        return static_cast<uint32_t>(map->idX / m_ScaleX + 1);
    }

    // -------------------------------

    namespace {
        SharedFont g_SharedFont = {};
        bool g_SharedFontInitialized = false;
    }

    void InitializeSharedFont() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!g_SharedFontInitialized);

        while (nn::pl::GetSharedFontLoadState(nn::pl::SharedFontType::SharedFontType_Standard) != nn::pl::SharedFontLoadState_Loaded)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        g_SharedFont.pointer = nn::pl::GetSharedFontAddress(nn::pl::SharedFontType::SharedFontType_Standard);
        g_SharedFont.size = nn::pl::GetSharedFontSize(nn::pl::SharedFontType::SharedFontType_Standard);
        g_SharedFontInitialized = true;
    }

    const SharedFont& GetSharedFont() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(g_SharedFontInitialized);

        return g_SharedFont;
    }
}} // namespace nn::fatalsrv
