﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include <nn/fan/fan_ControlDev.h> // For nn::fan::MaxRotationRate
#include <nn/fan/impl/fan_Api.h>
#include <nn/fan/impl/fan_Control.h>
#include <nn/fan/impl/fan_Lib.h>

#include "detail/fan_ControllerResourceManager.h"
#include "detail/fan_ControllerImpl.h"
#include "detail/fan_RotationSpeedLevel.h"
#include "detail/fan_SettingsHolder.h"

namespace nn {
namespace fan {
namespace impl {

namespace {

nn::fan::impl::detail::SettingsHolder g_SettingsHolder;

}

NN_STATIC_ASSERT(sizeof(detail::ControllerImplPadded) == detail::ControllerSize);
NN_STATIC_ASSERT(sizeof(detail::ControllerImpl) <= detail::ControllerSize);
NN_STATIC_ASSERT(std::alignment_of<detail::ControllerImplPadded>::value == detail::ControllerAlignment);

void Initialize() NN_NOEXCEPT
{
    g_SettingsHolder.LoadSettings();
}
void Finalize() NN_NOEXCEPT
{
    ;
}
void OpenController(Controller* pOutController, FanName name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutController);

    // Controller の構築
    detail::ControllerHolder* pControllerHolder = detail::AllocControllerHolder(name, g_SettingsHolder);
    NN_ABORT_UNLESS(pControllerHolder != nullptr, "FAN: Could not allocate control strategy holder.\n");

    // Construct as ControllerImpl
    new (&detail::ToControllerImpl(*pOutController)) detail::ControllerImpl(pControllerHolder, name);
}
void CloseController(Controller* pController) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);
    detail::ControllerHolder* pControllerHolder = controller.GetControllerHolder();

    // Destruct as ControllerImpl
    controller.~ControllerImpl();

    // Controller の破棄
    detail::FreeControllerHolder(pControllerHolder);
}
void SetRotationSpeedLevel(Controller* pController, RotationSpeedLevel level) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);
    NN_SDK_REQUIRES_MINMAX(level, 0, RotationSpeedLevelMax);

    controller.SetTargetRotationRate(detail::ToRotationRate(level));
}
RotationSpeedLevel SetRotationSpeedLevelByDiff(Controller* pController, int levelDiff) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);

    int currentLevel = static_cast<int>(detail::ToRotationSpeedLevel(controller.GetTargetRotationRate()));

    // 現在のレベルに差分を適用して新しい設定レベルを計算
    RotationSpeedLevel nextLevel;
    if (currentLevel + levelDiff > RotationSpeedLevelMax)
    {
        nextLevel = RotationSpeedLevelMax;
    }
    else if (currentLevel + levelDiff < 0)
    {
        nextLevel = 0;
    }
    else
    {
        nextLevel = static_cast<RotationSpeedLevel>(currentLevel + levelDiff);
    }

    // 計算したレベルの適用
    controller.SetTargetRotationRate(detail::ToRotationRate(nextLevel));

    return nextLevel;
}

RotationSpeedLevel GetRotationSpeedLevel(Controller* pController) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);

    return detail::ToRotationSpeedLevel(controller.GetTargetRotationRate());
}

RotationSpeedLevel GetActualRotationSpeedLevel(Controller* pController) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);

    return detail::ToRotationSpeedLevel(controller.GetCurrentRotationRate());
}

void SetRotationRate(Controller* pController, int rotationRate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);
    NN_SDK_REQUIRES_MINMAX(rotationRate, 0, nn::fan::MaxRotationRate);

    controller.SetTargetRotationRate(rotationRate);
}
int GetRotationRate(Controller* pController) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);

    return controller.GetTargetRotationRate();
}

void EnableFan(Controller* pController) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);

    controller.SetFanEnabled(true);
}

void DisableFan(Controller* pController) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pController);
    detail::ControllerImpl& controller = detail::ToControllerImpl(*pController);

    controller.SetFanEnabled(false);
}

void NotifyElapsedTime(int64_t elapsedTimeUsec) NN_NOEXCEPT
{
    detail::NotifyElapsedTime(elapsedTimeUsec);
}

void EnableFanFromServer() NN_NOEXCEPT
{
    detail::SetFanEnabledFromServer(true);
}

void DisableFanFromServer() NN_NOEXCEPT
{
    detail::SetFanEnabledFromServer(false);
}

} // impl
} // fan
} // nn
