﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/fan/detail/fan_Log.h>
#include <nn/fan/impl/fan_Lib.h>
#include <nn/fan/impl/fan_Control.h>
#include <nn/gpio/gpio.h>
#include <nn/pwm/pwm.h>

#include "fan_DeviceAccessorPwm-hardware.nx.h"

namespace nn {
namespace fan {
namespace impl {
namespace detail {

namespace {

const int TargetFrequency = 30 * 1000;  // 30kHz

struct DutyRateRange
{
    int lowerRateThreshold;
    int higherRateThreshold;
    int64_t usecPerRateUp;
    int64_t usecPerRateDown;
};

// RateRangeTable の編集に関して
// 1. 昇順に記述してください
// 2. 隣接する範囲は最低でも 1 つ共通の Rate 値を持たせてください
// 3. 初期状態は Table[0] の lowerRateThreshold になります
const DutyRateRange RateRangeTable[] =
{
    {  0,  51,   2500LL,   2500LL},
    { 51,  80,   2500LL, 250000LL},
    { 80, 110,  50000LL, 250000LL},
    {110, 255, 100000LL, 250000LL},
};

const int RotationStateNumber = sizeof(RateRangeTable) / sizeof(RateRangeTable[0]);

const int InitialState = 0;
const int InitialRate = RateRangeTable[InitialState].lowerRateThreshold;

const int MinimumDutyRate = 0;
const int MaximumDutyRate = 255;

int DutyFromRate(int rate) NN_NOEXCEPT
{
    int duty = MaximumDutyRate - rate; // NX Fan is on active low circuit; No rotation == 100% duty on PWM
    NN_ABORT_UNLESS(duty <= MaximumDutyRate);
    NN_ABORT_UNLESS(duty >= MinimumDutyRate);

    return duty;
}

} // namespace

DeviceAccessorPwm::RotationController::RotationController(FanName name) NN_NOEXCEPT
    : m_ActualRotationRate(InitialRate),
      m_RotationState(InitialState),
      m_FragmentedElapsedTimeUsec(0)
{
    NN_SDK_ASSERT(nn::fan::FanName_Cpu == name); // NX board has only one fan

    nn::pwm::Initialize();
    nn::pwm::OpenSession(&m_ChannelSession, nn::pwm::ChannelName_CpuFan);

    const nn::TimeSpan TargetPeriod = nn::TimeSpan::FromNanoSeconds((1 * 1000 * 1000 * 1000) / TargetFrequency);
    nn::pwm::SetPeriod(&m_ChannelSession, TargetPeriod);
    nn::pwm::SetDuty(&m_ChannelSession, DutyFromRate(InitialRate));
    nn::pwm::SetEnabled(&m_ChannelSession, true);
}

DeviceAccessorPwm::RotationController::~RotationController() NN_NOEXCEPT
{
    nn::pwm::CloseSession(&m_ChannelSession);
    nn::pwm::Finalize();
}

void DeviceAccessorPwm::RotationController::UpdateDutyRate() NN_NOEXCEPT
{
    nn::pwm::SetDuty(&m_ChannelSession, DutyFromRate(m_ActualRotationRate));
}

void DeviceAccessorPwm::RotationController::InitializeRotation() NN_NOEXCEPT
{
    m_FragmentedElapsedTimeUsec = 0;
    m_RotationState = InitialState;
    m_ActualRotationRate = InitialRate;

    // 開発者によるリストの変更を見込んで、m_ActualRotationRate と m_RotationState と対応関係に間違いが無いかチェックを入れる
    NN_ABORT_UNLESS(m_ActualRotationRate >= RateRangeTable[m_RotationState].lowerRateThreshold);
    NN_ABORT_UNLESS(m_ActualRotationRate <= RateRangeTable[m_RotationState].higherRateThreshold);

    UpdateDutyRate();
}

void DeviceAccessorPwm::RotationController::UpdateRotation(int targetRotationRate, int64_t elapsedTimeUsec) NN_NOEXCEPT
{
    int64_t duration = m_FragmentedElapsedTimeUsec + elapsedTimeUsec;

    while ( (m_ActualRotationRate != targetRotationRate) )
    {
        int threshold;
        int pm;
        int target;
        const DutyRateRange* pRange = &(RateRangeTable[m_RotationState]);

        int64_t usecPerRate = 100000LL;
        if ( m_ActualRotationRate > targetRotationRate )
        {
            usecPerRate = pRange->usecPerRateDown;
        }
        else
        {
            usecPerRate = pRange->usecPerRateUp;
        }

        if ( duration < usecPerRate )
        {
            break;
        }
        if ( (targetRotationRate - m_ActualRotationRate) < 0 )
        {
            pm = -1;
            threshold = pRange->lowerRateThreshold;
        }
        else
        {
            pm = 1;
            threshold = pRange->higherRateThreshold;
        }
        if ( std::abs(threshold - m_ActualRotationRate) < std::abs(targetRotationRate - m_ActualRotationRate) )
        {
            target = threshold;
        }
        else
        {
            target = targetRotationRate;
        }
        if ( (duration / usecPerRate) > static_cast<int64_t>(std::abs(target - m_ActualRotationRate)) )
        {
            m_ActualRotationRate = target;
            if ( target == threshold )
            {
                m_RotationState = std::min(std::max(m_RotationState + pm, 0), RotationStateNumber - 1);
            }
        }
        else
        {
            m_ActualRotationRate = m_ActualRotationRate + static_cast<int>(duration / usecPerRate) * pm;
            duration %= usecPerRate;
        }

        // 開発者によるリストの変更を見込んで、m_ActualRotationRate と m_RotationState と対応関係に間違いが無いかチェックを入れる
        NN_ABORT_UNLESS(m_ActualRotationRate >= RateRangeTable[m_RotationState].lowerRateThreshold);
        NN_ABORT_UNLESS(m_ActualRotationRate <= RateRangeTable[m_RotationState].higherRateThreshold);
    }

    if ( m_ActualRotationRate != targetRotationRate )
    {
        m_FragmentedElapsedTimeUsec = duration;
    }
    else
    {
        m_FragmentedElapsedTimeUsec = 0;
    }

    UpdateDutyRate();
}

int DeviceAccessorPwm::RotationController::GetCurrentRotationRate() const NN_NOEXCEPT
{
    return m_ActualRotationRate;
}

DeviceAccessorPwm::DeviceAccessorPwm(FanName name) NN_NOEXCEPT
    : m_TargetRotationRate(InitialRate),
      m_Enabled(true),
      m_Mutex(false),
      m_RotationController(name),
      m_EnabledFromServer(true)
{
    ;
}

int DeviceAccessorPwm::GetTargetRotationRate() const NN_NOEXCEPT
{
    return m_TargetRotationRate;
}

int DeviceAccessorPwm::GetCurrentRotationRate() const NN_NOEXCEPT
{
    return m_RotationController.GetCurrentRotationRate();
}

void DeviceAccessorPwm::SetTargetRotationRate(int rotationRate) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_TargetRotationRate = rotationRate;
}

void DeviceAccessorPwm::SetFanEnabled(bool enabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_Enabled = enabled;
    if ( !m_Enabled )
    {
        m_RotationController.InitializeRotation();
    }
}

void DeviceAccessorPwm::NotifyElapsedTime(int64_t elapsedTimeUsec) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if ( m_Enabled && m_EnabledFromServer )
    {
        m_RotationController.UpdateRotation(m_TargetRotationRate, elapsedTimeUsec);
    }
}

void DeviceAccessorPwm::SetFanEnabledFromServer(bool enabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_EnabledFromServer = enabled;
    if ( !m_EnabledFromServer )
    {
        m_RotationController.InitializeRotation();
    }
}

}}}} // nn::fan::impl::detail
