﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs.h>
#include <nn/time.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/eupld/eupld_Result.h>
#include <nn/eupld/sfdl/eupld.sfdl.h>
#include <nn/eupld/server/eupld_Server.h>

#include "eupldsrv_IRequest.h"
#include "eupldsrv_IControl.h"

namespace nn    {
namespace eupld {
namespace srv   {

extern nn::sf::ExpHeapAllocator g_ServiceFrameworkAllocator;

namespace      {

const int SessionThreadStackSize = (16 * 1024);
const int RequestSessionCount    = 5;
const int ControlSessionCount    = 2;
const int SessionMaxCount        = RequestSessionCount + ControlSessionCount;

enum ipcSessionPort
{
    sessionPort_Request,
    sessionPort_Control,
    sessionPort_Highest
};

class Service
    : public nn::sf::HipcSimpleAllInOneServerManager<SessionMaxCount, sessionPort_Highest>
{
private:
    nn::os::ThreadType m_SessionThread;
    uint8_t            m_SessionThreadStack[SessionThreadStackSize] NN_ALIGNAS(os::StackRegionAlignment);

    static void SessionThreadEntry(void* arg)
    NN_NOEXCEPT
    {
        Service* p = reinterpret_cast<Service*>(arg);
        p->LoopAuto();
    }

    nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort)
    NN_NOEXCEPT
    {
        if (portIndex == sessionPort_Request)
        {
            auto interface = nn::sf::ObjectFactory<
                     nn::sf::ExpHeapAllocator::Policy
                    >::CreateSharedEmplaced<
                     nn::eupld::sf::IRequest,
                     nn::eupld::srv::IRequest
                    >(&g_ServiceFrameworkAllocator);

            return interface != nullptr ?
                     this->AcceptImpl(pPort, interface) :
                     nn::eupld::ResultOutOfMemory();
        }
        else if (portIndex == sessionPort_Control)
        {
            auto interface = nn::sf::ObjectFactory<
                     nn::sf::ExpHeapAllocator::Policy
                    >::CreateSharedEmplaced<
                     nn::eupld::sf::IControl,
                     nn::eupld::srv::IControl
                    >(&g_ServiceFrameworkAllocator);

            return interface != nullptr ?
                     this->AcceptImpl(pPort, interface) :
                     nn::eupld::ResultOutOfMemory();
        }

        return nn::eupld::ResultInvalidArgument();
    }

public:
    void Wait()
    NN_NOEXCEPT
    {
        nn::os::WaitThread(&m_SessionThread);
    }

    nn::Result Initialize()
    NN_NOEXCEPT
    {
        nn::Result result;

        result = this->InitializePort(
                    sessionPort_Request,
                    RequestSessionCount,
                    RequestPortName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = this->InitializePort(
                    sessionPort_Control,
                    ControlSessionCount,
                    ControlPortName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        this->Start();

        result = nn::os::CreateThread(
                    &m_SessionThread,
                    SessionThreadEntry,
                    this,
                    m_SessionThreadStack,
                    SessionThreadStackSize,
                    NN_SYSTEM_THREAD_PRIORITY(eupld, IpcServer) );

        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::SetThreadNamePointer(&m_SessionThread, NN_SYSTEM_THREAD_NAME(eupld, IpcServer));
        nn::os::StartThread(&m_SessionThread);

        return ResultSuccess();
    }

} g_Service;

} // namespace

nn::Result InitializeService()
NN_NOEXCEPT
{
    return g_Service.Initialize();
}

void WaitService()
{
    g_Service.Wait();
}

}}}
