﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopedTransaction.h>

#include <nn/http/json/http_JsonPath.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_RapidJsonInputStream.h>
#include "./../es_ELicenseArchiveReader.h"

namespace nn { namespace es { namespace json {

// ImportJsonByRapidJson で使用する InputStreamType の定義
class MemoriedStreamForRapidJson : public nn::http::json::StringBufferForRapidJson
{
private:
    typedef nn::http::json::StringBufferForRapidJson Base;

public:
    typedef Base::Ch Ch;

public:
    MemoriedStreamForRapidJson(const Ch* json, size_t size) NN_NOEXCEPT
        : m_Json(json)
        , m_Size(size)
        , m_Position(0u)
    {
        NN_SDK_REQUIRES(json && size > 0u);
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(json) % NN_ALIGNOF(Ch) == 0);
        NN_SDK_ASSERT(size % sizeof(Ch) == 0);
    }
    ~MemoriedStreamForRapidJson() NN_NOEXCEPT
    {
    }

    // 以降は InputStreamType で定義が必要なメソッド
    nn::Result GetResult() const NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }
    inline Ch Peek() NN_NOEXCEPT
    {
        return (m_Position < m_Size) ? m_Json[m_Position] : '\0';
    }
    inline Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        ++m_Position;
        return c;
    }
    inline size_t Tell() const NN_NOEXCEPT
    {
        return m_Position;
    }

private:
    const Ch* m_Json;
    size_t m_Size;
    size_t m_Position;
};

template<int ReadBufferSize>
class FileStreamForRapidJson : public nn::http::json::StringBufferForRapidJson
{
private:
    typedef nn::http::json::StringBufferForRapidJson Base;

public:
    typedef Base::Ch Ch;

    explicit FileStreamForRapidJson() NN_NOEXCEPT
        : m_ReadOffset(0)
        , m_Size(0)
        , m_Position(0)
        , m_LastResult(ResultSuccess())
    {
    };

    nn::Result Initialize(const Ch* filePath) NN_NOEXCEPT
    {
        m_ReadOffset = 0;
        m_Size = 0;
        m_Position = 0;
        m_LastResult = ResultSuccess();

        NN_RESULT_DO(nn::fs::OpenFile(&m_Handle, filePath, nn::fs::OpenMode_Read));

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, m_Handle));
        m_Size = static_cast<size_t>(fileSize);

        NN_RESULT_DO(ReadFileToBuffer(0));

        NN_RESULT_SUCCESS;
    }

    nn::Result Finalize() NN_NOEXCEPT
    {
        nn::fs::CloseFile(m_Handle);

        NN_RESULT_SUCCESS;
    }

    // 以降は InputStreamType で定義が必要なメソッド
    nn::Result GetResult() const NN_NOEXCEPT
    {
        return m_LastResult;
    }
    inline Ch Peek() NN_NOEXCEPT
    {
        if (m_Position >= m_Size)
        {
            return '\0';
        }

        // ファイルをバッファに読み込みが必要な場合
        if (m_Position >= m_ReadOffset + sizeof(m_ReadBuffer))
        {
            m_LastResult = ReadFileToBuffer(m_ReadOffset + sizeof(m_ReadBuffer));
            m_ReadOffset = m_Position;
        }

        return m_ReadBuffer[m_Position - m_ReadOffset];
    }
    inline Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        ++m_Position;
        return c;
    }
    inline size_t Tell() const NN_NOEXCEPT
    {
        return m_Position;
    }

private:
    nn::Result ReadFileToBuffer(size_t fileOffset) NN_NOEXCEPT
    {
        size_t readFileSize = std::min(m_Size - fileOffset, sizeof(m_ReadBuffer));
        NN_RESULT_DO(nn::fs::ReadFile(m_Handle, fileOffset, m_ReadBuffer, readFileSize));

        NN_RESULT_SUCCESS;
    }

    const Ch* m_FilePath;
    nn::fs::FileHandle m_Handle;
    Ch m_ReadBuffer[ReadBufferSize];
    size_t m_ReadOffset;

    size_t m_Size;
    size_t m_Position;

    nn::Result m_LastResult;
};

class ELicenseArchiveFileStreamForRapidJson : public nn::http::json::StringBufferForRapidJson
{
private:
    typedef nn::http::json::StringBufferForRapidJson Base;

public:
    typedef Base::Ch Ch;

    explicit ELicenseArchiveFileStreamForRapidJson() NN_NOEXCEPT
        : m_Size(0)
        , m_Position(0)
        , m_LastResult(ResultSuccess())
    {
    };

    nn::Result Initialize(const Ch* filePath) NN_NOEXCEPT
    {
        m_Size = 0;
        m_Position = 0;
        m_LastResult = ResultSuccess();

        NN_RESULT_DO(nn::fs::OpenFile(&m_Handle, filePath, nn::fs::OpenMode_Read));

        // 全て処理が完了する前に失敗した場合、状態を巻き戻す
        util::ScopedTransaction transaction;
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            nn::fs::CloseFile(m_Handle);
        };

        NN_RESULT_DO(m_Reader.Initailize(m_Handle));
        m_Size = static_cast<size_t>(m_Reader.GetSize());

        transaction.Commit();

        NN_RESULT_SUCCESS;
    }

    nn::Result Finalize() NN_NOEXCEPT
    {
        nn::fs::CloseFile(m_Handle);

        NN_RESULT_SUCCESS;
    }

    // 以降は InputStreamType で定義が必要なメソッド
    nn::Result GetResult() const NN_NOEXCEPT
    {
        return m_LastResult;
    }
    inline Ch Peek() NN_NOEXCEPT
    {
        if (m_Position >= m_Size)
        {
            return '\0';
        }

        Ch c;
        m_LastResult = m_Reader.Read(m_Position, &c, 1);

        return c;
    }
    inline Ch Take() NN_NOEXCEPT
    {
        auto c = Peek();
        ++m_Position;
        return c;
    }
    inline size_t Tell() const NN_NOEXCEPT
    {
        return m_Position;
    }

private:
    nn::fs::FileHandle m_Handle;
    ELicenseArchiveFileReader m_Reader;

    size_t m_Size;
    size_t m_Position;

    nn::Result m_LastResult;
};

// ImportJsonByRapidJson で使用する CancellableType の定義
struct Cancellable
{
    bool IsCancelled() const { return false; }
};

template <typename JsonAdaptorType, typename InputStreamType>
nn::Result ImportJsonByRapidJson(JsonAdaptorType& adaptor, InputStreamType& stream) NN_NOEXCEPT
{
    Cancellable cancellable;
    return nn::http::json::ImportJsonByRapidJson<nn::http::json::DefaultJsonErrorMap>(adaptor, stream, &cancellable);
}

}}}
