﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/es/es_MountName.h>
#include <nn/fs/fs_Mount.h>
#include "es_CreateETicketService.h"
#include "es_ETicketServiceObject.h"

namespace nn { namespace es {
namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
sf::SharedPointer<IETicketService> g_ETicketService;

#ifdef NN_BUILD_CONFIG_OS_WIN
nn::Result MountSystemSaveData(const char* mountName, nn::fs::SystemSaveDataId id)
{
    const size_t SaveDataSize = 16 * 1024 * 1024;
    const size_t SaveDataJournalSize = 16 * 1024 * 1024;

    NN_RESULT_TRY(nn::fs::MountSystemSaveData(mountName, id))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, SaveDataSize, SaveDataJournalSize, 0));
            NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, id));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}
#endif // NN_BUILD_CONFIG_OS_WIN

}

void Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_ETicketService);

#ifdef NN_BUILD_CONFIG_OS_WIN
    // eTicketDataBase のマウント
    MountSystemSaveData(nn::es::CertificateDbMountName, nn::es::CertificateDbSystemSaveDataId);
    MountSystemSaveData(nn::es::CommonTicketDbMountName, nn::es::CommonTicketDbSystemSaveDataId);
    MountSystemSaveData(nn::es::PersonalizedTicketDbMountName, nn::es::PersonalizedTicketDbSystemSaveDataId);
    MountSystemSaveData(nn::es::TicketMetaRecordDbMountName, nn::es::TicketMetaRecordDbSystemSaveDataId);
    MountSystemSaveData(nn::es::ELicenseArchiveStoreMountName, nn::es::ELicenseArchiveStoreSystemSaveDataId);

    g_ETicketService = CreateETicketServiceByDfc();
#else
    g_ETicketService = CreateETicketServiceByHipc();
#endif // NN_BUILD_CONFIG_OS_WIN
}

void Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_ETicketService);
    // 共有ポインタへの nullptr の代入で解放できる。
    g_ETicketService = nullptr;

#ifdef NN_BUILD_CONFIG_OS_WIN
    nn::fs::Unmount(nn::es::CertificateDbMountName);
    nn::fs::Unmount(nn::es::CommonTicketDbMountName);
    nn::fs::Unmount(nn::es::PersonalizedTicketDbMountName);
    nn::fs::Unmount(nn::es::TicketMetaRecordDbMountName);
#endif // NN_BUILD_CONFIG_OS_WIN
}

sf::SharedPointer<IETicketService>& GetETicketServiceObject() NN_NOEXCEPT
{
    return g_ETicketService;
}
}}
