﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/util/util_FormatString.h>
#include "es_ELicenseArchiveStore.h"

namespace nn { namespace es {

namespace
{
    bool CheckFileExitstence(const char* path) NN_NOEXCEPT
    {
        fs::DirectoryEntryType type;
        Result result = fs::GetEntryType(&type, path);

        if (result.IsSuccess())
        {
            if (type == fs::DirectoryEntryType_File)
            {
                return true;
            }
            else
            {
                return false;
            }
        }
        else if (fs::ResultPathNotFound::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT();
        }
    }

    Result DeleteDirectoryRecursivelyIfExist(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_TRY(fs::DeleteDirectoryRecursively(path))
            NN_RESULT_CATCH(fs::ResultPathNotFound) {}
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result CreateDirectoryIfNotExist(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_TRY(fs::CreateDirectory(path))
            NN_RESULT_CATCH(fs::ResultPathAlreadyExists) {}
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result DeleteFileIfExist(const char* path) NN_NOEXCEPT
    {
        NN_RESULT_TRY(fs::DeleteFile(path))
            NN_RESULT_CATCH(fs::ResultPathNotFound) {}
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }
}

void ELicenseArchiveStore::Initialize() NN_NOEXCEPT
{
    ETicketServiceDatabase::Initialize();

    bool onlyConstructing = true;
    DeleteAllImpl(onlyConstructing);
}

Result ELicenseArchiveStore::Import(const void* archive, size_t size, ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    bool isAppendWrite = false;
    fs::FileHandle fileHandle;
    {
        Path path = GetArchiveFilePath(ArchiveState::Constructing, ownerId);

        NN_RESULT_TRY(fs::CreateFile(path, size))
            NN_RESULT_CATCH(fs::ResultPathAlreadyExists) { isAppendWrite = true; }
        NN_RESULT_END_TRY
        NN_RESULT_DO(fs::OpenFile(&fileHandle, path.Get(), fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    }
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    int64_t fileSize;
    NN_RESULT_DO(fs::GetFileSize(&fileSize, fileHandle));

    int64_t writeOffset = isAppendWrite ? fileSize : 0;
    NN_RESULT_DO(fs::WriteFile(fileHandle, writeOffset, archive, size, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveStore::EndImporting(ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    // Saved に保存されているアーカイブを削除する
    DeleteImpl(ELicenseArchiveStore::ArchiveState::Saved, ownerId);

    // Constructing に保存されているアーカイブを Saved に移す
    Path constructingFilePath = GetArchiveFilePath(ArchiveState::Constructing, ownerId);
    Path savedFilePath = GetArchiveFilePath(ArchiveState::Saved, ownerId);

    NN_RESULT_DO(fs::RenameFile(constructingFilePath.Get(), savedFilePath.Get()));

    NN_RESULT_SUCCESS;
}

void ELicenseArchiveStore::Delete(ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    DeleteImpl(ArchiveState::Constructing, ownerId);
    DeleteImpl(ArchiveState::Saved, ownerId);
}

void ELicenseArchiveStore::DeleteAll() NN_NOEXCEPT
{
    bool onlyConstructing = false;
    DeleteAllImpl(onlyConstructing);
}

int ELicenseArchiveStore::List(ELicenseOwnerId* outELicenseOwnerIdList, int listCount) NN_NOEXCEPT
{
    Path archiveDirectoryPath = GetArchiveDirectoryPath(ArchiveState::Saved);

    fs::DirectoryHandle handle;
    NN_RESULT_DO(fs::OpenDirectory(&handle, archiveDirectoryPath.Get(), fs::OpenDirectoryMode_File));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseDirectory(handle);
    };

    int64_t fileCount = 0;
    int64_t entryCount;
    fs::DirectoryEntry entry;

    while (NN_STATIC_CONDITION(true))
    {
        NN_RESULT_DO(fs::ReadDirectory(&entryCount, &entry, handle, 1));

        if (entryCount == 0 || (listCount - fileCount) <= 0)
        {
            break;
        }

        outELicenseOwnerIdList[fileCount] = { std::strtoull(entry.name, nullptr, 16) };
        fileCount++;
    }

    return static_cast<int>(fileCount);
}

bool ELicenseArchiveStore::GetArchivePathForParse(Path* outPath, ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    // eLicense アーカイブが存在するか確認する
    Path archivePath = GetArchiveFilePath(ArchiveState::Saved, ownerId);
    if (CheckFileExitstence(archivePath.Get()))
    {
        *outPath = archivePath;
        return true;
    }

    return false;
}

void ELicenseArchiveStore::DeleteImpl(ArchiveState state, ELicenseOwnerId ownerId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteFileIfExist(GetArchiveFilePath(state, ownerId).Get()));
}

void ELicenseArchiveStore::DeleteAllImpl(bool onlyConstructing) NN_NOEXCEPT
{
    // インポート中のアーカイブを保存ディレクトリごと削除する
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteDirectoryRecursivelyIfExist(GetArchiveDirectoryPath(ArchiveState::Constructing).Get()));

    if (!onlyConstructing)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteDirectoryRecursivelyIfExist(GetArchiveDirectoryPath(ArchiveState::Saved).Get()));
    }

    // アーカイブの保存ディレクトリを作成する
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateDirectoryIfNotExist(GetArchiveDirectoryPath(ArchiveState::Constructing).Get()));
    NN_ABORT_UNLESS_RESULT_SUCCESS(CreateDirectoryIfNotExist(GetArchiveDirectoryPath(ArchiveState::Saved).Get()));
}

}} // namespace nn::es
