﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/es.h>
#include <nn/es/debug/es_ELicenseArchiveJsonUtil.h>
#include <nn/os/os_Random.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_Span.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_UuidApi.h>
#include "..\es_Json.h"

namespace nn { namespace es { namespace debug {

namespace
{
    struct ELicenseIdString
    {
        char value[es::ELicenseId::StringSize + 1];
    };

    struct ELicenseArchiveIdString
    {
        char value[es::ELicenseArchiveId::StringSize + 1];
    };

    struct ChallengeString
    {
        char value[sizeof(uint64_t) * 2 + 1];
    };

    struct NintendoAccountIdString
    {
        char value[sizeof(account::NintendoAccountId) * 2 + 1];
    };

    struct RightsIdString
    {
        char value[sizeof(RightsId) * 2 + 1];
    };

    static nn::TimeSpan DefaultExpireSpan = nn::TimeSpan::FromHours(3);

    ELicenseArchiveInfoForDebug MakeELicenseArchiveInfoCommon(uint64_t challenge, account::NintendoAccountId ownerNaId, util::optional<ELicenseArchiveId> archiveId) NN_NOEXCEPT
    {
        ELicenseArchiveInfoForDebug info = {};

        info.challenge = challenge;
        info.ownerNaId = ownerNaId;

        if (archiveId)
        {
            info.archiveId = *archiveId;
        }
        else
        {
            util::Uuid archiveUuid = util::GenerateUuid();
            memcpy(&info.archiveId, &archiveUuid, sizeof(info.archiveId));
        }

        time::PosixTime time;
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardNetworkSystemClock::GetCurrentTime(&time));
        info.publishDate = time;

        return info;
    }

    ELicenseInfoForDebug MakeELicenseCommon(RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, TicketId ticketId, AccountId ticketOwnerVaId, util::optional<ELicenseId> eLicenseId) NN_NOEXCEPT
    {
        ELicenseInfoForDebug license = {};

        license.rightsId = rightsId;
        license.ticketOwnerNaId = ticketOwnerNaId;
        license.ticketId = ticketId;
        license.ticketOwnerVaId = ticketOwnerVaId;

        if (eLicenseId)
        {
            license.id = *eLicenseId;
        }
        else
        {
            util::Uuid uuid = util::GenerateUuid();
            memcpy(&license.id, &uuid, sizeof(license.id));
        }

        return license;
    }

    ELicenseInfoForDebug MakeDeviceLinkedPermanentELicenseCommon(RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, TicketId ticketId, AccountId ticketOwnerVaId, util::optional<ELicenseId> eLicenseId) NN_NOEXCEPT
    {
        ELicenseInfoForDebug license = MakeELicenseCommon(rightsId, ticketOwnerNaId, ticketId, ticketOwnerVaId, eLicenseId);

        license.expireDate.time.value = std::numeric_limits<int64_t>::max();             // 期限なしを示す値を入れる
        license.scope = ELicenseScope::Device;
        license.availableAfterReboot = true;
        license.recommendsServerInteraction = false;

        return license;
    }

    ELicenseInfoForDebug MakeAccountRestrictiveTemporaryELicenseCommon(RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, TicketId ticketId, AccountId ticketOwnerVaId, util::optional<ELicenseId> eLicenseId, const TimeSpan& expireSpan) NN_NOEXCEPT
    {
        ELicenseInfoForDebug license = MakeELicenseCommon(rightsId, ticketOwnerNaId, ticketId, ticketOwnerVaId, eLicenseId);

        time::PosixTime time;
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardNetworkSystemClock::GetCurrentTime(&time));
        license.expireDate.time = time + expireSpan;
        license.scope = ELicenseScope::Account;
        license.availableAfterReboot = false;
        license.recommendsServerInteraction = true;

        return license;
    }

    bool GetTicketInfoFormTicketDB(TicketId* outTicketId, AccountId* outAccountId, RightsId rightsId) NN_NOEXCEPT
    {
        RightsIdIncludingKeyId rightsIdIncludingKeyIdList[16];
        int rightsIdCount = ListTicketRightsIds(rightsIdIncludingKeyIdList, NN_ARRAY_SIZE(rightsIdIncludingKeyIdList), rightsId);

        for (auto&& rightsIdIncludingKeyId : util::MakeSpan(rightsIdIncludingKeyIdList, rightsIdCount))
        {
            LightTicketInfo info[16];
            int infoCount = ListLightTicketInfo(info, NN_ARRAY_SIZE(info), rightsIdIncludingKeyId);

            auto result = std::find_if(info, info + infoCount, [](const LightTicketInfo& info) { return info.accountId != 0; });

            // 最初に見つかった情報を返却する
            if (result != &info[infoCount])
            {
                *outTicketId = result->ticketId;
                *outAccountId = result->accountId;
                return true;
            }
        }

        return false;
    }
}

ELicenseArchiveInfoForDebug MakeELicenseArchiveInfo(uint64_t challenge, account::NintendoAccountId ownerNaId) NN_NOEXCEPT
{
    return MakeELicenseArchiveInfoCommon(challenge, ownerNaId, util::nullopt);
}

ELicenseArchiveInfoForDebug MakeELicenseArchiveInfo(uint64_t challenge, account::NintendoAccountId ownerNaId, ELicenseArchiveId archiveId) NN_NOEXCEPT
{
    return MakeELicenseArchiveInfoCommon(challenge, ownerNaId, archiveId);
}

ELicenseInfoForDebug MakeDeviceLinkedPermanentELicense(RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, TicketId ticketId, AccountId ticketOwnerVaId) NN_NOEXCEPT
{
    return MakeDeviceLinkedPermanentELicenseCommon(rightsId, ticketOwnerNaId, ticketId, ticketOwnerVaId, util::nullopt);
}

ELicenseInfoForDebug MakeDeviceLinkedPermanentELicense(RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, TicketId ticketId, AccountId ticketOwnerVaId, ELicenseId eLicenseId) NN_NOEXCEPT
{
    return MakeDeviceLinkedPermanentELicenseCommon(rightsId, ticketOwnerNaId, ticketId, ticketOwnerVaId, eLicenseId);
}

bool MakeDeviceLinkedPermanentELicenseFromTicketDBInfo(ELicenseInfoForDebug* outValue, RightsId rightsId, account::NintendoAccountId ticketOwnerNaId) NN_NOEXCEPT
{
    TicketId ticketId;
    AccountId accountId;

    if (!GetTicketInfoFormTicketDB(&ticketId, &accountId, rightsId))
    {
        return false;
    }

    *outValue = MakeDeviceLinkedPermanentELicenseCommon(rightsId, ticketOwnerNaId, ticketId, accountId, util::nullopt);
    return true;
}

bool MakeDeviceLinkedPermanentELicenseFromTicketDBInfo(ELicenseInfoForDebug* outValue, RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, ELicenseId eLicenseId) NN_NOEXCEPT
{
    TicketId ticketId;
    AccountId accountId;

    if (!GetTicketInfoFormTicketDB(&ticketId, &accountId, rightsId))
    {
        return false;
    }

    *outValue = MakeDeviceLinkedPermanentELicenseCommon(rightsId, ticketOwnerNaId, ticketId, accountId, eLicenseId);
    return true;
}

ELicenseInfoForDebug MakeAccountRestrictiveTemporaryELicense(RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, TicketId ticketId, AccountId ticketOwnerVaId) NN_NOEXCEPT
{
    return MakeAccountRestrictiveTemporaryELicenseCommon(rightsId, ticketOwnerNaId, ticketId, ticketOwnerVaId, util::nullopt, DefaultExpireSpan);
}

ELicenseInfoForDebug MakeAccountRestrictiveTemporaryELicense(RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, TicketId ticketId, AccountId ticketOwnerVaId, ELicenseId eLicenseId) NN_NOEXCEPT
{
    return MakeAccountRestrictiveTemporaryELicenseCommon(rightsId, ticketOwnerNaId, ticketId, ticketOwnerVaId, eLicenseId, DefaultExpireSpan);
}

bool MakeAccountRestrictiveTemporaryELicenseFromTicketDBInfo(ELicenseInfoForDebug* outValue, RightsId rightsId, account::NintendoAccountId ticketOwnerNaId) NN_NOEXCEPT
{
    TicketId ticketId;
    AccountId accountId;

    if (!GetTicketInfoFormTicketDB(&ticketId, &accountId, rightsId))
    {
        return false;
    }

    *outValue = MakeAccountRestrictiveTemporaryELicenseCommon(rightsId, ticketOwnerNaId, ticketId, accountId, util::nullopt, DefaultExpireSpan);
    return true;
}

bool MakeAccountRestrictiveTemporaryELicenseFromTicketDBInfo(ELicenseInfoForDebug* outValue, RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, ELicenseId eLicenseId) NN_NOEXCEPT
{
    TicketId ticketId;
    AccountId accountId;

    if (!GetTicketInfoFormTicketDB(&ticketId, &accountId, rightsId))
    {
        return false;
    }

    *outValue = MakeAccountRestrictiveTemporaryELicenseCommon(rightsId, ticketOwnerNaId, ticketId, accountId, eLicenseId, DefaultExpireSpan);
    return true;
}

bool MakeAccountRestrictiveTemporaryELicenseFromTicketDBInfo(ELicenseInfoForDebug* outValue, RightsId rightsId, account::NintendoAccountId ticketOwnerNaId, TimeSpan expireSpan) NN_NOEXCEPT
{
    TicketId ticketId;
    AccountId accountId;

    if (!GetTicketInfoFormTicketDB(&ticketId, &accountId, rightsId))
    {
        return false;
    }

    *outValue = MakeAccountRestrictiveTemporaryELicenseCommon(rightsId, ticketOwnerNaId, ticketId, accountId, util::nullopt, expireSpan);
    return true;
}

size_t BuildELicenseArchiveJsonString(char* outBuffer, size_t bufferSize, const ELicenseArchiveInfoForDebug& info, const ELicenseInfoForDebug* pList, int listCount) NN_NOEXCEPT
{
    nne::rapidjson::Document document;
    document.SetObject();

    ChallengeString challengeString;
    ELicenseArchiveIdString eLicenseArchiveIdString;
    NintendoAccountIdString eLicenseOwnerNaIdString;

    std::unique_ptr<ELicenseIdString[]> eLicenseIdStringList(new ELicenseIdString[listCount]);
    std::unique_ptr<RightsIdString[]> rightsIdStringList(new RightsIdString[listCount]);
    std::unique_ptr<NintendoAccountIdString[]> ticketOwnerNaIdStringList(new NintendoAccountIdString[listCount]);

    util::SNPrintf(challengeString.value, sizeof(challengeString.value), "%016llx", info.challenge);
    document.AddMember("challenge", nne::rapidjson::StringRef(challengeString.value), document.GetAllocator());
    document.AddMember("elicense_archive_id", nne::rapidjson::StringRef(info.archiveId.ToString(eLicenseArchiveIdString.value, sizeof(eLicenseArchiveIdString.value))), document.GetAllocator());
    document.AddMember("publish_date", nne::rapidjson::Value(info.publishDate.value), document.GetAllocator());
    util::SNPrintf(eLicenseOwnerNaIdString.value, sizeof(eLicenseOwnerNaIdString.value), "%016llx", info.ownerNaId.id);
    document.AddMember("elicense_owner_na_id", nne::rapidjson::StringRef(eLicenseOwnerNaIdString.value), document.GetAllocator());

    if (listCount > 0)
    {
        nne::rapidjson::Value eLicenses(nne::rapidjson::kArrayType);
        for (int i = 0; i < listCount; i++)
        {
            const auto& entry = pList[i];

            nne::rapidjson::Value eLicense(nne::rapidjson::kObjectType);
            eLicense.SetObject();

            eLicense.AddMember("elicense_id", nne::rapidjson::StringRef(entry.id.ToString(eLicenseIdStringList[i].value, sizeof(eLicenseIdStringList[i].value))), document.GetAllocator());
            util::SNPrintf(rightsIdStringList[i].value, sizeof(rightsIdStringList[i]), "%016llx", entry.rightsId);
            eLicense.AddMember("rights_id", nne::rapidjson::StringRef(rightsIdStringList[i].value), document.GetAllocator());
            eLicense.AddMember("ticket_id", nne::rapidjson::Value(entry.ticketId), document.GetAllocator());
            eLicense.AddMember("ticket_owner_va_id", nne::rapidjson::Value(entry.ticketOwnerVaId), document.GetAllocator());
            util::SNPrintf(ticketOwnerNaIdStringList[i].value, sizeof(ticketOwnerNaIdStringList[i].value), "%016llx", entry.ticketOwnerNaId.id);
            eLicense.AddMember("ticket_owner_na_id", nne::rapidjson::StringRef(ticketOwnerNaIdStringList[i].value), document.GetAllocator());

            if (!entry.expireDate.IsPermanent())
            {
                eLicense.AddMember("expire_date", nne::rapidjson::Value(entry.expireDate.time.value), document.GetAllocator());
            }

            switch (entry.scope)
            {
            case ELicenseScope::Account:
                eLicense.AddMember("scope", nne::rapidjson::Value("account"), document.GetAllocator());
                break;
            case ELicenseScope::Device:
                eLicense.AddMember("scope", nne::rapidjson::Value("device"), document.GetAllocator());
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

            eLicense.AddMember("available_after_reboot", nne::rapidjson::Value(entry.availableAfterReboot), document.GetAllocator());
            eLicense.AddMember("recommends_server_interaction", nne::rapidjson::Value(entry.recommendsServerInteraction), document.GetAllocator());

            eLicenses.PushBack(eLicense, document.GetAllocator());
        }
        document.AddMember("elicenses", eLicenses, document.GetAllocator());
    }

    nne::rapidjson::StringBuffer json;
    nne::rapidjson::Writer<nne::rapidjson::StringBuffer> writer(json);
    document.Accept(writer);

    if (bufferSize < json.GetSize())
    {
        return 0;
    }

    nn::util::Strlcpy(outBuffer, json.GetString(), static_cast<int>(bufferSize));

    return json.GetSize();
}

void ELicenseArchiveJsonReader::Initialize(const char* string) NN_NOEXCEPT
{
    m_Document.Parse(string);
    NN_ABORT_UNLESS(!m_Document.HasParseError(), "JSON Parser returned ParseErrorCode: %d at %d.", m_Document.GetParseError(), m_Document.GetErrorOffset());
}

ELicenseArchiveInfoForDebug ELicenseArchiveJsonReader::GetELicenseArchiveInfo() const NN_NOEXCEPT
{
    ELicenseArchiveInfoForDebug info = {};

    auto challengeObj = m_Document.FindMember("challenge");
    NN_ABORT_UNLESS(challengeObj != m_Document.MemberEnd() && challengeObj->value.IsUint64(), "\"challenge\" is invalid.");
    info.challenge = challengeObj->value.GetUint64();

    auto archiveIdObj = m_Document.FindMember("archive_id");
    NN_ABORT_UNLESS(archiveIdObj != m_Document.MemberEnd() && archiveIdObj->value.IsUint64(), "\"archive_id\" is invalid.");
    NN_ABORT_UNLESS(info.archiveId.FromString(archiveIdObj->value.GetString()), "\"archive_id\" is invalid.");

    auto publishDateObj = m_Document.FindMember("publish_date");
    NN_ABORT_UNLESS(publishDateObj != m_Document.MemberEnd() && publishDateObj->value.IsInt64(), "\"publish_date\" is invalid.");
    info.publishDate.value = publishDateObj->value.GetInt64();

    auto elicenseOwnerNaIdObj = m_Document.FindMember("elicense_owner_na_id");
    NN_ABORT_UNLESS(elicenseOwnerNaIdObj != m_Document.MemberEnd() && elicenseOwnerNaIdObj->value.IsUint64(), "\"elicense_owner_na_id\" is invalid.");
    info.ownerNaId.id = elicenseOwnerNaIdObj->value.GetUint64();

    return info;
}

int ELicenseArchiveJsonReader::CountELicense() const NN_NOEXCEPT
{
    auto eLicensesObj = m_Document.FindMember("eLicenses");
    if (eLicensesObj == m_Document.MemberEnd())
    {
        return 0;
    }

    NN_ABORT_UNLESS(eLicensesObj->value.IsArray(), "\"eLicenses\" is invalid.");

    const nne::rapidjson::Value& eLicensesValue = m_Document["eLicenses"];

    return static_cast<int>(eLicensesValue.Size());
}

ELicenseInfoForDebug ELicenseArchiveJsonReader::GetELicense(int index) const NN_NOEXCEPT
{
    ELicenseInfoForDebug eLicense = {};

    auto eLicensesObj = m_Document.FindMember("eLicenses");
    NN_ABORT_UNLESS(eLicensesObj != m_Document.MemberEnd() && eLicensesObj->value.IsArray(), "\"eLicenses\" is invalid.");

    const nne::rapidjson::Value& eLicensesValue = m_Document["eLicenses"];

    const auto& eLicenseValue = eLicensesValue[index];

    NN_ABORT_UNLESS(eLicenseValue.IsObject(), "\"eLicenses[%d]\" is invalid.", index);

    auto elicenseIdObj = eLicenseValue.FindMember("elicense_id");
    NN_ABORT_UNLESS(elicenseIdObj != eLicenseValue.MemberEnd() && elicenseIdObj->value.IsUint64(), "\"eLicenses[%d].elicense_id\" is invalid.", index);
    NN_ABORT_UNLESS(eLicense.id.FromString(elicenseIdObj->value.GetString()), "\"eLicenses[%d].elicense_id\" is invalid.", index);

    auto rightsIdObj = eLicenseValue.FindMember("rights_id");
    NN_ABORT_UNLESS(rightsIdObj != eLicenseValue.MemberEnd() && rightsIdObj->value.IsUint64(), "\"eLicenses[%d].rights_id\" is invalid.", index);
    eLicense.rightsId = rightsIdObj->value.GetUint64();

    auto ticketIdObj = eLicenseValue.FindMember("ticket_id");
    NN_ABORT_UNLESS(ticketIdObj != eLicenseValue.MemberEnd() && ticketIdObj->value.IsUint64(), "\"eLicenses[%d].ticket_id\" is invalid.", index);
    eLicense.ticketId = ticketIdObj->value.GetUint64();

    auto ticketOwnerVaIdObj = eLicenseValue.FindMember("ticket_owner_va_id");
    NN_ABORT_UNLESS(ticketOwnerVaIdObj != eLicenseValue.MemberEnd() && ticketOwnerVaIdObj->value.IsUint(), "\"eLicenses[%d].ticket_owner_va_id\" is invalid.", index);
    eLicense.ticketOwnerVaId = ticketOwnerVaIdObj->value.GetUint();

    auto ticketOwnerNaIdObj = eLicenseValue.FindMember("ticket_owner_na_id");
    NN_ABORT_UNLESS(ticketOwnerNaIdObj != eLicenseValue.MemberEnd() && ticketOwnerNaIdObj->value.IsUint64(), "\"eLicenses[%d].ticket_owner_na_id\" is invalid.", index);
    eLicense.ticketOwnerNaId.id = ticketOwnerNaIdObj->value.GetUint64();

    auto expireDateObj = eLicenseValue.FindMember("expire_date");
    NN_ABORT_UNLESS(expireDateObj != eLicenseValue.MemberEnd() && expireDateObj->value.IsInt64(), "\"eLicenses[%d].expire_date\" is invalid.", index);
    eLicense.expireDate.time.value = expireDateObj->value.GetInt64();

    auto scopeObj = eLicenseValue.FindMember("scope");
    NN_ABORT_UNLESS(scopeObj != eLicenseValue.MemberEnd() && scopeObj->value.IsString(), "\"eLicenses[%d].scope\" is invalid.", index);

    if (util::Strncmp("account", scopeObj->value.GetString(), scopeObj->value.GetStringLength()) == 0)
    {
        eLicense.scope = ELicenseScope::Account;
    }
    else if (util::Strncmp("device", scopeObj->value.GetString(), scopeObj->value.GetStringLength()) == 0)
    {
        eLicense.scope = ELicenseScope::Device;
    }
    else
    {
        NN_ABORT("\"eLicenses[%d].scope\" value is unexpected.", index);
    }

    auto availableAfterRebootObj = eLicenseValue.FindMember("available_after_reboot");
    NN_ABORT_UNLESS(availableAfterRebootObj != eLicenseValue.MemberEnd() && availableAfterRebootObj->value.IsBool(), "\"eLicenses[%d].available_after_reboot\" is invalid.", index);
    eLicense.availableAfterReboot = availableAfterRebootObj->value.GetBool();

    return eLicense;
}

}}}
