﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/err/err_ReportErrorApi.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/util/util_ScopeExit.h>

#include "detail\err_ErptUtil.h"

namespace nn { namespace err {

namespace
{
    void ReportErrorPreProcess(nn::ApplicationId applicationId) NN_NOEXCEPT
    {
        detail::SubmitApplicationId(applicationId);
        detail::SubmitFileSystemInfo();
    }

    void ReportErrorPostProcess() NN_NOEXCEPT
    {
        detail::ClearApplicationId();
        detail::ClearFileSystemInfo();
    }
}

void ReportError(nn::ApplicationId applicationId, nn::Result result) NN_NOEXCEPT
{
    ReportErrorPreProcess(applicationId);
    NN_UTIL_SCOPE_EXIT{ ReportErrorPostProcess(); };

    erpt::Context context(erpt::CategoryId::ErrorInfo);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AddErrorCode(result));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Invisible));
}

void ReportError(nn::ApplicationId applicationId, ErrorCode errorCode) NN_NOEXCEPT
{
    ReportErrorPreProcess(applicationId);
    NN_UTIL_SCOPE_EXIT{ ReportErrorPostProcess(); };

    erpt::Context context(erpt::CategoryId::ErrorInfo);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AddErrorCode(errorCode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Invisible));
}

void ReportError(nn::ApplicationId applicationId, nn::Result result, const ResultBacktrace& resultBacktrace) NN_NOEXCEPT
{
    ReportErrorPreProcess(applicationId);
    NN_UTIL_SCOPE_EXIT{ ReportErrorPostProcess(); };

    erpt::Context context(erpt::CategoryId::ErrorInfo);
    detail::AddResultBacktrace(&context, resultBacktrace);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AddErrorCode(result));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Invisible));
}

void ReportError(nn::ApplicationId applicationId, ErrorCode errorCode, const ResultBacktrace& resultBacktrace) NN_NOEXCEPT
{
    ReportErrorPreProcess(applicationId);
    NN_UTIL_SCOPE_EXIT{ ReportErrorPostProcess(); };

    erpt::Context context(erpt::CategoryId::ErrorInfo);
    detail::AddResultBacktrace(&context, resultBacktrace);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AddErrorCode(errorCode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Invisible));
}

void ReportError(nn::ApplicationId applicationId, nn::Result result, const ErrorContext& errorContext, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    // 1024u = sizeof(ErrorContext) + α。erpt::Context のデフォルトサイズに収まらないのでワークバッファが必要。
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, 1024u);

    ReportErrorPreProcess(applicationId);
    NN_UTIL_SCOPE_EXIT{ ReportErrorPostProcess(); };

    erpt::Context context(erpt::CategoryId::ErrorInfo, reinterpret_cast<uint8_t*>(workBuffer), static_cast<uint32_t>(workBufferSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AddErrorContext(errorContext));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AddErrorCode(result));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Invisible));
}

void ReportError(nn::ApplicationId applicationId, ErrorCode errorCode, const ErrorContext& errorContext, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    // 1024u = sizeof(ErrorContext) + α。erpt::Context のデフォルトサイズに収まらないのでワークバッファが必要。
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, 1024u);

    ReportErrorPreProcess(applicationId);
    NN_UTIL_SCOPE_EXIT{ ReportErrorPostProcess(); };

    erpt::Context context(erpt::CategoryId::ErrorInfo, reinterpret_cast<uint8_t*>(workBuffer), static_cast<uint32_t>(workBufferSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AddErrorContext(errorContext));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.AddErrorCode(errorCode));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Invisible));
}

}}
