﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ae.h>
#include <nn/am/am_Shim.h>
#include <nn/am/service/am_SystemProgramIds.h>
#include <nn/err/detail/err_Log.h>
#include <nn/err/err_SystemTypes.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_TFormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <cstring>

#include "err_ErptUtil.h"

namespace nn { namespace err { namespace detail {

namespace {
    void ClearErptContext(erpt::CategoryId category) NN_NOEXCEPT
    {
        auto context = erpt::Context(category);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
    }

    const char* GetStorageIdStr(const ncm::StorageId& storageId) NN_NOEXCEPT
    {
        switch( storageId )
        {
        case ncm::StorageId::BuildInSystem:
            return "BuildInSystem";
        case ncm::StorageId::BuildInUser:
            return "BuildInUser";
        case ncm::StorageId::Card:
            return "Card";
        case ncm::StorageId::Host:
            return "Host";
        case ncm::StorageId::None:
            return "None";
        case ncm::StorageId::SdCard:
            return "SdCard";
        default:
            return "(unknown)";
        }
    }

    struct SystemProgramIdPair
    {
        applet::AppletId        appletId;
        ncm::SystemProgramId    programId;
    };

    // エラービューア呼び出し時のメインアプレットがHOMEメニューだった場合に、エラー履歴の表示対象となるアプレット。
    SystemProgramIdPair g_ErrorReportVisibleApplet[] = {
        { applet::AppletId::AppletId_SystemAppletMenu,          am::service::ProgramId_SystemAppletMenu },
        { applet::AppletId::AppletId_LibraryAppletShop,         am::service::ProgramId_LibraryAppletShop },
        { applet::AppletId::AppletId_LibraryAppletPhotoViewer,  am::service::ProgramId_LibraryAppletPhotoViewer },
        { applet::AppletId::AppletId_LibraryAppletController,   am::service::ProgramId_LibraryAppletController },
        { applet::AppletId::AppletId_LibraryAppletMyPage,       am::service::ProgramId_LibraryAppletMyPage },
    };
}

Bit64 GetReportApplicationIdValue() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    auto mainAppletInfo = ae::GetMainAppletIdentityInfo();
    if( mainAppletInfo.IsApplication() )
    {
        // 呼び出し元のメインアプレットがアプリの場合には、途中の LA を問わずアプリのIDを記録する。
        return mainAppletInfo.applicationId.value;
    }
    else if( mainAppletInfo.IsSystemApplication() )
    {
        return mainAppletInfo.systemApplicationId.value;
    }
    else if( mainAppletInfo.appletId == applet::AppletId_SystemAppletMenu )
    {
        // 呼び出し元のメインアプレットが HOMEメニューだった場合、
        // アプレットの呼び出しスタック上エラービューアからもっとも近い特定のアプレット（ユーザー体験上別アプレットとみなすもの）のIDを記録する。
        const int MaxStackCount = 8; // 8 = メインアプレットまで到達するのに十分な、適当な値。
        am::service::AppletIdentityInfo info[MaxStackCount];
        int actualCount;
        sf::OutArray<am::service::AppletIdentityInfo> amInfoArray(info, MaxStackCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->GetCallerAppletIdentityInfoStack(&actualCount, amInfoArray));

        for( int i = 0; i < actualCount; i++ )
        {
            for( const auto& idPair : g_ErrorReportVisibleApplet )
            {
                if( info[i].appletId == idPair.appletId )
                {
                    return idPair.programId.value;
                }
            }
        }
        NN_SDK_ASSERT(false, "A valid applet is not found in the caller stack.");
        return am::service::ProgramId_SystemAppletMenu.value;
    }
    else
    {
        NN_SDK_ASSERT(false, "Unexpected main applet : %d", mainAppletInfo.appletId );
        return am::service::ProgramId_SystemAppletMenu.value;
    }
#elif( NN_BUILD_CONFIG_OS_WIN )
    return 0x1111222233334444LLU;
#endif
}

void SubmitApplicationInfo(void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(ns::ApplicationControlProperty));
    auto context = erpt::Context(erpt::CategoryId::ApplicationInfo);
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    auto appIdValue = GetReportApplicationIdValue();

    char idStr[17];
    util::SNPrintf(idStr, 17, "%016llx", appIdValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationID, idStr, static_cast<uint32_t>(util::Strnlen(idStr, 17))));

    auto pApplicationControlProperty = reinterpret_cast<ns::ApplicationControlProperty*>(workBuffer);
    if( ae::GetMainAppletApplicationControlProperty(pApplicationControlProperty).IsSuccess() )
    {
        auto appTitle = pApplicationControlProperty->GetDefaultTitle().name;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationTitle, appTitle, static_cast<uint32_t>(util::Strnlen(appTitle, sizeof(ns::ApplicationTitle::name)))));
        auto appVersion = pApplicationControlProperty->displayVersion;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationVersion, appVersion, static_cast<uint32_t>(util::Strnlen(appVersion, sizeof(ns::ApplicationControlProperty::displayVersion)))));
    }
    ncm::StorageId storageId;
    if( ae::GetMainAppletStorageId(&storageId) )
    {
        const char* storageName = GetStorageIdStr(storageId);
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationStorageLocation, storageName, static_cast<uint32_t>(strlen(storageName))));
    }
#elif( NN_BUILD_CONFIG_OS_WIN )
    NN_UNUSED(workBuffer);
    NN_UNUSED(workBufferSize);
    // テスト用に適当な固定値を入れる。
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationID, "1111222233334444", static_cast<uint32_t>(strlen("1111222233334444"))));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationTitle, "DummyApplication", static_cast<uint32_t>(strlen("DummyApplication"))));
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationVersion, "Ver. 1.2.3", static_cast<uint32_t>(strlen("Ver. 1.2.3"))));
    const char* storageName = GetStorageIdStr(ncm::StorageId::Card);
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.Add(erpt::ApplicationStorageLocation, storageName, static_cast<uint32_t>(strlen(storageName))));
#endif
    NN_ABORT_UNLESS_RESULT_SUCCESS(context.SubmitContext());
}

void AddResultBacktrace(erpt::Context* pOutValue, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(ResultBacktrace) + sizeof(uint32_t) * ResultBacktrace::CountMax);

    // workBuffer の先頭に ResultBackTrace を読み込む。
    applet::StorageHandle handle;
    NN_ABORT_UNLESS(ae::TryPopFromInChannel(&handle), "TrySubmitResultBackTrace : BackTrace is not available.\n");
    NN_UTIL_SCOPE_EXIT{ applet::ReleaseStorage(handle); };
    auto dataSize = applet::GetStorageSize(handle);
    NN_ABORT_UNLESS_EQUAL(dataSize, sizeof(ResultBacktrace));
    NN_ABORT_UNLESS_RESULT_SUCCESS(applet::ReadFromStorage(handle, 0, workBuffer, sizeof(ResultBacktrace)));
    auto pResultBacktrace = reinterpret_cast<const ResultBacktrace*>(workBuffer);

    // workBuffer の後半を Context に追加するため配列を作成するバッファに使用する。
    uint32_t* resultArray = reinterpret_cast<uint32_t*>( reinterpret_cast<Bit8*>(workBuffer) + sizeof(ResultBacktrace) );
    for( int i = 0; i < pResultBacktrace->count; i++ )
    {
        resultArray[i] = pResultBacktrace->results[i].GetInnerValueForDebug();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(pOutValue->Add(erpt::FieldId::ResultBacktrace, resultArray, static_cast<uint32_t>(pResultBacktrace->count)));
#else
    NN_DETAIL_ERR_INFO("AddResultBacktrace : Not supported on this platform.\n");
    NN_UNUSED(pOutValue);
    NN_UNUSED(workBuffer);
    NN_UNUSED(workBufferSize);
#endif
}

void AddErrorContext(erpt::Context* pOutValue, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, sizeof(ErrorContext));

    applet::StorageHandle handle;
    NN_ABORT_UNLESS(ae::TryPopFromInChannel(&handle), "TryPopFromInChannel : ErrorContext is not available.\n");
    NN_UTIL_SCOPE_EXIT{ applet::ReleaseStorage(handle); };
    auto dataSize = applet::GetStorageSize(handle);
    NN_ABORT_UNLESS_EQUAL(dataSize, sizeof(ErrorContext));
    NN_ABORT_UNLESS_RESULT_SUCCESS(applet::ReadFromStorage(handle, 0, workBuffer, sizeof(ErrorContext)));
    auto pErrorContext = reinterpret_cast<const ErrorContext*>(workBuffer);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pOutValue->AddErrorContext(*pErrorContext));
#else
    NN_DETAIL_ERR_INFO("AddErrorContext : Not supported on this platform.\n");
    NN_UNUSED(pOutValue);
    NN_UNUSED(workBuffer);
    NN_UNUSED(workBufferSize);
#endif
}

void ClearApplicationInfo() NN_NOEXCEPT
{
    ClearErptContext(erpt::ApplicationInfo);
}

}}}
