﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt.h>
#include <nn/pcv/pcv.h>

#include "eclct_PowerClockInfo.h"
#include "eclct_Util.h"

namespace nn    {
namespace eclct {

namespace {

bool g_Initialized = false;

#if (defined NN_BUILD_CONFIG_SPEC_NX) || (defined NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
// OscillatorClock
uint32_t g_OscillatorClock;

// DVFS
int32_t g_CpuDataCount;
int32_t g_GpuDataCount;
int32_t g_EmcDataCount;
uint32_t g_CpuClocks[nn::pcv::DvfsTableMaxDataCount];
uint32_t g_GpuClocks[nn::pcv::DvfsTableMaxDataCount];
uint32_t g_EmcClocks[nn::pcv::DvfsTableMaxDataCount];
int32_t g_CpuVoltages[nn::pcv::DvfsTableMaxDataCount];
int32_t g_GpuVoltages[nn::pcv::DvfsTableMaxDataCount];
int32_t g_EmcVoltages[nn::pcv::DvfsTableMaxDataCount];

// チップ情報
uint32_t g_FuseValues[nn::pcv::FuseValueCount];

// Dram情報
uint32_t g_DramId;
#endif

const uint32_t ContextBufferSize = 1024;
uint8_t g_ContextBuffer[ContextBufferSize];

void Initialize() NN_NOEXCEPT
{
    if ( !g_Initialized )
    {
        nn::pcv::Initialize();

#if (defined NN_BUILD_CONFIG_SPEC_NX) || (defined NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
        // 静的なデータは、一度だけ取得を行う
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(nn::pcv::GetOscillatorClock(&g_OscillatorClock));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(nn::pcv::GetDvfsTable(g_CpuClocks, g_CpuVoltages, &g_CpuDataCount, nn::pcv::Module_Cpu, nn::pcv::DvfsTableMaxDataCount));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(nn::pcv::GetDvfsTable(g_GpuClocks, g_GpuVoltages, &g_GpuDataCount, nn::pcv::Module_Gpu, nn::pcv::DvfsTableMaxDataCount));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(nn::pcv::GetDvfsTable(g_EmcClocks, g_EmcVoltages, &g_EmcDataCount, nn::pcv::Module_Emc, nn::pcv::DvfsTableMaxDataCount));

        int32_t fuseValueCount;
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(nn::pcv::GetFuseInfo(g_FuseValues, &fuseValueCount, nn::pcv::FuseValueCount));
        NN_SDK_ASSERT_EQUAL(fuseValueCount, nn::pcv::FuseValueCount);

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(nn::pcv::GetDramId(&g_DramId));
#endif

        g_Initialized = true;
    }
}

}

void GetPowerClockInfoEvent(nn::os::SystemEventType* pOutSystemEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    Initialize();

    nn::pcv::GetPowerClockInfoEvent(pOutSystemEvent, eventClearMode);
}

void UpdatePowerClockInfo() NN_NOEXCEPT
{
    Initialize();

    nn::erpt::Context context(nn::erpt::PowerClockInfo, g_ContextBuffer, ContextBufferSize);

#if (defined NN_BUILD_CONFIG_SPEC_NX) || (defined NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
    // 静的なデータの送信
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::OscillatorClock, g_OscillatorClock));

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::CpuDvfsTableClocks, g_CpuClocks, g_CpuDataCount));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::CpuDvfsTableVoltages, g_CpuVoltages, g_CpuDataCount));

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::GpuDvfsTableClocks, g_GpuClocks, g_GpuDataCount));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::GpuDvfsTableVoltages, g_GpuVoltages, g_GpuDataCount));

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::EmcDvfsTableClocks, g_EmcClocks, g_EmcDataCount));
    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::EmcDvfsTableVoltages, g_EmcVoltages, g_EmcDataCount));

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::FuseInfo, g_FuseValues, nn::pcv::FuseValueCount));

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::DramId, g_DramId));

    // ModuleState の送信
    {
        int32_t count;
        nn::pcv::ModuleState states[nn::pcv::ModuleStateTableSize];

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(nn::pcv::GetModuleStateTable(states, &count, nn::pcv::ModuleStateTableSize));

        uint32_t clockFrequencies[nn::pcv::ModuleStateTableSize];
        uint8_t clockEnableFlags[nn::pcv::ModuleStateTableSize];
        uint8_t powerEnableFlags[nn::pcv::ModuleStateTableSize];
        uint8_t resetAssertFlags[nn::pcv::ModuleStateTableSize];
        uint32_t minimumVotageClockRates[nn::pcv::ModuleStateTableSize];

        for (int i = 0; i < nn::pcv::ModuleStateTableSize; i++)
        {
            clockFrequencies[i] = states[i].clockFrequency;
            clockEnableFlags[i] = states[i].clockEnabled;
            powerEnableFlags[i] = states[i].powerEnabled;
            resetAssertFlags[i] = states[i].resetAsserted;
            minimumVotageClockRates[i] = states[i].minVClockRate;
        }

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ModuleClockFrequencies, clockFrequencies, count));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ModuleClockEnableFlags, clockEnableFlags, count));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ModulePowerEnableFlags, powerEnableFlags, count));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ModuleResetAssertFlags, resetAssertFlags, count));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ModuleMinimumVoltageClockRates, minimumVotageClockRates, count));
    }

    // PowerDomainState の送信
    {
        int32_t count = 0;
        nn::pcv::PowerDomainState states[nn::pcv::PowerDomainStateTableSizeMax];

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(nn::pcv::GetPowerDomainStateTable(states, &count, nn::pcv::PowerDomainStateTableSizeMax));

        uint8_t enableFlags[nn::pcv::PowerDomainStateTableSizeMax];
        int32_t voltages[nn::pcv::PowerDomainStateTableSizeMax];

        for (int i = 0; i < count; i++)
        {
            enableFlags[i] = states[i].enabled;
            voltages[i] = states[i].voltage;
        }

        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PowerDomainEnableFlags, enableFlags, count));
        NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::PowerDomainVoltages, voltages, count));
    }
#endif

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
}

}}
