﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/bgtc.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/eclct/detail/eclct_Log.h>
#include <nn/eclct/eclct.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/eupld/eupld_Control.h>
#include <nn/fs/fs_SystemDataUpdateEvent.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiTelemetry.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/os.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/settings/system/settings_Telemetry.h>
#include <nn/time.h>

#include "eclct_BatteryCharge.h"
#include "eclct_Display.h"
#include "eclct_Nifm.h"
#include "eclct_Pcie.h"
#include "eclct_PowerClockInfo.h"
#include "eclct_Settings.h"
#include "eclct_Util.h"

namespace nn    {
namespace eclct {
namespace       {

const int AccountUpdateTimeoutSeconds = (5 * 60);

uint32_t  g_AccountInfoUpdatePeriodSeconds;
bool      g_AnalyticsFlag;
char      g_PlayReportBuffer[1024];

#ifdef USE_THREAD
const uint32_t     ThreadStackSize = 16 * 1024;
nn::os::ThreadType g_WorkerThread;
uint8_t            g_WorkerThreadStack[ThreadStackSize] NN_ALIGNAS(os::StackRegionAlignment);
#endif

nn::Result GetOptInFlagAndNintendoId(const nn::account::Uid& uid, nn::account::NintendoAccountId& nintendoId, bool& optInFlag)
NN_NOEXCEPT
{
    static Bit8 buffer[nn::account::RequiredBufferSizeForCachedNintendoAccountInfo];

    nn::account::NetworkServiceAccountManager accountManager;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountManager(&accountManager, uid));

    bool isCachRefreshStarted = false;
    nn::account::AsyncContext accountContext;
    // If the cached Nintendo Account info is updated within (g_AccountInfoUpdatePeriodSeconds / 8), use that info without update. 8 is just an arbitrary value.
    NN_RESULT_DO(accountManager.RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed(&isCachRefreshStarted, &accountContext, TimeSpan::FromSeconds(g_AccountInfoUpdatePeriodSeconds / 8)));
    if( isCachRefreshStarted )
    {
        NN_DETAIL_ECLCT_TRACE("[GetOptInFlagAndNintendoId] Refresh cached Nintendo Account started.\n");
        nn::os::SystemEvent accountContextEvent;
        NN_ABORT_UNLESS_RESULT_SUCCESS(accountContext.GetSystemEvent(&accountContextEvent));

        if( !accountContextEvent.TimedWait(nn::TimeSpan::FromSeconds(AccountUpdateTimeoutSeconds)) )
        {
            accountContext.Cancel();
            accountContextEvent.Wait();
            return nn::os::ResultTimedout();
        }
        NN_RESULT_DO(accountContext.GetResult());
    }
    else
    {
        NN_DETAIL_ECLCT_TRACE("[GetOptInFlagAndNintendoId] Refresh cached Nintendo Account is not necessary.\n");
    }

    nn::account::CachedNintendoAccountInfoForSystemService accountInfo;
    NN_RESULT_DO(accountManager.LoadCachedNintendoAccountInfo(&accountInfo, buffer, sizeof(buffer)));
    optInFlag  = accountInfo.GetAnalyticsOptedInFlag();
    nintendoId = accountInfo.GetId();

    return ResultSuccess();
}

void UpdateAccountInfo()
NN_NOEXCEPT
{
    nifm::NetworkConnection nc;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::SetRequestRequirementPreset(nc.GetRequestHandle(), nifm::RequirementPreset_InternetForSystemProcessSharable));
    nc.SubmitRequestAndWait();
    if( !nc.IsAvailable() )
    {
        NN_DETAIL_ECLCT_TRACE("[UpdateAccountInfo] Network is not available. ErrorReport auto upload will be disabled unless cached Nintendo Account Info can be used.\n");
    }

    int                userIdCount;
    nn::account::Uid   userIds[nn::account::UserCountMax] = {};
    nn::eupld::Control eupldControl;
    bool               analyticsFlag = g_AnalyticsFlag;

    if ((nn::account::ListAllUsers(&userIdCount, userIds, sizeof(userIds) / sizeof(userIds[0]))).IsSuccess())
    {
        for (int i = 0; i < userIdCount && !analyticsFlag; i++)
        {
            bool optInFlag;
            nn::account::NintendoAccountId nintendoId;

            auto result = GetOptInFlagAndNintendoId(userIds[i], nintendoId, optInFlag);
            if( result.IsFailure() )
            {
                NN_DETAIL_ECLCT_TRACE("[UpdateAccountInfo] Failed to get opt-in flag of uid<%016llx-%016llx> : %03d-%04d (0x%08x)\n",
                    userIds[i]._data[0], userIds[i]._data[1], result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
                continue;
            }
            NN_UNUSED(nintendoId);
            analyticsFlag |= optInFlag;
        }
    }

    NN_DETAIL_ECLCT_TRACE(
        "[UpdateAccountInfo] analytics state \"%s\", re-scanning in %d seconds\n",
        analyticsFlag ? "enabled" : "disabled",
        g_AccountInfoUpdatePeriodSeconds
    );

    eupldControl.SetAutoUpload(analyticsFlag, g_AccountInfoUpdatePeriodSeconds);
}

void ReportNpnsInfo()
NN_NOEXCEPT
{
    NN_DETAIL_ECLCT_TRACE("[ReportNpnsInfo] Generating play report...\n");
    // Unique id for this boot session
    static const nn::util::Uuid uuid = nn::util::GenerateUuid();

    // Collect statistics to report
    nn::npns::Statistics statistics;
    nn::Result result = nn::npns::GetStatistics(&statistics);
    if (result.IsFailure())
    {
        return;
    }

    nn::prepo::SystemPlayReport report("statistics");
    report.SetBuffer(g_PlayReportBuffer, sizeof(g_PlayReportBuffer));
    {
        const nn::ApplicationId AppId = { nn::npns::ApplicationIdRaw };
        report.SetApplicationId(AppId);
    }
    {
        char uuidBuffer[util::Uuid::StringSize];
        report.Add("uuid", uuid.ToString(uuidBuffer, sizeof(uuidBuffer)));
    }

    for (auto it = statistics.cbegin(); it != statistics.cend(); ++it)
    {
        report.Add(statistics.GetName(it), static_cast<int64_t>(*it));
    }

    report.Save();
    NN_DETAIL_ECLCT_TRACE("[ReportNpnsInfo] Saved play report.\n");
}

nn::Result ReportErrorReportStorageUsageStatistics()
NN_NOEXCEPT
{
    NN_DETAIL_ECLCT_TRACE("[ReportErrorReportStorageUsageStatistics] Generating play report...\n");

    erpt::Manager manager;
    NN_RESULT_DO(manager.Initialize());
    NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };

    nn::erpt::StorageUsageStatistics usage;
    NN_RESULT_DO(manager.GetStorageUsageStatistics(&usage));

    nn::prepo::SystemPlayReport report("storage_usage");
    report.SetBuffer(g_PlayReportBuffer, sizeof(g_PlayReportBuffer));
    NN_RESULT_DO(report.SetApplicationId(nn::ApplicationId{ 0x010000000000002B }));
    char uuidStr[64];
    NN_RESULT_DO(report.Add("journal_id", usage.journalUuid.ToString(uuidStr, sizeof(uuidStr))));
    NN_RESULT_DO(report.Add("used_storage_size", static_cast<int64_t>(usage.usedStorageSize)));
    NN_RESULT_DO(report.Add("max_report_size", usage.maxReportSize));
    NN_RESULT_DO(report.Add("visible_stored_count", static_cast<int64_t>(usage.reportCount[erpt::ReportType_Visible])));
    NN_RESULT_DO(report.Add("invisible_stored_count", static_cast<int64_t>(usage.reportCount[erpt::ReportType_Invisible])));
    NN_RESULT_DO(report.Add("visible_transmitted_count", static_cast<int64_t>(usage.transmittedCount[erpt::ReportType_Visible])));
    NN_RESULT_DO(report.Add("invisible_transmitted_count", static_cast<int64_t>(usage.transmittedCount[erpt::ReportType_Invisible])));
    NN_RESULT_DO(report.Add("visible_untransmitted_count", static_cast<int64_t>(usage.untransmittedCount[erpt::ReportType_Visible])));
    NN_RESULT_DO(report.Add("invisible_untransmitted_count", static_cast<int64_t>(usage.untransmittedCount[erpt::ReportType_Invisible])));
    NN_RESULT_DO(report.Save());
    NN_RESULT_SUCCESS;
}

void UpdateRebootlessSystemUpdateVersion()
NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::CategoryId::RebootlessSystemUpdateVersionInfo);

    nn::settings::system::RebootlessSystemUpdateVersion version;
    nn::settings::system::GetRebootlessSystemUpdateVersion(&version);

    context.Add(nn::erpt::RebootlessSystemUpdateVersion, version.displayVersion, static_cast<uint32_t>(std::strlen(version.displayVersion)));

    context.SubmitContext();
}

void WorkerThread(void* arg)
NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::os::SystemEvent settingsEvent;
    nn::os::SystemEvent nifmEvent;
    nn::os::SystemEvent batteryChargeInfoEvent;
    nn::os::SystemEvent batteryChargeCalibratedEvent;
    nn::os::SystemEvent powerClockEvent;
    nn::os::SystemEvent npnsPlayReportEvent;
    nn::os::SystemEvent compositorEvent;
    nn::os::TimerEvent  reportErrorReportStorageUsageStatisticsTimer(nn::os::EventClearMode_ManualClear);
    nn::os::SystemEvent hotplugEvent;
    nn::os::SystemEvent resolutionChangeEvent;
    std::unique_ptr<nn::fs::IEventNotifier> rebootlessSystemUpdataEventNotifier;
    nn::os::SystemEvent rebootlessSystemUpdateEvent;
    nn::os::SystemEvent pcieLoggedDataEvent;

    nn::account::InitializeForSystemService();
    nn::nifm::InitializeSystem();
    nn::npns::InitializeForSystem();
    nn::prepo::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(InitializePcie());

    bgtc::Task bgtcTask;
    NN_ABORT_UNLESS_RESULT_SUCCESS(bgtcTask.Initialize());
    bgtcTask.SchedulePeriodic(static_cast<bgtc::Interval>(g_AccountInfoUpdatePeriodSeconds), static_cast<bgtc::Interval>(g_AccountInfoUpdatePeriodSeconds));
    auto& bgtcScheduleEvent = bgtcTask.GetScheduleEvent();

    nn::settings::system::BindTelemetryDirtyFlagEvent(
        settingsEvent.GetBase(),
        nn::os::EventClearMode_ManualClear
    );

    nn::nifm::BindTelemetryInfoEvent(
        nifmEvent.GetBase(),
        nn::os::EventClearMode_ManualClear
    );

    nn::npns::BindPlayReportRequestEvent(
        npnsPlayReportEvent.GetBase(),
        nn::os::EventClearMode_ManualClear
    );

    GetBatteryChargeInfoEvent(batteryChargeInfoEvent.GetBase(), nn::os::EventClearMode_ManualClear);
    GetBatteryChargeCalibratedEvent(batteryChargeCalibratedEvent.GetBase(), nn::os::EventClearMode_ManualClear);
    GetPowerClockInfoEvent(powerClockEvent.GetBase(), nn::os::EventClearMode_ManualClear);
    GetCompositorInfoEvent(compositorEvent.GetBase());
    reportErrorReportStorageUsageStatisticsTimer.StartOneShot(nn::TimeSpan::FromMinutes(1));
    GetHotplugEvent(hotplugEvent.GetBase());
    GetResolutionChangeEvent(&resolutionChangeEvent);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenSystemDataUpdateEventNotifier(&rebootlessSystemUpdataEventNotifier));
    NN_ABORT_UNLESS_RESULT_SUCCESS(rebootlessSystemUpdataEventNotifier.get()->BindEvent(rebootlessSystemUpdateEvent.GetBase(), nn::os::EventClearMode_ManualClear));
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPcieLoggedStateEvent(pcieLoggedDataEvent.GetBase(), nn::os::EventClearMode_ManualClear));

    UpdateStaticFields();
    UpdateSettingsInfo();
    UpdateBatteryChargeInfo();
    UpdateAccountInfo();
    UpdateNifmInfo();
    UpdatePowerClockInfo();
    UpdateMonitorSettings();
    UpdateRebootlessSystemUpdateVersion();
    UpdatePcieLoggedState();

    while (NN_STATIC_CONDITION(true))
    {
        int eventId = nn::os::WaitAny(
                          bgtcScheduleEvent.GetBase(),
                          settingsEvent.GetBase(),
                          nifmEvent.GetBase(),
                          batteryChargeInfoEvent.GetBase(),
                          batteryChargeCalibratedEvent.GetBase(),
                          powerClockEvent.GetBase(),
                          npnsPlayReportEvent.GetBase(),
                          compositorEvent.GetBase(),
                          reportErrorReportStorageUsageStatisticsTimer.GetBase(),
                          hotplugEvent.GetBase(),
                          resolutionChangeEvent.GetBase(),
                          rebootlessSystemUpdateEvent.GetBase(),
                          pcieLoggedDataEvent.GetBase());

        switch (eventId)
        {
        case 0:
            {
                bgtcScheduleEvent.Clear();
                auto bgtcTaskNotifyStartingResult = bgtcTask.NotifyStarting();
                if( bgtcTaskNotifyStartingResult.IsFailure() )
                {
                    NN_DETAIL_ECLCT_TRACE("[WorkerThread] bgtcTask.NotifyStarting failed : %03d-%04d (0x%08x)\n",
                        bgtcTaskNotifyStartingResult.GetModule(), bgtcTaskNotifyStartingResult.GetDescription(), bgtcTaskNotifyStartingResult.GetInnerValueForDebug());
                }
                NN_UTIL_SCOPE_EXIT{
                    if( bgtcTaskNotifyStartingResult.IsSuccess() )
                    {
                        bgtcTask.NotifyFinished();
                    }
                };
                UpdateAccountInfo();
            }
            break;
        case 1:
            settingsEvent.Clear();
            UpdateSettingsInfo();
            break;
        case 2:
            nifmEvent.Clear();
            UpdateNifmInfo();
            break;
        case 3:
            batteryChargeInfoEvent.Clear();
            UpdateBatteryChargeInfo();
            break;
        case 4:
            batteryChargeCalibratedEvent.Clear();
            ReportBatteryChargeCalibrated();
            break;
        case 5:
            powerClockEvent.Clear();
            UpdatePowerClockInfo();
            break;
        case 6:
            npnsPlayReportEvent.Clear();
            ReportNpnsInfo();
            break;
        case 7:
            compositorEvent.Clear();
            UpdateCompositorInfo();
            break;
        case 8:
            reportErrorReportStorageUsageStatisticsTimer.Clear();
            NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(ReportErrorReportStorageUsageStatistics());
            reportErrorReportStorageUsageStatisticsTimer.StartOneShot(nn::TimeSpan::FromDays(1));
            break;
        case 9:
            // Update current display mode after every hotplug:
            //   1. HPD high => nvnflinger will choose some mode based on EDID and system settings.
            //   2. HPD low  => zero out current resolution/framerate due to handheld mode.
            hotplugEvent.Clear();
            UpdateMonitorSettings();
            break;
        case 10:
            // Users may change external display mode and system menus will issue a signal.
            resolutionChangeEvent.Clear();
            UpdateMonitorSettings();
            break;
        case 11:
            rebootlessSystemUpdateEvent.Clear();
            UpdateRebootlessSystemUpdateVersion();
            break;
        case 12:
            pcieLoggedDataEvent.Clear();
            UpdatePcieLoggedState();
            break;
        default:
            break;
        }
    }
} // NOLINT(impl/function_size)

} // namespace

#ifdef USE_THREAD
nn::Result Initialize(bool analyticsOverride, uint32_t analyticsPollPeriodSeconds)
NN_NOEXCEPT
{
    nn::Result result;

    g_AnalyticsFlag                  = analyticsOverride;
    g_AccountInfoUpdatePeriodSeconds = analyticsPollPeriodSeconds;

    result = nn::os::CreateThread(
                &g_WorkerThread,
                WorkerThread,
                nullptr,
                g_WorkerThreadStack,
                sizeof(g_WorkerThreadStack),
                NN_SYSTEM_THREAD_PRIORITY(eclct, Worker));

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SetThreadNamePointer(&g_WorkerThread, NN_SYSTEM_THREAD_NAME(eclct, Worker));
    nn::os::StartThread(&g_WorkerThread);

    return ResultSuccess();
}

void Wait()
NN_NOEXCEPT
{
    nn::os::WaitThread(&g_WorkerThread);
}
#endif

void InitializeAndRun(bool analyticsOverride, uint32_t analyticsPollPeriodSeconds)
NN_NOEXCEPT
{
    g_AnalyticsFlag                  = analyticsOverride;
    g_AccountInfoUpdatePeriodSeconds = analyticsPollPeriodSeconds;
    WorkerThread(nullptr);
}

}}
