﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <iomanip>
#include <nn/ec/ec_ConsumableServiceItemTypes.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_TFormatString.h>
#include <nn/ec/ec_ResultConsumableServiceItem.h>
#include "ec_ConsumableServiceItemRightDataAdaptor.h"
#include "../../ec_ConsumableServiceItemInternalTypes.h"

namespace
{
#define NN_RESULT_SUCCESS_UNLESS(expression)\
    do { auto _r = (expression); if (!_r) NN_RESULT_SUCCESS; } while (NN_STATIC_CONDITION(false))
}

namespace nn { namespace ec { namespace json { namespace adaptor {

Result ConsumableServiceItemRightDataAdaptor::UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    //! Update, NotifyObject にて、取得する最大権利数を超過した場合、出力バッファの容量不足をアプリケーションに返す
    NN_RESULT_THROW_UNLESS(m_RightIndex < m_MaxRightCount, nn::ec::ResultConsumableServiceItemInsufficientBuffer());

    MakeRightsPath(m_PathBuffer, m_PathBufferSize, "purchased_date_time");
    if (jsonPath.Match(m_PathBuffer))
    {
        m_pRights[m_RightIndex].purchasedPosixTime.value = value;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_SUCCESS;
}

Result ConsumableServiceItemRightDataAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_RightIndex < m_MaxRightCount, nn::ec::ResultConsumableServiceItemInsufficientBuffer());

    //! status の判定結果によって、エラーが返るため最初に判定する
    MakeRightsPath(m_PathBuffer, m_PathBufferSize, "status");
    if (jsonPath.Match(m_PathBuffer))
    {
        //! 固定文字列と比較するので std::strcmp で十分（nn::util::Strncmp はあえて使用しない）
        if (std::strcmp(value, "PURCHASED") == 0)
        {
            m_pRights[m_RightIndex].status = ConsumableServiceItemServerRightStatus_Purchased;
            NN_RESULT_SUCCESS;
        }
        else if (strcmp(value, "CONSUMED") == 0)
        {
            m_pRights[m_RightIndex].status = ConsumableServiceItemServerRightStatus_Consumed;
            NN_RESULT_SUCCESS;
        }
        else
        {
            //! サーバー上の権利情報が "購入済み"、"消費済み" ではない状態になっている。
            //! 現状の Sugar の仕様では、これら以外の状態が存在しない。
            //! そのため、このスコープに来る時点で、権利情報が異常な状態に陥っていると考える。
            //! Sugar の仕様変更で、権利情報の状態が増えた場合、適宜、ココの判定処理を追従させる必要がある。
            NN_RESULT_THROW(nn::ec::ResultConsumableServiceItemInvalidServerRightStatus());
        }
    }

    MakeRightsPath(m_PathBuffer, m_PathBufferSize, "right_id");
    if (jsonPath.Match(m_PathBuffer))
    {
        util::Strlcpy(m_pRights[m_RightIndex].rightId.value, value, sizeof(m_pRights[m_RightIndex].rightId.value));
        NN_RESULT_SUCCESS;
    }

    MakeRightsPath(m_PathBuffer, m_PathBufferSize, "item_id");
    if (jsonPath.Match(m_PathBuffer))
    {
        util::Strlcpy(m_pRights[m_RightIndex].itemId.value, value, sizeof(m_pRights[m_RightIndex].itemId.value));
        NN_RESULT_SUCCESS;
    }

    MakeRightsPath(m_PathBuffer, m_PathBufferSize, "consumption_request_id");
    if (jsonPath.Match(m_PathBuffer))
    {
        util::Strlcpy(m_pRights[m_RightIndex].consumptionRequestId.value, value, sizeof(m_pRights[m_RightIndex].consumptionRequestId.value));
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_SUCCESS;
}

Result ConsumableServiceItemRightDataAdaptor::NotifyObjectBeginImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_RightIndex < m_MaxRightCount, nn::ec::ResultConsumableServiceItemInsufficientBuffer());
    NN_RESULT_SUCCESS;
}

Result ConsumableServiceItemRightDataAdaptor::NotifyObjectEndImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    //! @internal_details
    //!     NotifyObjectEndImpl は JSON テキスト内の "}" を検知するたびに呼ばれる
    //!     オブジェクト終端時の処理は、以下のようになる
    //!
    //!     - "rights" オブジェクトごとの終端処理
    //!
    //!       m_RightIndex < m_MaxRightCount を満たすかを確認する (出力バッファの充足判定)
    //!       m_RightIndex をインクリメントする
    //!
    //!     - 最上位オブジェクト (Root オブジェクト) の終端処理
    //!
    //!       m_RightIndex を出力する権利数として返す
    //!       この時点で m_RightIndex は権利数にインクリメントされている。別途、インクリメントは不要
    //!
    //! @code
    //!           : {
    //!           :     "total_results":int,
    //!           :     "rights": [
    //!           :         {
    //!           :             "right_id":string,
    //!           :             "item_id":string,
    //!           :             ...
    //!           :             "consumption_request_id":string
    //! Right End :         },
    //!           :         {
    //!           :             ...
    //! Right End :         },
    //!           :         ...
    //!           :         {
    //!           :             ...
    //! Right End :         }
    //!           :     ]
    //!  Root End : }
    //!

    //! NotifyObjectEndImpl の呼び出しが Root オブジェクト終端によるものかを判定する
    MakeEndPath(m_PathBuffer, m_PathBufferSize);
    if (jsonPath.Match(m_PathBuffer))
    {
        //! レスポンスの終端で RightCount を決定する。total_results との正当性はチェックしない
        *m_pRightCount = m_RightIndex;
        NN_RESULT_SUCCESS;
    }

    //! 以下、Root オブジェクト終端以外で関数が呼び出された場合の処理を記述する

    NN_RESULT_THROW_UNLESS(m_RightIndex < m_MaxRightCount, nn::ec::ResultConsumableServiceItemInsufficientBuffer());

    //! 現時点では、"rights" オブジェクト以外での終端処理が走らないため、権利情報に対するインデックスをインクリメントするだけ
    m_RightIndex++;

    NN_RESULT_SUCCESS;
}

int ConsumableServiceItemRightDataAdaptor::MakeRightsPath(char* buffer, size_t bufferSize, const char* propertyName) NN_NOEXCEPT
{
    return util::TSNPrintf(buffer, bufferSize, "$.rights[%d].%s", m_RightIndex, propertyName);
}

int ConsumableServiceItemRightDataAdaptor::MakeEndPath(char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return util::TSNPrintf(buffer, bufferSize, "$");
}

}}}}
