﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/ec/ec_ConsumableServiceItemTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Uuid.h>

#include "ec_ConsumableServiceItemInternalTypes.h"
#include "ec_ConsumableServiceItemAllocatedBuffer.h"
#include "ec_ConsumableServiceItemInternalApi.h"
#include "json/ec_InputStream.h"
#include "json/ec_OutputStream.h"
#include "json/adaptor/ec_ConsumableServiceItemRightDataAdaptor.h"

#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_JsonErrorMap.h>

#include <nn/ec/ec_ResultConsumableServiceItem.h>

//! POST データ作成用にインクルード
#include <rapidjson/allocators.h>
#include <rapidjson/writer.h>

namespace
{
//! BasePath の最大サイズ
//! ASIS: GetConsumableRightData の BasePath が最大で 256 byte 確保しておく
static const int BasePathSizeMax = 256;

//! 消費型サービスアイテムの Right Id の文字列長
//! GetProvidableRightData を実行するための URL 形成のために用います
const size_t ConsumableServiceItemRightIdLength = nn::ec::ConsumableServiceItemRightId::Length;

struct Cancellable
{
    bool IsCancelled() const { return false; }
};

/**
    @brief MakePath を実行するために必要なパスサイズを求めます。

    @details
        想定される最大確保メモリは
        basePathLength 21 byte + (ConsumableServiceItemRightIdLength 36 byte + sizeof(',')) * ConsumableServiceItemRightDataCountMax 20  - sizeof(',') + sizeof('\0')
*/
size_t GetRequiredPathSizeForMakePath(const char basePath[], size_t rightDataCount) NN_NOEXCEPT
{
    //! ライブラリから固定長の文字列を受け取るため、std::strlen で十分
    size_t basePathLength = std::strlen(basePath);

    //! /v1/rights?right_ids=uuid,uuid,uuid,...,uuid\0
    return basePathLength + (ConsumableServiceItemRightIdLength + sizeof(',')) * rightDataCount - sizeof(',') + sizeof('\0');
}

//! Right Id を複数指定した URL を作成します。ConsumableServiceItemRightData 指定での権利参照 API に使用します
nn::Result MakePath(char outPath[], const size_t outPathSizeMax, const char pBasePath[], const nn::ec::ConsumableServiceItemRightData rightDataArray[], size_t rightDataCount) NN_NOEXCEPT
{
    auto outPathSize = GetRequiredPathSizeForMakePath(pBasePath, rightDataCount);
    NN_RESULT_THROW_UNLESS(outPathSize <= outPathSizeMax, nn::ec::ResultInsufficientWorkMemory());

    //! 以下、outPath に全ての Path を格納できることが保証される

    auto path = reinterpret_cast<char*>(outPath);

    //! Base Path コピー
    //! ライブラリから固定長の文字列を受け取るため、std::strlen で十分
    size_t basePathLength = std::strlen(pBasePath);
    std::strncat(path, pBasePath, basePathLength + sizeof('\0'));

    //! Right Id コピー
    for (int i = 0; i < rightDataCount; i++)
    {
        if (i > 0)
        {
            std::strncat(path, ",", sizeof(","));
        }

        auto info = nn::util::Get(rightDataArray[i].value);
        std::strncat(path, info.rightId.value, sizeof(info.rightId.value));
    }

    NN_RESULT_SUCCESS;
}

//! Right Id と Consumption Request Id の組を POST データにします。権利消費 API に使用します
nn::Result MakePostData(char pOutPostData[], int postDataSize, const nn::ec::ConsumableServiceItemRightData rightDataArray[], int rightDataCount, nn::mem::StandardAllocator* pLibAllocator) NN_NOEXCEPT
{
    //! 渡したメモリブロック以上の領域にメモリ確保が行われた場合 Abort させるアロケータ
    typedef nne::rapidjson::MemoryPoolAllocator<nn::http::json::NoAllocatorForRapidJson> NotAddChunkMemoryPoolAllocator;

    //! MemoryPoolAllocator のワークメモリを確保する
    size_t workMemorySizeForMemoryPoolAllocator = nn::os::MemoryPageSize;
    nn::ec::AllocatedBuffer<char> pWorkMemoryForMemoryPoolAllocator(pLibAllocator);
    NN_RESULT_DO(pWorkMemoryForMemoryPoolAllocator.Allocate(workMemorySizeForMemoryPoolAllocator));

    //! MemoryPoolAllocator のインスタンスを生成する
    //! AddChunk はしないため、容量不足の時に確保するチャンクサイズは 1u に設定しておく
    nn::http::json::NoAllocatorForRapidJson noAllocator;
    NotAddChunkMemoryPoolAllocator memoryPoolAllocator(pWorkMemoryForMemoryPoolAllocator.Get(), workMemorySizeForMemoryPoolAllocator, 1u, &noAllocator);

    //! OutputStream の出力メモリを postDataSize byte 確保する
    nn::ec::AllocatedBuffer<char> pWorkMemoryForOutputStream(pLibAllocator);
    NN_RESULT_DO(pWorkMemoryForOutputStream.Allocate(postDataSize));

    //! OutputStream のインスタンスを生成する
    nn::ec::json::NoAllocationOutputStream outputStream;
    outputStream.SetStringBuffer(pWorkMemoryForOutputStream.Get(), postDataSize);

    //! RapidJSON Writer のインスタンスを生成する
    //! outputStream で渡したワークメモリ以上に、メモリ割り当てが発生した場合、NotAddChunkMemoryPoolAllocator が発動して Abort する
    //! nn::ec::RequiredWorkMemorySizeForConsumableServiceItem 以上のワークメモリをアプリケーションから渡されているかぎり、Abort しないことをライブラリが担保する
    typedef nne::rapidjson::Writer<nn::ec::json::NoAllocationOutputStream, nne::rapidjson::UTF8<>, nne::rapidjson::UTF8<>, NotAddChunkMemoryPoolAllocator> Writer;
    Writer writer(outputStream, &memoryPoolAllocator);

    //! {"rights":[]}
    writer.StartObject();
    writer.Key("rights");
    writer.StartArray();
    for (int i = 0; i < rightDataCount; i++)
    {
        //! {"right_id":"","consumption_request_id":""}
        auto info = nn::util::Get(rightDataArray[i].value);
        writer.StartObject();
        writer.Key("right_id");
        writer.String(info.rightId.value);
        writer.Key("consumption_request_id");
        writer.String(info.consumptionRequestId.value);
        writer.EndObject();
    }
    writer.EndArray();
    writer.EndObject();

    memcpy(pOutPostData, outputStream.GetBufferAddress(), outputStream.GetFilledSize());
    NN_RESULT_SUCCESS;
}

//! 権利情報を取得します
nn::Result GetRightInfo(int* pOutRightInfoCount, nn::ec::ConsumableServiceItemRightDataImpl outRightInfoArray[], int maxRightInfoCount, nn::ec::ShopServiceAccessor::AsyncResponse* pAsyncResponse, nn::mem::StandardAllocator* pAllocator) NN_NOEXCEPT
{
    //! レスポンスサイズを取得する
    size_t responseSize;
    NN_RESULT_DO(pAsyncResponse->GetSize(&responseSize));

    //! レスポンス取得用のメモリを確保する
    nn::ec::AllocatedBuffer<char> pResponse(pAllocator);
    NN_RESULT_DO(pResponse.Allocate(responseSize));

    //! SAX JSON パーサが使用する文字列バッファを void* で確保する
    size_t streamBufferSize = nn::os::MemoryPageSize;
    nn::ec::AllocatedBuffer<char> streamBuffer(pAllocator);
    NN_RESULT_DO(streamBuffer.Allocate(streamBufferSize));

    //! JSON Adaptor がパス解釈するのに必要なバッファを確保する
    size_t adaptorBufferSize = nn::os::MemoryPageSize;
    nn::ec::AllocatedBuffer<char> adaptorBuffer(pAllocator);
    NN_RESULT_DO(adaptorBuffer.Allocate(adaptorBufferSize));

    //! サーバレスポンスの取得。非同期処理が終わってなければ、Get 内で Wait する
    NN_RESULT_DO(pAsyncResponse->Get(pResponse.Get(), responseSize));

    //! 文字列バッファをセットする
    nn::ec::json::MemoryInputStreamForRapidJson inputStream(streamBuffer.Get(), streamBufferSize);
    inputStream.Set(pResponse.Get(), responseSize);

    //! パス解釈用のバッファをセットする
    //! pOutRightCount にはデコードできた権利数が入り、サーバーレスポンスの total_results とは異なる
    nn::ec::json::adaptor::ConsumableServiceItemRightDataAdaptor adaptor(outRightInfoArray, pOutRightInfoCount, maxRightInfoCount, adaptorBuffer.Get(), adaptorBufferSize);

    //! RapidJson を In-situ パースモードで使用する
    //! In-situ モードでは、ランタイムで必要になるメモリは文字列バッファ ( streamBuffer ) のみである
    //! その文字列バッファは、ワークメモリから確保して JSON パーサに渡す
    NN_RESULT_DO(nn::http::json::ImportJsonByRapidJson<nn::http::json::DefaultJsonErrorMap>(adaptor, inputStream, static_cast<const Cancellable*>(nullptr)));
    NN_RESULT_DO(adaptor.GetLastResult());
    NN_RESULT_SUCCESS;
}
}

namespace nn { namespace ec {

Result BeginGetConsumableRightData(ShopServiceAccessor::AsyncResponse* pAsyncResponse, const nn::account::Uid& user, int page, int perPage, ShopServiceAccessor *pAccessor, void* pWorkMemory, const size_t workMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAsyncResponse);
    NN_SDK_REQUIRES(static_cast<bool>(user) == true);
    NN_SDK_REQUIRES(page >= 0);
    NN_SDK_REQUIRES(perPage > 0 && perPage <= ConsumableServiceItemRightDataCountMax);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);
    NN_SDK_REQUIRES_NOT_NULL(pWorkMemory);
    NN_SDK_REQUIRES(workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem);

    //! アロケータを定義する
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(pWorkMemory, workMemorySize);

    //! パスのメモリを確保する
    AllocatedBuffer<char> pPath(&allocator);
    NN_RESULT_DO(pPath.Allocate(BasePathSizeMax));

    //! パス構築
    const char* pBasePath = "/v1/rights/my?application_id={applicationId}&status=PURCHASED&page=%d&per_page=%d";
    nn::util::SNPrintf(pPath.Get(), BasePathSizeMax, pBasePath, page, perPage);
    NN_RESULT_DO(pAccessor->Request(pAsyncResponse, user, ec::ShopService::Method_Get, pPath.Get()));
    NN_RESULT_SUCCESS;
}

Result EndGetConsumableRightData(int* pOutRightDataCount, ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, ShopServiceAccessor::AsyncResponse* pAsyncResponse, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRightDataCount);
    NN_SDK_REQUIRES_NOT_NULL(outRightDataArray);
    NN_SDK_REQUIRES(maxRightDataCount > 0 && maxRightDataCount <= ConsumableServiceItemRightDataCountMax);
    NN_SDK_REQUIRES_NOT_NULL(pAsyncResponse);
    NN_SDK_REQUIRES_NOT_NULL(pWorkMemory);
    NN_SDK_REQUIRES(workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem);

    //! アロケータを定義する
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(pWorkMemory, workMemorySize);

    //! 権利情報のメモリを確保する
    AllocatedBuffer<ConsumableServiceItemRightDataImpl> pInfo(&allocator);
    NN_RESULT_DO(pInfo.Allocate(maxRightDataCount));

    //! レスポンスを Parse して、権利情報として取得します
    NN_RESULT_DO(GetRightInfo(pOutRightDataCount, pInfo.Get(), maxRightDataCount, pAsyncResponse, &allocator));

    for (int i = 0; i < *pOutRightDataCount && i < maxRightDataCount; i++)
    {
        memcpy(&outRightDataArray[i], &pInfo.Get()[i], sizeof(outRightDataArray[0]));
        auto& info = *reinterpret_cast<ConsumableServiceItemRightDataImpl*>(&outRightDataArray[i]);

        //! 取得した権利情報の status が PURCHASED になっていることをチェックする
        NN_SDK_ASSERT(info.status == ConsumableServiceItemServerRightStatus_Purchased);

        //! ConsumableServiceItemConsumptionRequestId を新規に発行する
        nn::util::GenerateUuid().ToString(info.consumptionRequestId.value, sizeof(info.consumptionRequestId.value));

        //! 権利情報をサービス提供できない状態にする
        info.isServiceProvidableOnLocal = false;
    }

    NN_RESULT_SUCCESS;
}

Result BeginProvidableRightDataInquiry(ShopServiceAccessor::AsyncResponse* pAsyncResponse, const nn::account::Uid& user, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount, ShopServiceAccessor *pAccessor, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAsyncResponse);
    NN_SDK_REQUIRES(static_cast<bool>(user) == true);
    NN_SDK_REQUIRES_NOT_NULL(rightDataArray);
    NN_SDK_REQUIRES(rightDataCount > 0 && rightDataCount <= ConsumableServiceItemRightDataCountMax);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);
    NN_SDK_REQUIRES_NOT_NULL(pWorkMemory);
    NN_SDK_REQUIRES(workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem);
    for (int i = 0; i < rightDataCount; i++)
    {
        NN_SDK_REQUIRES(rightDataArray[i].IsValid());
    }

    //! アロケータを定義する
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(pWorkMemory, workMemorySize);

    //! Base Path のサイズを求める
    const char basePath[] = "/v1/rights?right_ids=";

    //! 最大権利数分のメモリ領域を確保する
    auto pathSizeMax = GetRequiredPathSizeForMakePath(basePath, nn::ec::ConsumableServiceItemRightDataCountMax);
    AllocatedBuffer<char> pPath(&allocator);
    NN_RESULT_DO(pPath.Allocate(pathSizeMax));

    //! パス構築
    NN_RESULT_DO(MakePath(pPath.Get(), pathSizeMax, basePath, rightDataArray, rightDataCount));
    NN_RESULT_DO(pAccessor->Request(pAsyncResponse, user, ec::ShopService::Method_Get, pPath.Get()));
    NN_RESULT_SUCCESS;
}

Result EndProvidableRightDataInquiry(int* pOutProvidableRightDataCount, ConsumableServiceItemRightData outProvidableRightDataArray[], int maxRightDataCount, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount, ShopServiceAccessor::AsyncResponse* pAsyncResponse, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutProvidableRightDataCount);
    NN_SDK_REQUIRES_NOT_NULL(outProvidableRightDataArray);
    NN_SDK_REQUIRES(maxRightDataCount > 0);
    NN_SDK_REQUIRES_NOT_NULL(rightDataArray);
    NN_SDK_REQUIRES(rightDataCount > 0 && rightDataCount <= ConsumableServiceItemRightDataCountMax);
    NN_SDK_REQUIRES_NOT_NULL(pAsyncResponse);
    NN_SDK_REQUIRES_NOT_NULL(pWorkMemory);
    NN_SDK_REQUIRES(workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem);

    //! アロケータを定義する
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(pWorkMemory, workMemorySize);

    //! 権利情報のメモリを確保する
    int infoCount = 0;
    AllocatedBuffer<ConsumableServiceItemRightDataImpl> pInfo(&allocator);
    NN_RESULT_DO(pInfo.Allocate(maxRightDataCount));

    //! 権利情報を取得する
    NN_RESULT_DO(GetRightInfo(&infoCount, pInfo.Get(), maxRightDataCount, pAsyncResponse, &allocator));

    //! 提供可能判定処理
    int provideCount = 0;
    for (int i = 0; i < rightDataCount && i < maxRightDataCount; i++)
    {
        auto& target = *reinterpret_cast<const ConsumableServiceItemRightDataImpl*>(rightDataArray + i);

        //! Right Id が等しい権利情報を探索する
        auto endInfo = pInfo.Get() + infoCount;
        auto atInfo = std::find_if(pInfo.Get(), endInfo, [&target](const ConsumableServiceItemRightDataImpl& info) NN_NOEXCEPT
        {
            return (nn::util::Strncmp(target.rightId.value, info.rightId.value, sizeof(info.rightId.value)) == 0);
        });

        if (atInfo == endInfo)
        {
            //! 該当する権利情報がサーバーに存在しない
            continue;
        }

        if (nn::util::Strncmp(target.consumptionRequestId.value, atInfo->consumptionRequestId.value, sizeof(atInfo->consumptionRequestId.value)) != 0)
        {
            //! サーバー上の権利が、他の Consumption Request Id を使って消費されている
            //! status が PURCHASED の場合、Consumption Request Id が "" なので、このチェックで弾かれる
            continue;
        }

        //! 権利情報をサービス提供できる状態にする
        auto& dest = *reinterpret_cast<ConsumableServiceItemRightDataImpl*>(&outProvidableRightDataArray[provideCount]);
        std::memcpy(&dest, &target, sizeof(dest));
        dest.isServiceProvidableOnLocal = true;
        ++provideCount;
    }

    *pOutProvidableRightDataCount = provideCount;
    NN_RESULT_SUCCESS;
}

Result BeginConsumeRightData(ShopServiceAccessor::AsyncResponse* pAsyncResponse, const nn::account::Uid& user, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount, ShopServiceAccessor *pAccessor, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAsyncResponse);
    NN_SDK_REQUIRES(static_cast<bool>(user) == true);
    NN_SDK_REQUIRES_NOT_NULL(rightDataArray);
    NN_SDK_REQUIRES(rightDataCount > 0 && rightDataCount <= ConsumableServiceItemRightDataCountMax);
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);
    NN_SDK_REQUIRES_NOT_NULL(pWorkMemory);
    NN_SDK_REQUIRES(workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem);
    for (int i = 0; i < rightDataCount; i++)
    {
        NN_SDK_REQUIRES(rightDataArray[i].IsValid());
    }

    //! Path にはアロケータを使用しない
    const char* pPath = "/v1/rights/consume";

    //! アロケータを定義する
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(pWorkMemory, workMemorySize);

    //! 権利情報のメモリを確保する
    int dataCount = 0;
    AllocatedBuffer<ConsumableServiceItemRightData> pData(&allocator);
    NN_RESULT_DO(pData.Allocate(rightDataCount));

    //! 権利消費を実行する権利情報をまとめる
    for (int i = 0; i < rightDataCount; i++)
    {
        if (rightDataArray[i].IsServiceProvidableOnLocal())
        {
            //! 既に提供可能な権利情報には、権利消費を実行しない
            continue;
        }

        pData.Get()[dataCount] = rightDataArray[i];
        dataCount++;
    }

    //! POST データのメモリを確保する
    //! ASIS: ID 1 組あたり 128 Byte 確保すれば確実に格納できる
    size_t idUnitSize = 128;
    size_t postDataBodySize = sizeof("{\"rights\": []}") + idUnitSize * dataCount;
    AllocatedBuffer<char> pPostDataBody(&allocator);
    NN_RESULT_DO(pPostDataBody.Allocate(postDataBodySize));

    //! POST データ構築
    NN_RESULT_DO(MakePostData(pPostDataBody.Get(), postDataBodySize, pData.Get(), dataCount, &allocator));
    nn::ec::ShopServiceAccessor::PostData postData = { pPostDataBody.Get(), static_cast<size_t>(nn::util::Strnlen(pPostDataBody.Get(), postDataBodySize)) };
    NN_RESULT_DO(pAccessor->Request(pAsyncResponse, user, ec::ShopService::Method_Post, pPath, postData));
    NN_RESULT_SUCCESS;
}

Result EndConsumeRightData(int* pOutConsumedRightDataCount, ConsumableServiceItemRightData outConsumedRightDataArray[], int maxRightDataCount, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount, ShopServiceAccessor::AsyncResponse* pAsyncResponse, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutConsumedRightDataCount);
    NN_SDK_REQUIRES_NOT_NULL(outConsumedRightDataArray);
    NN_SDK_REQUIRES(maxRightDataCount > 0);
    NN_SDK_REQUIRES_NOT_NULL(rightDataArray);
    NN_SDK_REQUIRES(rightDataCount > 0 && rightDataCount <= ConsumableServiceItemRightDataCountMax);
    NN_SDK_REQUIRES_NOT_NULL(pAsyncResponse);
    NN_SDK_REQUIRES_NOT_NULL(pWorkMemory);
    NN_SDK_REQUIRES(workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem);

    //! ワークメモリを使用しない
    NN_UNUSED(pWorkMemory);
    NN_UNUSED(workMemorySize);

    //! レスポンスサイズを取得する (この処理によって成否を判定する)
    size_t responseSize;
    NN_RESULT_DO(pAsyncResponse->GetSize(&responseSize));

    //! 消費に成功した場合、全ての権利情報を提供可能にする
    for (int i = 0; i < rightDataCount && i < maxRightDataCount; i++)
    {
        memcpy(&outConsumedRightDataArray[i], &rightDataArray[i], sizeof(outConsumedRightDataArray[0]));
        auto& target = *reinterpret_cast<ConsumableServiceItemRightDataImpl*>(&outConsumedRightDataArray[i]);

        //! 権利情報をサービス提供できる状態にする
        target.isServiceProvidableOnLocal = true;
    }

    *pOutConsumedRightDataCount = rightDataCount;
    NN_RESULT_SUCCESS;
}

}}
