﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/ec/ec_ConsumableServiceItemApi.h>
#include <nn/ec/ec_ConsumableServiceItemTypes.h>
#include <nn/mem.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Uuid.h>

#include "ec_ConsumableServiceItemInternalTypes.h"
#include "ec_ConsumableServiceItemInternalApi.h"

#include <nn/ec/ec_ResultConsumableServiceItem.h>

namespace
{
//! 入力バッファ・バッファサイズが、ConsumableServiceItemRightData をデシリアライズするのに妥当かを検証します
nn::Result ValidateBufferForDeserializeRightData(int* pOutCount, const void* buffer, const size_t bufferSize) NN_NOEXCEPT
{
    //! @internal_details
    //!    ResultSuccess が返った場合、以下が保証されます。
    //!    - buffer のバージョンが既定のバージョンを満たす
    //!    - bufferSize で、バージョン情報・権利数を取得できる
    //!    - bufferSize で、buffer に記載されている権利数分の ConsumableServiceItemRightData を取得できる

    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize > 0);

    size_t position = 0;
    auto inBuffer = reinterpret_cast<const nn::Bit8*>(buffer);

    //! buffer の検証
    int version;

    //! bufferSize で、buffer からバージョン情報を取得できることを検証する
    NN_RESULT_THROW_UNLESS(bufferSize >= sizeof(version), nn::ec::ResultConsumableServiceItemInvalidSaveData());

    //! バージョン取得
    memcpy(&version, &inBuffer[position], sizeof(version));
    position += sizeof(version);

    //! buffer のバージョンが既定のバージョンを満たすことを検証する
    //! ASIS: 現在は、セーブデータバージョンが最新バージョンになっていることを確認するだけ
    NN_RESULT_THROW_UNLESS(version == nn::ec::ConsumableServiceItemLatestSaveDataVersion, nn::ec::ResultConsumableServiceItemInvalidSaveData());

    //! ASIS: リリース時は version == 1 でのデシリアライズしか意図しないため
    //! バージョンごとのロジック分岐を行わない

    //! bufferSize の検証
    int count;

    //! bufferSize で、buffer からバージョン情報・権利数を取得できることを検証する
    auto headerSize = sizeof(version) + sizeof(count);
    NN_RESULT_THROW_UNLESS(bufferSize >= headerSize, nn::ec::ResultConsumableServiceItemInvalidSaveData());

    //! 権利数取得
    memcpy(&count, &inBuffer[position], sizeof(count));
    position += sizeof(count);

    //! bufferSize で、buffer に記載されている権利数分の ConsumableServiceItemRightData を取得できることを検証する
    auto rightDataSize = sizeof(nn::ec::ConsumableServiceItemRightData) * count;
    NN_RESULT_THROW_UNLESS(headerSize + rightDataSize <= bufferSize, nn::ec::ResultConsumableServiceItemInvalidSaveData());

    //! 権利数出力
    *pOutCount = count;
    NN_RESULT_SUCCESS;
}
}

namespace nn { namespace ec {

bool ConsumableServiceItemAsyncRequestBase::IsRequestAvailable() const NN_NOEXCEPT
{
    //! @internal_details
    //!     非同期要求を行うために必要な、以下の条件を満たすかを判定する
    //!     - ネットワーク利用要求が提出されている
    //!     - ShopServiceAccessor が利用可能な状態である
    //!     - this->Initialize() が実行済み

    return true
        && nn::nifm::IsNetworkAvailable()
        && m_pAccessor->IsAvailable()
        && m_IsInitialized;
}

Result AsyncGetConsumableRightDataRequest::Begin(int page, int perPage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsRequestAvailable());

    NN_RESULT_DO(BeginGetConsumableRightData(
        &GetAsyncResponse(), GetUser(), page, perPage, GetAccessor(), GetWorkMemory(), GetWorkMemorySize()));

    m_Page = page;
    m_PerPage = perPage;
    m_IsRequestBegin = true;
    NN_RESULT_SUCCESS;
}

Result AsyncGetConsumableRightDataRequest::End(int* pOutRightDataCount, ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount) NN_NOEXCEPT
{
    //! 指定した権利数を、出力バッファに確保できることを担保する
    NN_SDK_REQUIRES(maxRightDataCount >= m_PerPage);

    NN_SDK_REQUIRES(IsRequestAvailable());
    NN_SDK_REQUIRES(m_IsRequestBegin, "AsyncGetConsumableRightDataRequest::Begin() not called!");

    return EndGetConsumableRightData(
        pOutRightDataCount, outRightDataArray, maxRightDataCount,
        &GetAsyncResponse(), GetWorkMemory(), GetWorkMemorySize());
}

Result AsyncGetProvidableRightDataRequest::Begin(const ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsRequestAvailable());

    NN_RESULT_DO(BeginProvidableRightDataInquiry(
        &GetAsyncResponse(), GetUser(), rightDataArray, rightDataCount, GetAccessor(), GetWorkMemory(), GetWorkMemorySize()));

    m_pRightData = rightDataArray;
    m_RightDataCount = rightDataCount;
    m_IsRequestBegin = true;
    NN_RESULT_SUCCESS;
}

Result AsyncGetProvidableRightDataRequest::End(int* pOutRightDataCount, ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount) NN_NOEXCEPT
{
    //! 指定した権利数を、出力バッファに確保できることを担保する
    NN_SDK_REQUIRES(maxRightDataCount >= m_RightDataCount);

    NN_SDK_REQUIRES(IsRequestAvailable());
    NN_SDK_REQUIRES(m_IsRequestBegin, "AsyncGetProvidableRightDataRequest::Begin() not called!");

    return EndProvidableRightDataInquiry(
        pOutRightDataCount, outRightDataArray, maxRightDataCount,
        m_pRightData, m_RightDataCount, &GetAsyncResponse(), GetWorkMemory(), GetWorkMemorySize());
}

Result AsyncConsumeRightDataRequest::Begin(const ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsRequestAvailable());

    NN_RESULT_DO(BeginConsumeRightData(
        &GetAsyncResponse(), GetUser(), rightDataArray, rightDataCount, GetAccessor(), GetWorkMemory(), GetWorkMemorySize()));

    m_pRightData = rightDataArray;
    m_RightDataCount = rightDataCount;
    m_IsRequestBegin = true;
    NN_RESULT_SUCCESS;
}

Result AsyncConsumeRightDataRequest::End(int* pOutRightDataCount, ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount) NN_NOEXCEPT
{
    //! 指定した権利数を、出力バッファに確保できることを担保する
    NN_SDK_REQUIRES(maxRightDataCount >= m_RightDataCount);

    NN_SDK_REQUIRES(IsRequestAvailable());
    NN_SDK_REQUIRES(m_IsRequestBegin, "AsyncConsumeRightDataRequest::Begin() not called!");

    return EndConsumeRightData(
        pOutRightDataCount, outRightDataArray, maxRightDataCount,
        m_pRightData, m_RightDataCount, &GetAsyncResponse(), GetWorkMemory(), GetWorkMemorySize());
}

size_t GetRequiredBufferSizeToSerializeConsumableServiceItemRightData(const ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(rightDataArray);
    NN_SDK_REQUIRES(rightDataCount > 0);
    return sizeof(ConsumableServiceItemLatestSaveDataVersion) + sizeof(rightDataCount) + sizeof(rightDataArray[0]) * rightDataCount;
}

size_t SerializeConsumableServiceItemRightData(void* buffer, size_t bufferSize, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize >= GetRequiredBufferSizeToSerializeConsumableServiceItemRightData(rightDataArray, rightDataCount));
    NN_SDK_REQUIRES_NOT_NULL(rightDataArray);
    NN_SDK_REQUIRES(rightDataCount > 0);

    //! 以後、出力バッファに権利情報が全て格納できることが担保される

    size_t position = 0;
    auto outBuffer = reinterpret_cast<Bit8*>(buffer);

    //! セーブデータバージョンの入力
    memcpy(&outBuffer[position], &ConsumableServiceItemLatestSaveDataVersion, sizeof(ConsumableServiceItemLatestSaveDataVersion));
    position += sizeof(ConsumableServiceItemLatestSaveDataVersion);

    //! 権利数の入力
    memcpy(&outBuffer[position], &rightDataCount, sizeof(rightDataCount));
    position += sizeof(rightDataCount);

    //! 権利の入力
    auto copySize = sizeof(rightDataArray[0]) * rightDataCount;
    memcpy(&outBuffer[position], rightDataArray, copySize);
    position += copySize;

    return position;
}

Result GetSerializedConsumableServiceItemRightDataInfo(SerializedConsumableServiceItemRightDataInfo* pOutInfo, const void* buffer, const size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize > 0);

    //! バッファの妥当性チェック（リザルトを返す）
    NN_RESULT_DO(ValidateBufferForDeserializeRightData(&pOutInfo->rightDataCount, buffer, bufferSize));
    NN_RESULT_SUCCESS;
}

int DeserializeConsumableServiceItemRightData(ConsumableServiceItemRightData outRightDataArray[], const int maxRightDataCount, const void* buffer, const size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outRightDataArray);
    NN_SDK_REQUIRES(maxRightDataCount > 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize > 0);

    //! バッファの妥当性チェック（事前条件）
    SerializedConsumableServiceItemRightDataInfo serializedRightDataInfo = {};
    Result result = GetSerializedConsumableServiceItemRightDataInfo(&serializedRightDataInfo, buffer, bufferSize);
    NN_SDK_REQUIRES(result.IsSuccess());

    //! Release ビルド時に、result が未使用変数になるため、コンパイラ警告を抑制する。
    NN_UNUSED(result);

    //! 上記の妥当性チェックにより
    //! bufferSize で扱える権利数 >= buffer に記載されている権利数、が保証される
    //! そのため、「maxRightDataCount >= bufferSize で扱える権利数」を行えば、「maxRightDataCount >= buffer 内の権利数」の事前条件は不要
    NN_SDK_REQUIRES(maxRightDataCount >= serializedRightDataInfo.rightDataCount);

    //! 以後、バージョン情報・権利数が読み取れること、バッファ内の権利情報が全てデシリアライズできることが担保される

    size_t position = 0;
    auto inBuffer = reinterpret_cast<const Bit8*>(buffer);

    //! セーブデータバージョンの取得
    int version;
    memcpy(&version, &inBuffer[position], sizeof(version));
    position += sizeof(version);

    //! ASIS: リリース時は version == 1 でのデシリアライズしか意図しないため
    //! デシリアライズ・ロジックの分岐を行わない

    //! 権利数の取得
    //! serializedRightDataInfo を使いまわす
    int count = serializedRightDataInfo.rightDataCount;
    position += sizeof(count);

    //! 権利の取得
    auto copySize = sizeof(ConsumableServiceItemRightData) * count;
    memcpy(outRightDataArray, &inBuffer[position], copySize);

    return count;
}

}}
