﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/socket_Api.h>
#include <nn/dns/parser.h>

//#define NN_DNSPARSER_LOG_LEVEL NN_DNSPARSER_LOG_LEVEL_HEX
#define NN_DNSPARSER_LOG_MODULE_NAME "Record" // NOLINT(preprocessor/const)
#include "dns_ParserLog.h"
#include "dns_ParserMacros.h"

extern "C"
{
#include <nnc/dns/parser.h>
};

NN_DNSPARSER_STATIC_ASSERT(sizeof(struct nndnsparserRecord) == sizeof(nn::dns::parser::Record));
NN_DNSPARSER_STATIC_ASSERT(alignof(struct nndnsparserRecord) == alignof(nn::dns::parser::Record));

namespace nn { namespace dns { namespace parser {

Record::Record() :
    m_pMessage(nullptr),
    m_DirtySize(0),
    m_Name(Label()),
    m_Type {},
    m_Class {},
    m_TimeToLive(0),
    m_Length(0),
    m_pData(nullptr)
{
};

Record::Record(const Record& rhs) :
    m_pMessage(rhs.m_pMessage),
    m_Range(rhs.m_Range),
    m_DirtySize(rhs.m_DirtySize),
    m_Name(rhs.m_Name),
    m_Type(rhs.m_Type),
    m_Class(rhs.m_Class),
    m_TimeToLive(rhs.m_TimeToLive),
    m_Length(rhs.m_Length),
    m_pData(rhs.m_pData)
{
};

Record::~Record()
{
};

Record& Record::operator=(const Record& rhs)
{
    if (this == &rhs)
    {
        goto bail;
    };

    m_pMessage = rhs.m_pMessage;
    m_Range = rhs.m_Range;
    m_DirtySize = rhs.m_DirtySize;
    m_Name = rhs.m_Name;
    m_Type = rhs.m_Type;
    m_Class = rhs.m_Class;
    m_TimeToLive = rhs.m_TimeToLive;
    m_Length = rhs.m_Length;

bail:
    return *this;
};

const Message* & Record::GetMessage() NN_NOEXCEPT
{
    return m_pMessage;
};

Label& Record::GetName() NN_NOEXCEPT
{
    return m_Name;
};

TypeConstant& Record::GetType() NN_NOEXCEPT
{
    return m_Type;
};

ClassConstant& Record::GetClass() NN_NOEXCEPT
{
    return m_Class;
};

uint32_t& Record::GetTimeToLive() NN_NOEXCEPT
{
    return m_TimeToLive;
};

uint16_t& Record::GetLength() NN_NOEXCEPT
{
    return m_Length;
};

uint8_t const *& Record::GetData() NN_NOEXCEPT
{
    return m_pData;
};

void Record::Initialize(const Message* pMessage) NN_NOEXCEPT
{
    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, pMessage) ==
                               offsetof(nn::dns::parser::Record, m_pMessage));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, range) ==
                               offsetof(nn::dns::parser::Record, m_Range));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, dirtySize) ==
                               offsetof(nn::dns::parser::Record, m_DirtySize));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, name) ==
                               offsetof(nn::dns::parser::Record, m_Name));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, type) ==
                               offsetof(nn::dns::parser::Record, m_Type));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, rclass) ==
                               offsetof(nn::dns::parser::Record, m_Class));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, timeToLive) ==
                               offsetof(nn::dns::parser::Record, m_TimeToLive));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, length) ==
                               offsetof(nn::dns::parser::Record, m_Length));

    NN_DNSPARSER_STATIC_ASSERT(offsetof(struct nndnsparserRecord, data) ==
                               offsetof(nn::dns::parser::Record, m_pData));

    memset(this, 0, sizeof(*this));
    m_pMessage = pMessage;
    m_Name.Initialize(pMessage);

    return;
};

ssize_t Record::SizeOf() const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("\n");
    ssize_t rc = -1;

    if (m_Length > static_cast<unsigned int>(RecordConstant::DataMaximumSize))
    {
        goto bail;
    };

    if (-1 == (rc = m_Name.SizeOf()))
    {
        NN_DNSPARSER_LOG_MINOR("\n");
        goto bail;
    };

    rc += sizeof(m_Type);
    rc += sizeof(m_Class);
    rc += sizeof(m_TimeToLive);
    rc += sizeof(m_Length);
    rc += m_Length;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

bool Record::operator==(const Record& that) const NN_NOEXCEPT
{
    bool rc = false;

    if (this == &that)
    {
        rc = true;
        goto bail;
    }
    else if (false == (m_Name == that.m_Name))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Type != that.m_Type)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Class != that.m_Class)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_TimeToLive != that.m_TimeToLive)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_Length != that.m_Length)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        goto bail;
    }
    else if (m_pData == that.m_pData)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = true;
        goto bail;
    };

    switch (m_Type)
    {
        // in this case the data is a label that contains the real hostname
        case TypeConstant::CanonicalName:
        case TypeConstant::Pointer:
        {
            Label label1;
            Label label2;

            label1.Initialize(m_pMessage);
            label2.Initialize(that.m_pMessage);

            if (-1 == label1.FromBuffer(m_pData, m_Length))
            {
                NN_DNSPARSER_LOG_DEBUG("\n");
                goto bail;
            };

            if (-1 == label2.FromBuffer(that.m_pData, that.m_Length))
            {
                NN_DNSPARSER_LOG_DEBUG("\n");
                goto bail;
            };

            rc = label1 == label2;
        };
        break;
        // this is not the most correct way to compare record data, but
        // it is sufficient for the current scope of the DnsParser library.
        case TypeConstant::Address:
        default:
        {
            if (0 == memcmp(m_pData, that.m_pData, m_Length))
            {
                rc = true;
            }
            else
            {
                NN_DNSPARSER_LOG_DEBUG("\n");
            };
        };
    };

bail:
    return rc;
};

ssize_t Record::FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zu\n", pBuffer, size);

    ssize_t rc = -1;
    auto pCursor = pBuffer;

    NN_DNSPARSER_ERROR_IF(nullptr == pCursor, bail);

    m_Range.pStart = pCursor;

    if (-1 == (rc = m_Name.FromBuffer(pCursor, size)))
    {
        goto bail;
    };
    size -= rc;
    pCursor += rc;

    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_Type,       pCursor, size, nn::socket::InetNtohs, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_Class,      pCursor, size, nn::socket::InetNtohs, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint32_t, m_TimeToLive, pCursor, size, nn::socket::InetNtohl, bail);
    NN_DNSPARSER_FROMBUFFER_ERROR(rc, uint16_t, m_Length,     pCursor, size, nn::socket::InetNtohs, bail);

    if (m_Length > static_cast<unsigned int>(RecordConstant::DataMaximumSize))
    {
        rc = -1;
    }
    else if (size < m_Length)
    {
        goto bail;
    };

    m_pData = pCursor;
    pCursor += m_Length;
    rc += m_Length;

    m_Range.pEnd = pCursor;
    rc = m_DirtySize = pCursor - pBuffer;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

ssize_t Record::ToBuffer(uint8_t * const pBuffer, size_t size) const NN_NOEXCEPT
{
    NN_DNSPARSER_LOG_DEBUG("pBuffer: %p, size: %zu\n", pBuffer, size);

    ssize_t rc = -1;
    uint8_t* pCursor = pBuffer;

    NN_DNSPARSER_ERROR_IF(nullptr == pCursor, bail);

    if (SizeOf() > size)
    {
        goto bail;
    }
    else if (-1 == ( rc = m_Name.ToBuffer(pCursor, size)))
    {
        goto bail;
    };
    size -= rc;
    pCursor += rc;

    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetNtohs, uint16_t, m_Type,       bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetNtohs, uint16_t, m_Class,      bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetNtohl, uint32_t, m_TimeToLive, bail);
    NN_DNSPARSER_TOBUFFER_ERROR(rc, pCursor, size, nn::socket::InetNtohs, uint16_t, m_Length,     bail);

    if (m_Length > static_cast<unsigned int>(RecordConstant::DataMaximumSize))
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = -1;
        goto bail;
    }
    else if ( m_Length > size )
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        rc = -1;
        goto bail;
    }
    else if (nullptr == m_pData)
    {
        NN_DNSPARSER_LOG_DEBUG("\n");
        NN_SDK_ASSERT(false);
        rc = -1;
        goto bail;
    };

    memcpy(pCursor, m_pData, m_Length);
    pCursor += m_Length;

    rc = pCursor - pBuffer;

bail:
    NN_DNSPARSER_LOG_DEBUG("returning: %zd\n", rc);
    return rc;
};

}}}; //nn::dns::parser
