﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <stdint.h>

/**
 * @file
 *
 * @details
 * The use of macros are so that certain functions are compiled out
 * at runtime. The logger has been designed in such a way that it
 * can be used with the telemetry API.
 */

namespace nn { namespace dns { namespace parser {

/**
 * @brief This preprocessor macro is used as a log level constant
 * for major errors.
 */
#define NN_DNSPARSER_LOG_LEVEL_MAJOR 0 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used as a log level constant
 * for minor errors.
 */
#define NN_DNSPARSER_LOG_LEVEL_MINOR 1 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used as a log level constant
 * for informational messages.
 */
#define NN_DNSPARSER_LOG_LEVEL_INFO  2 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used as a log level constant
 * for debug messages.
 */
#define NN_DNSPARSER_LOG_LEVEL_DEBUG 3 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used as a log level constant
 * for for debug messages that include a hex string
 */
#define NN_DNSPARSER_LOG_LEVEL_HEX   4 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used to determine the maximum
 * log level; it is currently synonymous with @ref NN_DNSPARSER_LOG_LEVEL
 * @ref NN_DNSPARSER_LOG_LEVEL_HEX
 */
#define NN_DNSPARSER_LOG_LEVEL_MAX   NN_DNSPARSER_LOG_LEVEL_HEX // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor definition is used to determine which
 * messages will be logged.
 *
 * @details LOG_LEVEL defines the highest level that is logged. If it
 * is not already defined when this file is included then it defaults
 * to LOG_LEVEL_INFO.
 */
#ifndef NN_DNSPARSER_LOG_LEVEL
#define NN_DNSPARSER_LOG_LEVEL NN_DNSPARSER_LOG_LEVEL_INFO // NOLINT(preprocessor/const)
#endif

/**
 * @brief A module is composed of one or more files; it is a logical
 * definition.
 */
#ifndef NN_DNSPARSER_LOG_MODULE_NAME
#define NN_DNSPARSER_LOG_MODULE_NAME "undefined" // NOLINT(preprocessor/const)
#endif

/**
 * @brief This function is a printf-style log with an optional hexdump.
 *
 * @param[in] hexbuffer A pointer to a buffer that contains bytes to
 * print. If this parameter is nullptr then it will not print bytes.
 *
 * @param[in] hexsize The size of the hexbuffer in bytes.
 *
 * @param[in] module The module name.
 *
 * @param[in] line The line number.
 *
 * @param[in] function The function name.
 *
 * @param[in] level The level number.
 *
 * @param[in] format The format string.
 */
void Log(const void* hexbuffer,
         unsigned int hexsize,
         const char* module,
         unsigned int line,
         const char* function,
         unsigned int level,
         const char* format,
         ...) NN_NOEXCEPT;

}}}; //nn::dns::parser

#define NN_DNSPARSER_LOG(level, format,  ...)                           \
    do                                                                  \
    {                                                                   \
        nn::dns::parser::Log(NULL,                                      \
                             0,                                         \
                             NN_DNSPARSER_LOG_MODULE_NAME,              \
                             __LINE__,                                  \
                             __FUNCTION__,                              \
                             level,                                     \
                             format,                                    \
                             ##__VA_ARGS__);                            \
    } while (NN_STATIC_CONDITION(false))

/**
 * @brief Define the Major Error Dnsparser Log facility
 *
 * @details
 * When enabled this block of code is used to log major errors. It
 * prints to the console and is designed to be used with the telemetry
 * API.
 */
#if defined NN_DNSPARSER_LOG_MAJOR
#undef NN_DNSPARSER_LOG_MAJOR
#endif
#if NN_DNSPARSER_LOG_LEVEL >= NN_DNSPARSER_LOG_LEVEL_MAJOR
#define NN_DNSPARSER_LOG_MAJOR(format, ...) NN_DNSPARSER_LOG( \
        NN_DNSPARSER_LOG_LEVEL_MAJOR, format, ##__VA_ARGS__)
#else
#define NN_DNSPARSER_LOG_MAJOR(ignore, ...)
#endif

/**
 * @brief Define the Minor Error Dnsparser Log facility
 *
 * @details
 * When enabled this block of code is used to log major errors to the
 * console. It is designed to be used with the telemetry API.
 */
#if defined NN_DNSPARSER_LOG_MINOR
#undef NN_DNSPARSER_LOG_MINOR
#endif
#if NN_DNSPARSER_LOG_LEVEL >= NN_DNSPARSER_LOG_LEVEL_MINOR
#define NN_DNSPARSER_LOG_MINOR(format, ...) NN_DNSPARSER_LOG(NN_DNSPARSER_LOG_LEVEL_MINOR, format, ##__VA_ARGS__)
#else
#define NN_DNSPARSER_LOG_MINOR(ignore, ...)
#endif

/**
 * @brief Define the Informational Dnsparser Log facility
 *
 * @details
 * When enabled this block of code is used to log informational
 * messages. It prints to the console and is designed to be used with
 * the telemetry API.
 */
#if defined NN_DNSPARSER_LOG_INFO
#undef NN_DNSPARSER_LOG_INFO
#endif
#if NN_DNSPARSER_LOG_LEVEL >= NN_DNSPARSER_LOG_LEVEL_INFO
#define NN_DNSPARSER_LOG_INFO(format, ...) NN_DNSPARSER_LOG(NN_DNSPARSER_LOG_LEVEL_INFO, format, ##__VA_ARGS__)
#else
#define NN_DNSPARSER_LOG_INFO(ignore, ...)
#endif

/**
 * @brief Define the Debug Dnsparser Log facility
 *
 * @details
 * When enabled this block of code is used to log debug messages. It
 * prints to the console and is designed to be used with the telemetry
 * API.
 *
 * Even though it is designed to be used with the telemetry API it
 * will only be enabled in rare cases if ever used at all because
 * debug logging is expensive.
 */
#if defined NN_DNSPARSER_LOG_DEBUG
#undef NN_DNSPARSER_LOG_DEBUG
#endif
#if NN_DNSPARSER_LOG_LEVEL >= NN_DNSPARSER_LOG_LEVEL_DEBUG
#define NN_DNSPARSER_LOG_DEBUG(format, ...) NN_DNSPARSER_LOG(NN_DNSPARSER_LOG_LEVEL_DEBUG, format, ##__VA_ARGS__)
#else
#define NN_DNSPARSER_LOG_DEBUG(ignore, ...)
#endif

/**
 * @brief Define the Hexdump Dnsparser Log facility
 *
 * @details
 * When enabled this block of code is used to log hex buffers with a string.
 * it prints to the console and is designed to be used with the
 * telemetry API.
 *
 * Even though it is designed to be used with the telemetry API it
 * will only be enabled in rare cases if ever used at all because
 * debug logging is expensive.
 */
#if defined NN_DNSPARSER_LOG_HEX
#undef NN_DNSPARSER_LOG_HEX
#endif
#if NN_DNSPARSER_LOG_LEVEL >= NN_DNSPARSER_LOG_LEVEL_HEX
#define NN_DNSPARSER_LOG_HEX(buffer, size, format,  ...)                \
    do                                                                  \
    {                                                                   \
        nn::dns::parser::Log(buffer,                                    \
                             size,                                      \
                             NN_DNSPARSER_LOG_MODULE_NAME,              \
                             __LINE__,                                  \
                             __FUNCTION__,                              \
                             NN_DNSPARSER_LOG_LEVEL_HEX,                \
                             format,                                    \
                             ##__VA_ARGS__);                            \
    } while (NN_STATIC_CONDITION(false))
#else
#define NN_DNSPARSER_LOG_HEX(ignore, ...)
#endif
