﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcps_ProtocolStack.h"
#include "dhcps_Coordinator.h"

//#define NN_DETAIL_DHCPS_LOG_LEVEL NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
#define NN_DETAIL_DHCPS_LOG_MODULE_NAME "ProtocolStack"
#include "dhcps_Log.h"

namespace nn { namespace dhcps { namespace detail {

extern Config g_Config;

namespace {

void SendDhcpPacketEvent(Coordinator* pCoordinator,
                         NetworkLayersContainer pPacket[],
                         size_t layersCount)
{
    NetworkLayersNtoh(pPacket, layersCount);

    if (nullptr == pCoordinator)
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("Coordinator is nullptr\n");
    }
    else
    {
        InternalEvent e(EventType::OnPacketRead, pPacket, layersCount);
        pCoordinator->OnEvent(e);
    };
};

}; // end anonymous namespace

const char* NetworkLayerToString(nn::dhcps::detail::NetworkLayer in)
{
    switch (in)
    {
        NN_DETAIL_DHCPS_STRINGIFY_CASE(NetworkLayer::BpfHeader);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(NetworkLayer::Ethernet);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(NetworkLayer::Internet);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(NetworkLayer::UserDatagram);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(NetworkLayer::BootProtocol);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(NetworkLayer::Icmp);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(NetworkLayer::Max);
    default:
        ;
    };

    return "Unknown NetworkLayer";
}

int NetworkLayersGetLayer(NetworkLayersContainer* pOutContainer,
                          NetworkLayer layer,
                          const NetworkLayersContainer pPacket[],
                          size_t layersCount)
{
    int rc = -1;
    if (nullptr == pOutContainer)
    {
        goto bail;
    };

    for (unsigned int idx = 0; idx < layersCount; ++idx)
    {
        if  (pPacket[idx].type == layer)
        {
            *pOutContainer = pPacket[idx];
            rc = 0;
            goto bail;
        };
    };

bail:
    return rc;
};

void NetworkLayersFlip(NetworkLayersContainer pPacket[], size_t layersCount)
{
    for (unsigned int idx=0; idx < layersCount; ++idx)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("Flipping %s(%u)\n",
                                       NetworkLayerToString(pPacket[idx].type),
                                       pPacket[idx].type);

        switch (pPacket[idx].type)
        {
        case NetworkLayer::Ethernet:
        {
            nn::socket::EtherHeader* eh = reinterpret_cast<nn::socket::EtherHeader*>(pPacket[idx].pBuffer);
            nn::socket::EtherHeader copy = *eh;
            memcpy(eh->ether_shost, copy.ether_dhost, sizeof(eh->ether_shost));
            memcpy(eh->ether_dhost, copy.ether_shost, sizeof(eh->ether_dhost));
            break;
        };
        case NetworkLayer::Internet:
        {
            nn::socket::Ip* iph = reinterpret_cast<nn::socket::Ip*>(pPacket[idx].pBuffer);
            nn::socket::InAddr tmp = iph->ip_src;
            iph->ip_src = iph->ip_dst;
            iph->ip_dst = tmp;
            break;
        };
        case NetworkLayer::UserDatagram:
        {
            nn::socket::UdpHdr* udph = reinterpret_cast<nn::socket::UdpHdr*>(pPacket[idx].pBuffer);
            uint16_t tmp = udph->uh_sport;
            udph->uh_sport = udph->uh_dport;
            udph->uh_dport = tmp;
            break;
        };
        default:
            ;
        };
    };
};

uint32_t checksum(unsigned char *buf, unsigned nbytes, uint32_t sum)
{
    unsigned int idx;

    for (idx = 0; idx < (nbytes & ~1U); idx += 2)
    {
        sum += static_cast<uint16_t>(nn::socket::InetNtohs(*((u_int16_t *)(buf + idx))));
        if (sum > 0xFFFF)
        {
            sum -= 0xFFFF;
        };
    };

    if (idx < nbytes)
    {
        sum += buf[idx] << 8;
        if (sum > 0xFFFF)
        {
            sum -= 0xFFFF;
        };
    };

    return (sum);
};

int32_t wrapsum(u_int32_t sum)
{
    sum = ~sum & 0xFFFF;
    return sum;
};

void NetworkLayersChecksum(NetworkLayersContainer pPacket[], size_t layersCount)
{
    int rc = 0;

    for (int idx = layersCount - 1; idx >= 0; --idx)
    {
        NetworkLayersContainer& container = pPacket[idx];

        switch (container.type)
        {
        case NetworkLayer::Internet:
        {
            NN_DETAIL_DHCPS_LOG_DEBUG("Calculating checksum for: %s (%u)\n",
                                           NetworkLayerToString(pPacket[idx].type),
                                           container.type);

            nn::socket::Ip* iph = reinterpret_cast<nn::socket::Ip*>(container.pBuffer);
            iph->ip_sum = 0;
            iph->ip_sum = nn::socket::InetHtons(wrapsum(checksum(container.pBuffer,
                                                                 container.size,
                                                                 0)));
            break;
        };
        case NetworkLayer::UserDatagram:
        {
            NN_DETAIL_DHCPS_LOG_DEBUG("Calculating checksum for: %s(%u)\n",
                                           NetworkLayerToString(pPacket[idx].type),
                                           pPacket[idx].type);

            nn::socket::UdpHdr* udph = reinterpret_cast<nn::socket::UdpHdr*>(pPacket[idx].pBuffer);
            nn::socket::Ip* iph = nullptr;
            uint8_t* data = nullptr;
            size_t dataSize = 0;
            uint16_t ipsum, datasum, udpsum;
            if (NetworkLayer::Internet != pPacket[idx - 1].type )
            {
                NN_DETAIL_DHCPS_LOG_MAJOR("\n");
                rc = -1;
                goto bail;
            }
            else if (idx == layersCount - 1)
            {
                NN_DETAIL_DHCPS_LOG_MAJOR("\n");
                rc = -1;
                goto bail;
            };

            iph = reinterpret_cast<nn::socket::Ip*>(pPacket[idx - 1].pBuffer);
            data = pPacket[idx + 1].pBuffer;
            dataSize = pPacket[idx + 1].size;
            udph->uh_sum = 0;

            ipsum = checksum(reinterpret_cast<unsigned char *>(&iph->ip_src),
                             2 * sizeof(iph->ip_src),
                             IPPROTO_UDP + static_cast<u_int32_t>(
                                 nn::socket::InetNtohs(udph->uh_ulen)));

            datasum = checksum(data, dataSize, ipsum);

            udpsum = checksum(reinterpret_cast<unsigned char *>(udph),
                              sizeof(*udph),
                              datasum);

            udph->uh_sum = nn::socket::InetHtons(wrapsum(udpsum));
            break;
        };
        default:
            ;
        };
    };

bail:
    return;
};


size_t NetworkLayersCopy(NetworkLayersContainer pOutContainer[],
                         size_t outContainerCount,
                         uint8_t* pOutBuffer, size_t pBufferSize,
                         const NetworkLayersContainer pContainerIn[],
                         size_t containerInCount)
{
    ssize_t rc = -1;

    if (containerInCount > outContainerCount)
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        goto bail;
    };

    // This assumes pContainerIn[0].pBuffer is the same buffer
    // throughout the entire packet; which is sufficient for our
    // use case.
    rc = 0;
    for (unsigned int idx = 0; idx < outContainerCount; ++idx)
    {
        rc += pContainerIn[idx].size;
    };

    memcpy(pOutBuffer, pContainerIn[0].pBuffer, rc);

    rc = 0;

    for (unsigned int idx = 0; idx < outContainerCount; ++idx)
    {
        pOutContainer[idx].pBuffer = pOutBuffer + rc;
        pOutContainer[idx].type = pContainerIn[idx].type;
        rc += pOutContainer[idx].size = pContainerIn[idx].size;
    };

    rc = 0;

bail:
    return rc;
};


void NetworkLayersNtoh(NetworkLayersContainer pPacket[], size_t layersCount)
{
    for (unsigned int idx=0; idx < layersCount; ++idx)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("layer: %s(%u)\n",
                                       NetworkLayerToString(pPacket[idx].type),
                                       pPacket[idx].type);

        switch (pPacket[idx].type)
        {
        case NetworkLayer::Internet:
        {
            nn::socket::Ip* iph = reinterpret_cast<nn::socket::Ip*>(pPacket[idx].pBuffer);
            iph->ip_len = nn::socket::InetNtohs(iph->ip_len);
            iph->ip_id = nn::socket::InetNtohs(iph->ip_id);
            iph->ip_off = nn::socket::InetNtohs(iph->ip_off);
            iph->ip_sum = nn::socket::InetNtohs(iph->ip_sum);
            iph->ip_src.S_addr = nn::socket::InetNtohl(iph->ip_src.S_addr);
            iph->ip_dst.S_addr = nn::socket::InetNtohl(iph->ip_dst.S_addr);
            break;
        }
        case NetworkLayer::UserDatagram:
        {
            nn::socket::UdpHdr* udph = reinterpret_cast<nn::socket::UdpHdr*>(pPacket[idx].pBuffer);
            udph->uh_sport = nn::socket::InetNtohs(udph->uh_sport);
            udph->uh_dport = nn::socket::InetNtohs(udph->uh_dport);
            udph->uh_ulen = nn::socket::InetNtohs(udph->uh_ulen);
            udph->uh_sum = nn::socket::InetNtohs(udph->uh_sum);
            break;
        }
        default:
            ;
        };
    };
};

void NetworkLayersHton(NetworkLayersContainer pPacket[], size_t layersCount)
{
    for (unsigned int idx=0; idx < layersCount; ++idx)
    {
        NN_DETAIL_DHCPS_LOG_DEBUG("layer: %s(%u)\n",
                                       NetworkLayerToString(pPacket[idx].type),
                                       pPacket[idx].type);

        switch (pPacket[idx].type)
        {
        case NetworkLayer::Internet:
        {
            nn::socket::Ip* iph = reinterpret_cast<nn::socket::Ip*>(pPacket[idx].pBuffer);
            iph->ip_len = nn::socket::InetHtons(iph->ip_len);
            iph->ip_id = nn::socket::InetHtons(iph->ip_id);
            iph->ip_off = nn::socket::InetHtons(iph->ip_off);
            iph->ip_sum = nn::socket::InetHtons(iph->ip_sum);
            iph->ip_src.S_addr = nn::socket::InetHtonl(iph->ip_src.S_addr);
            iph->ip_dst.S_addr = nn::socket::InetHtonl(iph->ip_dst.S_addr);
            break;
        }
        case NetworkLayer::UserDatagram:
        {
            nn::socket::UdpHdr* udph = reinterpret_cast<nn::socket::UdpHdr*>(pPacket[idx].pBuffer);
            udph->uh_sport = nn::socket::InetHtons(udph->uh_sport);
            udph->uh_dport = nn::socket::InetHtons(udph->uh_dport);
            udph->uh_ulen = nn::socket::InetHtons(udph->uh_ulen);
            udph->uh_sum = nn::socket::InetHtons(udph->uh_sum);
            break;
        }
        default:
            ;
        };
    };
};

/**
 * Packet Parser
 */

const char* PacketParserStateToString(PacketParser::State in)
{
    switch (in)
    {
        NN_DETAIL_DHCPS_STRINGIFY_CASE(PacketParser::State::Uninitialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(PacketParser::State::Initialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(PacketParser::State::Ready);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(PacketParser::State::OnEthernet);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(PacketParser::State::OnInternet);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(PacketParser::State::OnUserDatagram);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(PacketParser::State::OnBootProtocol);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(PacketParser::State::Error);
    default:
        ;
    };
    return "Unknown PacketParser::State";
};

PacketParser::PacketParser() :
    m_State(State::Uninitialized), m_pCoordinator(nullptr)
{

};

void PacketParser::Initialize(Coordinator* pCoordinator)
{
    if (nullptr == pCoordinator)
    {
        ChangeState(State::Error);
        goto bail;
    };

    m_pCoordinator = pCoordinator;
    ChangeState(State::Initialized);

bail:
    return;
};

void PacketParser::Finalize()
{
    ChangeState(State::Uninitialized);
};

ssize_t PacketParser::OnEthernetPacket(NetworkLayersContainer pPacket[],
                                       size_t layer,
                                       uint8_t* pBuffer,
                                       size_t left)
{
    ChangeState(State::OnEthernet);
    int rc = -1;
    nn::socket::EtherHeader* eh = reinterpret_cast<nn::socket::EtherHeader*>(pBuffer);

    if (nullptr == eh)
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (left < sizeof(*eh))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (static_cast<uint16_t>(nn::socket::EtherType::EtherType_Ip) == nn::socket::InetNtohs(eh->ether_type))
    {
        pPacket[layer].type = NetworkLayer::Ethernet;
        pPacket[layer].pBuffer = pBuffer;
        pPacket[layer].size = sizeof(*eh);

        rc = OnInternetPacket(pPacket,
                              ++layer,
                              pBuffer + sizeof(*eh),
                              left - sizeof(*eh));
        goto bail;
    }
    else
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("Ether type is not EtherType_Ip (%d) but %d\n",
                                       nn::socket::EtherType::EtherType_Ip, eh->ether_type);
    };

    // Signal error to application
    {
        InternalEvent e(EventType::OnPacketParseError, nullptr, 0);
        m_pCoordinator->OnEvent(e);
    };

bail:
    ChangeState(State::Ready);
    return rc;
};

ssize_t PacketParser::OnInternetPacket(NetworkLayersContainer pPacket[],
                                       size_t layer,
                                       uint8_t* pBuffer,
                                       size_t left)
{
    ChangeState(State::OnInternet);

    int rc = -1;
    nn::socket::Ip* iph = reinterpret_cast<nn::socket::Ip*>(pBuffer);
    size_t ihlSize;
    uint16_t ipsum;

    if (nullptr == iph)
    {
        // logic error
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
    }
    else if (left < sizeof(*iph))
    {
        // truncated packet
        NN_DETAIL_DHCPS_LOG_MINOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (sizeof(*iph) > (ihlSize = iph->ip_hl * 4))
    {
        // bad packet
        NN_DETAIL_DHCPS_LOG_MINOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (left < ihlSize)
    {
        // truncated packet
        NN_DETAIL_DHCPS_LOG_MINOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (0 != (ipsum = nn::socket::InetHtons(wrapsum(checksum(pBuffer, ihlSize, 0)))))
    {
        // packet failed checksum
        NN_DETAIL_DHCPS_LOG_MINOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (IPPROTO_UDP == iph->ip_p)
    {
        pPacket[layer].type = NetworkLayer::Internet;
        pPacket[layer].pBuffer = pBuffer;
        pPacket[layer].size = ihlSize;

        rc = OnUserDatagramPacket(pPacket,
                                  ++layer,
                                  pBuffer + ihlSize,
                                  left - ihlSize);
        goto bail;
    }
    else
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("Packet protocol is not IPPROTO_UDP(%d) but %d\n",
                                       IPPROTO_UDP, iph->ip_p);
    };

    // Signal error to application
    {
        InternalEvent e(EventType::OnPacketParseError, nullptr, 0);
        m_pCoordinator->OnEvent(e);
    };

bail:
    ChangeState(State::Ready);
    return rc;
};

ssize_t PacketParser::OnUserDatagramPacket(NetworkLayersContainer pPacket[],
                                           size_t layer,
                                           uint8_t* pBuffer,
                                           size_t left)
{
    ChangeState(State::OnUserDatagram);

    int rc = -1;
    uint16_t sport = 0;
    uint16_t dport = 0;
    uint8_t* data;
    size_t dataSize;
    uint16_t udpsum;
    nn::socket::Ip* iph = nullptr;

    nn::socket::UdpHdr* udph = reinterpret_cast<nn::socket::UdpHdr*>(pBuffer);
    uint16_t uhlen = 0;

    if (NetworkLayer::Internet != pPacket[layer - 1].type)
    {
        // can't calculate checksum
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
    }
    else if (nullptr == (iph = reinterpret_cast<nn::socket::Ip*>(pPacket[layer - 1].pBuffer)))
    {
        // logic error
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
    }
    else if (nullptr == udph)
    {
        // logic error
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
    }
    else if (left < sizeof(*udph))
    {
        // truncated packet
        NN_DETAIL_DHCPS_LOG_MINOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (left < (uhlen = nn::socket::InetNtohs(udph->uh_ulen)))
    {
        // truncated packet
        NN_DETAIL_DHCPS_LOG_MINOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (0 == (dataSize = uhlen - sizeof(*udph)))
    {
        // no data
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else if (nullptr == (data = pBuffer + sizeof(*udph)))
    {
        // logic error
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
    }
    else if (0 != (udpsum =
                   nn::socket::InetNtohs(wrapsum(
                                             // udp checksum
                                             checksum(reinterpret_cast<unsigned char *>(udph),
                                                      sizeof(*udph),
                                                      // data checksum
                                                      checksum(data,
                                                               dataSize,
                                                               // ipv4 "pseudo header"
                                                               checksum(reinterpret_cast<unsigned char *>(&iph->ip_src),
                                                                        2 * sizeof(iph->ip_src),
                                                                        IPPROTO_UDP + static_cast<u_int32_t>(uhlen))))))))
    {
        // packet failed checksum
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        rc = 0; // do not signal unrecoverable error
    }
    else
    {
        data = pBuffer + sizeof(*udph);
        dataSize = left - sizeof(*udph);

        sport = nn::socket::InetNtohs(udph->uh_sport);
        dport = nn::socket::InetNtohs(udph->uh_dport);

        if (g_Config.GetDhcpClientUdpPort() == sport
            && g_Config.GetDhcpServerUdpPort() == dport)
        {
            pPacket[layer].type = NetworkLayer::UserDatagram;
            pPacket[layer].pBuffer = pBuffer;
            pPacket[layer].size = sizeof(*udph);

            rc = OnBootProtocolRequestPacket(pPacket,
                                             ++layer,
                                             pBuffer + sizeof(*udph),
                                             uhlen - sizeof(*udph));
            goto bail;
        }
        else
        {
            NN_DETAIL_DHCPS_LOG_MAJOR("Expected source port is %d but is %d and "
                                           "expected destination port is %d but is %d\n",
                                           g_Config.GetDhcpClientUdpPort(), sport,
                                           g_Config.GetDhcpServerUdpPort(), dport);
        };
    };

    // Signal error to application
    {
        InternalEvent e(EventType::OnPacketParseError, nullptr, 0);
        m_pCoordinator->OnEvent(e);
    };

bail:
    ChangeState(State::Ready);
    return rc;
};

ssize_t PacketParser::OnBootProtocolRequestPacket(NetworkLayersContainer pPacket[],
                                                  size_t layer,
                                                  uint8_t* pBuffer,
                                                  size_t left)
{
    ChangeState(State::OnBootProtocol);

    int rc = -1;
    BootpDhcpMessage* bootph = reinterpret_cast<BootpDhcpMessage*>(pBuffer);
    uint8_t* pCursor = nullptr;
    int necessaryOptionsCount = 2;

    if (left < sizeof(BootpDhcpMessage))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("Buffer is not sufficient size to be a bootp message\n");
        rc = 0; // do not signal unrecoverable error
    }
    // the On-xxx-Packet functions are used on the read path
    // and so the only messages we ought to be reading from the Bpf
    // device are requests. responses should not be read
    else if (BootProtocolOperationCode::Request != bootph->operationCode)
    {
        NN_DETAIL_DHCPS_LOG_MINOR("Bootp packet is not BootProtocolOperationCode_Request but %s (%d).\n",
                                       BootProtocolOperationCodeToString(bootph->operationCode),
                                       bootph->operationCode);
        rc = 0;
    }
    else if (static_cast<uint32_t>(LibraryConstants::DhcpMagicCookie) !=
             nn::socket::InetNtohl(bootph->magicCookie))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("DHCP magic cookie does not match!\n");
        rc = 0;
    }
    else
    {
        for (pCursor = bootph->options; pCursor < pBuffer + left; )
        {
            DhcpOptionContainer<uint8_t[]>* pOption =
                reinterpret_cast<DhcpOptionContainer<uint8_t[]>*>(pCursor);

            if (DhcpOptionCode::DhcpMessageType == pOption->type)
            {
                --necessaryOptionsCount;
            };

            // end is always at the end of an options stream; the rest of
            // the bytes might be padding or something else and it might
            // well be that the 'something else' is another end option
            // code such as tested with windows and the dhcpinfo message
            // so we break out of the loop at the first sign of an end message
            if (DhcpOptionCode::End == pOption->type)
            {
                --necessaryOptionsCount;
                break;
            };

            pCursor += sizeof(pOption->type) + sizeof(pOption->length) + pOption->length;
        };

        if ( 0 == necessaryOptionsCount )
        {
            pPacket[layer].type = NetworkLayer::BootProtocol;
            pPacket[layer].pBuffer = pBuffer;
            pPacket[layer].size = pCursor - pBuffer;
            SendDhcpPacketEvent(m_pCoordinator, pPacket, ++layer);
            rc = 0;
            goto bail;
        }
        else
        {
            NN_DETAIL_DHCPS_LOG_MAJOR("\n");
            rc = 0;
        };
    };

    // Signal error to application
    {
        InternalEvent e(EventType::OnPacketParseError, nullptr, 0);
        m_pCoordinator->OnEvent(e);
    };

bail:
    ChangeState(State::Ready);
    return rc;
};

void PacketParser::ChangeState(State next)
{
    NN_DETAIL_DHCPS_LOG_DEBUG("Changing state from: %s (%u) to %s (%u)\n",
                                   PacketParserStateToString(m_State), m_State,
                                   PacketParserStateToString(next), next);

    SetGlobalState(GlobalState::ProtocolStack, static_cast<uint8_t>(next));

    m_State = next;
};

}}}; // nn::dhcps::detail
