﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcps_LeaseTableManager.h"
#include "dhcps_Coordinator.h"

//#define NN_DETAIL_DHCPS_LOG_LEVEL NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
#define NN_DETAIL_DHCPS_LOG_MODULE_NAME "LeaseTableManager"
#include "dhcps_Log.h"

namespace nn { namespace dhcps { namespace detail {

extern Config g_Config;

const char* ClientStateToString(ClientState in) NN_NOEXCEPT
{
    switch (in)
    {
        NN_DETAIL_DHCPS_STRINGIFY_CASE(ClientState::Reserved);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(ClientState::Unbound);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(ClientState::Offered);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(ClientState::Bound);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(ClientState::Renewing);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(ClientState::Rebinding);
    default:
        ;
    };
    return "Unknown ClientState";
};

void PrintLeaseTable(const LeaseRecord* pLeaseTable, unsigned int totalRecordCount) NN_NOEXCEPT
{
#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    static const char* header =
        "+------C Hash------+---IP Address----+----MAC Address----+------Client State------+lease+-t1--+-t2--+-pri-+\n";

    static const char* footer =
        "+------------------+-----------------+-------------------+------------------------+-----+-----+-----+-----+\n";

    NN_DETAIL_DHCPS_LOG_DEBUG("%s", header);
    for (unsigned int idx = 0; idx < totalRecordCount; ++idx)
    {
        const LeaseRecord& record = pLeaseTable[idx];
        char ipString[static_cast<size_t>(LibraryConstants::IpAddressStringBufferSize)];
        char macAddressString[static_cast<size_t>(LibraryConstants::MacAddressStringBufferSize)];
        uint32_t now = Time();

        NN_DETAIL_DHCPS_LOG_DEBUG(
            "| %16llx | %15s | %17s | %22s | %3d | %3d | %3d | %3d |\n",
            record.client.hash,
            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &record.client.ipAddress, ipString, sizeof(ipString)),
            MacAddressToString(macAddressString, sizeof(macAddressString), record.client.macAddress),
            ClientStateToString(record.state),
            RelativeTime(now, record.lease),
            RelativeTime(now, record.t1),
            RelativeTime(now, record.t2),
            record.priority);
    };
    NN_DETAIL_DHCPS_LOG_DEBUG("%s", footer);
#endif
};

/**
 * @brief Sort the table by priority.
 *
 *
 * @return The number of table record swaps.
 */
unsigned SortTableByPriority(LeaseRecord* pLeaseTable, unsigned int totalRecordCount) NN_NOEXCEPT
{
    unsigned changes = 0;
    unsigned int idx, jdx;

    for (idx = 0; idx < totalRecordCount - 1; ++idx)
    {
        unsigned int minimum = idx;

        for (jdx = idx + 1; jdx < totalRecordCount; jdx++)
        {
            if (pLeaseTable[jdx].priority < pLeaseTable[minimum].priority)
            {
                minimum = jdx;
            };
        };

        if (minimum != idx)
        {
            ++changes;
            std::swap(pLeaseTable[idx], pLeaseTable[minimum]);
        };
    };

    return changes;
};

/**
 * @brief Convert a LeaseRecord to a DhcpLease structure
 */
void LeaseRecordToDhcpLease(DhcpLease* pOutLease, const LeaseRecord& record) NN_NOEXCEPT
{
    if (nullptr == pOutLease)
    {
        NN_SDK_ASSERT(false);
        return;
    };

    pOutLease->client.hash = record.client.hash;
    pOutLease->client.ipAddress = record.client.ipAddress;
    memcpy(pOutLease->client.macAddress, record.client.macAddress, sizeof(pOutLease->client.macAddress));
    pOutLease->state = static_cast<nn::dhcps::State>(record.state);
    pOutLease->lease = record.lease;
    pOutLease->t1 = record.t1;
    pOutLease->t2 = record.t2;
};

void ZeroRecord(LeaseRecord* pRecord) NN_NOEXCEPT
{
    if (nullptr == pRecord)
    {
        NN_SDK_ASSERT(false);
        return;
    };

    pRecord->state = ClientState::Unbound;
    pRecord->client.hash = 0;
    memset(pRecord->client.macAddress, 0, sizeof(pRecord->client.macAddress));
    pRecord->lease = pRecord->t1 = pRecord->t2 = 0;
};

const char* LeaseTableManagerStateToString(LeaseTableManager::State in) NN_NOEXCEPT
{
    switch (in)
    {
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LeaseTableManager::State::Uninitialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LeaseTableManager::State::Initialized);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LeaseTableManager::State::Available);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LeaseTableManager::State::Full);
        NN_DETAIL_DHCPS_STRINGIFY_CASE(LeaseTableManager::State::Error);
    default:
        ;
    };
    return "Unknown LeaseTableManager state";
};

LeaseTableManager::LeaseTableManager() NN_NOEXCEPT :
m_State(State::Uninitialized),
    m_pCoordinator(nullptr),
    m_ActionCount(0)
{

};

void LeaseTableManager::ChangeState(State in) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("Changing state from: %s (%d) to %s (%d)\n",
                                   LeaseTableManagerStateToString(m_State), m_State,
                                   LeaseTableManagerStateToString(in), in);

    SetGlobalState(GlobalState::LeaseTableManager, static_cast<uint8_t>(in));

    m_State = in;
};

int LeaseTableManager::Initialize(Coordinator* pCoordinator) NN_NOEXCEPT
{
    int rc = -1;
    unsigned int index = 0;
    uint32_t rangeBegin;
    uint32_t rangeEnd;
    uint32_t current;

    if (m_State != State::Uninitialized )
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        ChangeState(State::Error);
        goto bail;
    }
    else if (0 == GetTotalRecordCount())
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        ChangeState(State::Error);
        goto bail;
    }
    else if (g_Config.GetLeaseTableMemorySize() < sizeof(LeaseRecord) * m_TotalRecordCount)
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        ChangeState(State::Error);
        goto bail;
    }
    else if (nullptr == (m_pLeaseTable = reinterpret_cast<LeaseRecord*>(g_Config.GetLeaseTableMemory())))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        ChangeState(State::Error);
        goto bail;
    }
    else if ( nullptr == (m_pCoordinator = pCoordinator))
    {
        NN_DETAIL_DHCPS_LOG_MAJOR("\n");
        ChangeState(State::Error);
        goto bail;
    };

    current = rangeBegin = nn::socket::InetHtonl(g_Config.GetRangeBegin().S_addr);
    rangeEnd = nn::socket::InetHtonl(g_Config.GetRangeEnd().S_addr);

    for (index = 0; current <= rangeEnd; ++current)
    {
        LeaseRecord & record = m_pLeaseTable[index];
        ZeroRecord(&record);
        record.client.ipAddress.S_addr = static_cast<nn::socket::InAddrT>(nn::socket::InetHtonl(current) );
        record.priority = 0;

        if (g_Config.GetIpAddress().S_addr == m_pLeaseTable[index].client.ipAddress.S_addr)
        {
            record.state = ClientState::Reserved;
        }
        else
        {
            index += 1;
        };
    };

    PrintLeaseTable(m_pLeaseTable, m_TotalRecordCount);
    ChangeState(State::Initialized);
    rc = 0;

bail:
    return rc;
};

int LeaseTableManager::Finalize() NN_NOEXCEPT
{
    m_TotalRecordCount = 0;

    m_pLeaseTable = NULL;

    m_pCoordinator = nullptr;

    ChangeState(State::Uninitialized);
    return 0;
};

void LeaseTableManager::OnEvent(const InternalEvent& e) NN_NOEXCEPT
{
    switch (e.GetType())
    {
    case EventType::OnTimerExpired:
        OnTimerExpired();
        break;
    default:
        ;
    };

    return;
};

void LeaseTableManager::GetTimeout(nn::socket::TimeVal* pTv) NN_NOEXCEPT
{
    if (pTv == nullptr)
    {
        NN_SDK_ASSERT(false);
        return;
    };

    // if an event happened that might affect
    // the state of a lease record then we wake up quickly
    // so that we can receive a timeout event and process
    // lease record data
    if (0 != m_ActionCount)
    {
        pTv->tv_sec = 0;
        pTv->tv_usec = 250000;
        m_ActionCount = 0;
    }
    else
    {
        const int MaximumTimeoutSeconds = 30;
        int lowest = MaximumTimeoutSeconds;
        uint32_t now = Time();

        for (unsigned int idx = 0; idx < m_TotalRecordCount; ++idx)
        {
            LeaseRecord & record = m_pLeaseTable[idx];
            switch (record.state)
            {
            case ClientState::Bound:
                lowest = std::min<uint32_t>(lowest, RelativeTime(now, record.t1));
                break;
            case ClientState::Renewing:
                lowest = std::min<uint32_t>(lowest, RelativeTime(now, record.t2));
                break;
            case ClientState::Offered:
            case ClientState::Rebinding:
                lowest = std::min<uint32_t>(lowest, RelativeTime(now, record.lease));
                break;
            case ClientState::Reserved:
            case ClientState::Unbound:
            default:
                continue;
            };
        };

        pTv->tv_sec = lowest;
        pTv->tv_usec = 0;
    };
};


unsigned int LeaseTableManager::RunStateMachine() NN_NOEXCEPT
{
#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char macAddressString[static_cast<uint32_t>(LibraryConstants::MacAddressStringBufferSize)] = { 0 };
#endif

    int changes = 0;
    uint32_t now = Time();

    for (unsigned int idx = 0; idx < m_TotalRecordCount; ++idx)
    {
        LeaseRecord & record = m_pLeaseTable[idx];

        switch (record.state)
        {
        case ClientState::Unbound:
        case ClientState::Reserved:
            break;
        case ClientState::Offered:
            if (0 == RelativeTime(now, record.lease))
            {
                NN_DETAIL_DHCPS_LOG_DEBUG(
                    "Expiring offered lease with Hash: %llx, ip %s, and mac %s; "
                    "advancing to State::Unbound.\n",
                    record.client.hash,
                    nn::socket::InetNtoa(record.client.ipAddress),
                    MacAddressToString(macAddressString, sizeof(macAddressString), record.client.macAddress));

                DhcpClientIdentity id;
                memcpy(&id, &record.client, sizeof(id));

                InternalEvent e(static_cast<EventType>(Event::OnLeaseOfferExpired),
                        reinterpret_cast<void*>(&id), sizeof(id));

                m_pCoordinator->OnEvent(e);
                ++changes;
                ZeroRecord(&record);
            };
            break;
        case ClientState::Bound:
            if (0 == RelativeTime(now, record.t1))
            {
                NN_DETAIL_DHCPS_LOG_DEBUG(
                    "T1 timer expired for lease with Hash: %llx, ip %s, and mac address: %s."
                    "Advancing to State::Renewing.\n",
                    record.client.hash,
                    nn::socket::InetNtoa(record.client.ipAddress),
                    MacAddressToString(macAddressString, sizeof(macAddressString), record.client.macAddress));

                record.priority += 1;

                DhcpLease l;
                LeaseRecordToDhcpLease(&l, record);

                InternalEvent e(static_cast<EventType>(Event::OnLeaseRenewing),
                        reinterpret_cast<void*>(&l),
                        sizeof(l));

                m_pCoordinator->OnEvent(e);

                ++changes;
                record.state = ClientState::Renewing;
            };
            break;
        case ClientState::Renewing:
            if (0 == RelativeTime(now, record.t2))
            {
                NN_DETAIL_DHCPS_LOG_DEBUG(
                    "T2 timer expired for lease with client: %llx, ip %s, and mac address %s. "
                    "Advancing to State::Rebinding.\n",
                    record.client.hash,
                    nn::socket::InetNtoa(record.client.ipAddress),
                    MacAddressToString(macAddressString, sizeof(macAddressString), record.client.macAddress));

                record.priority += 1;

                DhcpLease l;
                LeaseRecordToDhcpLease(&l, record);

                InternalEvent e(static_cast<EventType>(Event::OnLeaseRebinding),
                        reinterpret_cast<void*>(&l),
                        sizeof(l));

                m_pCoordinator->OnEvent(e);

                ++changes;
                record.state = ClientState::Rebinding;
            };
            break;
        case ClientState::Rebinding:
            if (0 == RelativeTime(now, record.lease))
            {
                NN_DETAIL_DHCPS_LOG_DEBUG(
                    "Lease timer expired for lease with Hash: %llx, ip: %s, and mac: %s. "
                    "Advancing to State::Unbound.\n",
                    record.client.hash,
                    nn::socket::InetNtoa(record.client.ipAddress),
                    MacAddressToString(macAddressString, sizeof(macAddressString), record.client.macAddress));

                record.priority += 1;

                DhcpLease l;
                LeaseRecordToDhcpLease(&l, record);

                InternalEvent e(static_cast<EventType>(Event::OnLeaseExpired),
                        reinterpret_cast<void*>(&l),
                        sizeof(l));

                m_pCoordinator->OnEvent(e);

                ++changes;
                ZeroRecord(&record);
            };
            break;
        default:
            break;
        };
    };

    changes += SortTableByPriority(m_pLeaseTable, m_TotalRecordCount);

    if ( changes != 0 )
    {
        PrintLeaseTable(m_pLeaseTable, m_TotalRecordCount);
    };

    return changes;
};

void LeaseTableManager::OnTimerExpired() NN_NOEXCEPT
{
    if ( 0 == RunStateMachine() )
    {
        PrintLeaseTable(m_pLeaseTable, m_TotalRecordCount);
    };
};

unsigned int LeaseTableManager::GetTotalRecordCount() NN_NOEXCEPT
{
    if (0 == m_TotalRecordCount)
    {
        uint32_t begin = nn::socket::InetNtohl(g_Config.GetRangeBegin().S_addr);
        uint32_t end = nn::socket::InetNtohl(g_Config.GetRangeEnd().S_addr);
        uint32_t self = nn::socket::InetNtohl(g_Config.GetIpAddress().S_addr);
        uint32_t broadcast = nn::socket::InetNtohl(g_Config.GetBroadcast().S_addr);

        m_TotalRecordCount = end - begin;

        if ( self >= begin && self <= end )
        {
            m_TotalRecordCount -= 1;
        };

        if ( broadcast >= begin && broadcast <= end )
        {
            m_TotalRecordCount -= 1;
        };

        NN_DETAIL_DHCPS_LOG_DEBUG("Total records in lease table: %d\n", m_TotalRecordCount);
    };

    return m_TotalRecordCount;
};

int LeaseTableManager::GetRecordByClientIdentifierHash(LeaseRecord *& pOutRecord, const ClientIdentifierHash& clientHash) NN_NOEXCEPT
{
#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutRecord: %p, Hash: %llx\n", pOutRecord, clientHash);
#endif

    int rc = -1;
    m_ActionCount++;

    for (unsigned int idx = 0; idx < m_TotalRecordCount; ++idx)
    {
        LeaseRecord & record = m_pLeaseTable[idx];

        NN_DETAIL_DHCPS_LOG_DEBUG("clientHash: %llx, record.client.hash: %llx\n",
                                  clientHash, record.client.hash);
        if (record.client.hash == clientHash)
        {
            NN_DETAIL_DHCPS_LOG_DEBUG("GOT IT!\n");
            pOutRecord = &record;
            rc = 0;
            goto bail;
        };
    };

bail:
    return rc;
};

int LeaseTableManager::GetRecordByEthernetMacAddress(LeaseRecord *& pOutRecord, const EthernetMacAddress& mac) NN_NOEXCEPT
{
#if NN_DETAIL_DHCPS_LOG_LEVEL >= NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
    char macAddressString[static_cast<size_t>(LibraryConstants::MacAddressStringBufferSize)];
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutRecord: %p, mac: %s\n",
                                   pOutRecord,
                                   MacAddressToString(macAddressString, sizeof(macAddressString), mac));
#endif

    int rc = -1;
    m_ActionCount++;

    for (unsigned int idx = 0; idx < m_TotalRecordCount; ++idx)
    {
        LeaseRecord & record = m_pLeaseTable[idx];

        if (0 == memcmp(record.client.macAddress, mac, sizeof(mac)))
        {
            pOutRecord = &record;
            rc = 0;
            goto bail;
        };
    };

bail:
    return rc;
};

int LeaseTableManager::GetRecordByIpAddress(LeaseRecord *& pOutRecord,
                                            const nn::socket::InAddr& ip) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutRecord: %p, ip: %s\n", pOutRecord, nn::socket::InetNtoa(ip));
    int rc = -1;
    m_ActionCount++;

    for (unsigned int idx = 0; idx < m_TotalRecordCount; ++idx)
    {
        LeaseRecord & record = m_pLeaseTable[idx];

        if (record.client.ipAddress.S_addr == ip.S_addr)
        {
            pOutRecord = &record;
            rc = 0;
            goto bail;
        };
    };

bail:
    return rc;
};

int LeaseTableManager::AllocateRecord(LeaseRecord *& pOutRecord) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutRecord: %p\n", pOutRecord);
    int rc = -1;
    m_ActionCount++;

    for (unsigned int idx = 0; idx < m_TotalRecordCount; ++idx)
    {
        LeaseRecord & record = m_pLeaseTable[idx];

        if (record.state == ClientState::Unbound)
        {
            pOutRecord = &record;
            rc = 0;
            goto bail;
        };
    };

bail:
    if ( -1 == rc )
    {
        if (m_State != State::Full)
        {
            ChangeState(State::Full);
            InternalEvent e(static_cast<EventType>(Event::OnTableFull), nullptr, 0);
            m_pCoordinator->OnEvent(e);
        }
        else
        {
            ChangeState(State::Available);
            InternalEvent e(static_cast<EventType>(Event::OnTableAvailable), nullptr, 0);
            m_pCoordinator->OnEvent(e);
        };
    };

    return rc;
};

}}}; // nn::dhcps::detail
