﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"
#include "dhcps_ProtocolStack.h"

/**
 * @file
 *
 * @brief Header file for the Berkley Packet Filter manager.
 */

namespace nn { namespace dhcps { namespace detail {

class Coordinator;

/**
 * @brief The BpfManager class opens, manages, reads, and writes
 * packets to and from the Berkley Packet Filter device.
 */
class BpfManager : public IEventHandler
{
public:
    /**
     * @brief The states of the @ref BpfManager.
     */
    enum class State : uint8_t
    {
        Uninitialized = 0,  ///< 0
        Initialized   = 1,  ///< 1
        FileOpen      = 2,  ///< 2
        SetOptions    = 3,  ///< 3
        CheckDatalink = 4,  ///< 4
        SetFilter     = 5,  ///< 5
        Ready         = 6,  ///< 6
        FileRead      = 7,  ///< 7
        FileWrite     = 8,  ///< 8
        FileClosed    = 9,  ///< 9
        FileError     = 10, ///< 10
    };

    /**
     * @brief Constructor.
     */
    BpfManager() NN_NOEXCEPT;

    /**
     * @brief Destructor.
     */
    ~BpfManager() NN_NOEXCEPT;

    /**
     * @brief Get the file descriptor.
     *
     * @details Intended only for use with @ref nn::socket::Ioctl()
     * and mostly @ref nn::socket::Select.
     *
     * @return The file descriptor.
     */
    int GetFileDescriptor() NN_NOEXCEPT;

    /**
     * @brief Initializes the BpfManager with the provided
     * @ref Coordinator.
     *
     * @param[in] pCoordinator The provided @ref Coordinator.
     *
     * @return On success return 0, on error return -1.
     */
    int Initialize(Coordinator* pCoordinator) NN_NOEXCEPT;

    /**
     * @brief Finalizes the BpfManager.
     *
     * @return On success return 0, on error return -1.
     */
    int Finalize() NN_NOEXCEPT;

    /**
     * @brief The @ref IEventHandler event handler.
     *
     * @param[in] e The event to handle.
     */
    virtual void OnEvent(const InternalEvent& e) NN_NOEXCEPT;

    /**
     * @brief Get the desired time for the next timeout event.
     *
     * @param[out] pOutTimeval The desired time for the next timeout event.
     */
    virtual void GetTimeout(nn::socket::TimeVal* pOutTimeval) NN_NOEXCEPT;

private:

    /**
     * @brief The current state.
     */
    State m_State;

    /**
     * @brief The socket file descriptor.
     */
    int m_FileDescriptor;

    /**
     * @brief The event @ref Coordinator.
     */
    Coordinator* m_pCoordinator;

    /**
     * @brief The packet parser sends packets to upper layers and
     * lower situations.
     */
    PacketParser m_PacketParser;

    /**
     * @brief Changes the state variable.
     *
     * @param[in] next The next state to transition to.
     */
    void ChangeState(State next) NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnFileOpen events.
     */
    void OnFileOpen() NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnFileRead events.
     */
    void OnFileRead() NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnPacketWrite events.
     *
     * @param[in] pStack An array of @ref NetworkLayersContainer
     * that contain the layers of the packet.
     *
     * @param[in] count The number of layers in @a pStack.
     *
     * @details The NetworkLayersContainer must all be in a single
     * contiguous buffer.
     *
     * BPFManager writes Ethernet packets so the lowest layer must be
     * Ethernet and the upper layers must produce well formed packets.
     */
    void OnPacketWrite(const NetworkLayersContainer pStack[],
                       size_t count) NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnFileClose events.
     */
    void OnFileClose() NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnFileError events.
     */
    void OnFileError() NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnTimerExpired events.
     */
    void OnTimerExpired() NN_NOEXCEPT;
};

}}}; // nn::dhcps::detail
