﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/util/util_BitUtil.h>

#include "codec_DeviceBuffer-spec.NX.h"
#include "codec_HardwareOpusDecoderManagerImpl-spec.NX.h"

namespace nn {
namespace codec {
namespace detail {

class HardwareOpusDecoderImpl
{
    friend class HardwareOpusDecoderManagerImpl;

private:

    static const int Alignment = 64;                // 念のため
    static const int SampleRateMaximum = 48000;     // 48KHz
    static const int FrameInMilliMaximum = 20;      // 20msec
    static const int OutputBufferSizeMaximumPerChannel = SampleRateMaximum * FrameInMilliMaximum / 1000 * sizeof(int16_t);
    static const int InputBufferSizeMaximum = 1500; // (TBD) opus の仕様上 1276 で良いはず。

public:

    HardwareOpusDecoderImpl() NN_NOEXCEPT;
    ~HardwareOpusDecoderImpl() NN_NOEXCEPT;

    nn::Result Initialize(
        int sampleRate,
        int channelCount,
        void* workBufferAddress,
        size_t workBufferSize) NN_NOEXCEPT;

    nn::Result Finalize() NN_NOEXCEPT;

    nn::Result Map(
        int sampleRate,
        int channelCount,
        int totalStreamCount,
        int stereoStreamCount,
        void* virtualAddress,
        size_t size) NN_NOEXCEPT;

    nn::Result Remap() NN_NOEXCEPT;

    nn::Result Unmap() NN_NOEXCEPT;

    nn::Result DecodeInterleaved(
        int* pConsumedFrameSize,
        int* pOutputSampleCount,
        int16_t* outputBuffer,
        size_t outputBufferSize,
        const uint8_t* inputBuffer,
        size_t inputBufferSize,
        nn::TimeSpan* pPerf,
        bool isResetRequested) NN_NOEXCEPT;

    nn::Result SetContext(const uint8_t* address, size_t size) NN_NOEXCEPT;

    inline bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    inline bool IsMemoryMapped() const NN_NOEXCEPT
    {
        return m_IsMemoryMappedOnAdsp;
    }

    nn::Result InitializeForMultiStream(
        int sampleRate,
        int channelCount,
        int totalStreamCount,
        int stereoStreamCount,
        const uint8_t channelMapping[],
        void* workBufferAddress,
        size_t workBufferSize) NN_NOEXCEPT;

    nn::Result FinalizeForMultiStream() NN_NOEXCEPT;

    nn::Result DecodeInterleavedForMultiStream(
        int* pConsumedFrameSize,
        int* pOutputSampleCount,
        int16_t* outputBuffer,
        size_t outputBufferSize,
        const uint8_t* inputBuffer,
        size_t inputBufferSize,
        nn::TimeSpan* pPerf,
        bool isResetRequested) NN_NOEXCEPT;

    static size_t GetInputBufferSize(int sampleRate, int totalStreamCount, int stereoStreamCount) NN_NOEXCEPT
    {
        NN_UNUSED(sampleRate);
        NN_UNUSED(stereoStreamCount);
        return nn::util::align_up(InputBufferSizeMaximum * totalStreamCount, Alignment);
    }

    static size_t GetOutputBufferSize(int sampleRate, int channelCount) NN_NOEXCEPT
    {
        const int divisor = SampleRateMaximum / sampleRate;
        const int outputBufferSize = OutputBufferSizeMaximumPerChannel * channelCount / divisor; // For OutputBuffer to receive from ADSP
        return nn::util::align_up(outputBufferSize, Alignment);
    }

private:

    nn::Result MapMemoryFromAdsp() NN_NOEXCEPT;
    nn::Result UnmapMemoryFromAdsp() NN_NOEXCEPT;

    nn::Result Sleep() NN_NOEXCEPT;
    nn::Result Wake() NN_NOEXCEPT;

    DeviceBuffer m_InputBuffer;
    DeviceBuffer m_OutputBuffer;

    void* m_WorkBufferVirtualAddress;
    size_t m_WorkBufferSize;
    uint32_t m_WorkBufferDeviceAddress; // Device Address
    int m_SampleRate;
    int m_ChannelCount;
    int m_TotalStreamCount;
    int m_StereoStreamCount;
    bool m_IsMemoryMappedOnAdsp;
    bool m_IsInitialized;
};

}  // namespace detail
}  // namespace codec
}  // namespace nn
