﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <stdint.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/nn_TimeSpan.h>

#include <nvdc.h>
#include <hos/nvcec.h>

#include "cec_DetailLog.h"
#include "cec_Error.h"
#include "cec_Receiver.h"
#include "cec_LowLevelInterface.h"
#include "cec_DisplayPortInterface.h"

namespace nn { namespace cec { namespace detail {

typedef struct {
    int             nvMapFd;
    nvdcHandle      dcHandle;
    nvcecHandle     cecHandle;
    void*           pCecContext;
    void*           pDcContext;
    uint32_t        flags;
    int64_t         lastTick;
    int64_t         rxBaseTick;
    int64_t         txBaseTick;
} DisplayPortStateType;

namespace {

const uint32_t CecDpStateFlag_Init = (1 << 0);
const uint32_t CecDpStateFlag_Start = (1 << 1);

const int   CecDisplayPortParamReceiveFormatBufferLength = 128;

DisplayPortStateType s_CecDpState;

typedef struct {
    uint32_t    mantissa;
    uint32_t    fraction;
} CecDecimalThousandsT;

const int64_t CecDpMaxTimeValue = 999999999;
const int64_t CecDpTimeFractRes = 1000;

const int32_t CecDisplayPortParamNackOrBusy = -5;
const int32_t CecDisplayPortParamTimeout = -110;

void    DisplayPortInterfaceComputeFractTime(CecDecimalThousandsT* pTimeValue, int64_t delta) NN_NOEXCEPT
{
    int64_t     deltaUs;

    deltaUs = nn::os::ConvertToTimeSpan(nn::os::Tick(delta)).GetMicroSeconds();
    if(deltaUs > CecDpMaxTimeValue)
    {
        deltaUs = CecDpMaxTimeValue;
    }
    pTimeValue->mantissa = static_cast<uint32_t>(deltaUs / CecDpTimeFractRes);
    deltaUs -= static_cast<int64_t>(pTimeValue->mantissa) * CecDpTimeFractRes;
    pTimeValue->fraction = static_cast<int32_t>(deltaUs);
}

void    DisplayPortInterfaceCecCallback(nvcecHandle cecHandle, uint8_t* pBuffer, int length, void* pContext) NN_NOEXCEPT
{
    DisplayPortStateType*   pState;
    int64_t                 currentTick;
    CecDecimalThousandsT    fractDelta;

    currentTick = nn::os::GetSystemTick().GetInt64Value();
    pState = static_cast<DisplayPortStateType*>(pContext);
    if((pState == &s_CecDpState) && (length > 0) && (length <= CecReceiveParamMaxMessageLength))
    {
        int     idx;
        int64_t rxDelta;
        int64_t lastTrafficDelta;
        char    buffer[CecDisplayPortParamReceiveFormatBufferLength];
        int     bufferSize = CecDisplayPortParamReceiveFormatBufferLength;
        int     count;

        rxDelta = currentTick - pState->rxBaseTick;
        lastTrafficDelta = currentTick - s_CecDpState.lastTick;
        pState->rxBaseTick = currentTick;
        pState->lastTick = currentTick;
        DisplayPortInterfaceComputeFractTime(&fractDelta, lastTrafficDelta);
        count = ::std::snprintf(buffer, bufferSize, "[%08X][%6d.%03d] ",
                                (unsigned int)(currentTick & 0x00000000FFFFFFFFLL), fractDelta.mantissa, fractDelta.fraction);
        DisplayPortInterfaceComputeFractTime(&fractDelta, rxDelta);
        count += ::std::snprintf(&buffer[count], bufferSize - count, "%6d.%03d Rcv:\t",
                                 fractDelta.mantissa, fractDelta.fraction);
        for(idx=0; (idx < length) && (count < bufferSize); idx++)
        {
            count += ::std::snprintf(&buffer[count], bufferSize - count, "%02X ", pBuffer[idx]);
        }
        if(count < bufferSize)
        {
            ::std::snprintf(&buffer[count], bufferSize - count, "\n");
        }
        NN_CEC_INFO(buffer);
        CecLowLevelInterfaceCecCallback(pState->pCecContext, pBuffer, length);
    }
    else
    {
        NN_CEC_INFO("%s: buffer %X length %d pContext %X state %X\n", NN_CURRENT_FUNCTION_NAME, pBuffer, length,
                   pContext, &s_CecDpState);
    }
}

void DisplayPortInterfaceHotPlugCallback(nvdcHandle dcHandle, nvdcDisplayHandle displayHandle, void* pContext) NN_NOEXCEPT
{
    DisplayPortStateType*   pState;
    struct nvdcDisplayInfo displayInfo;
    bool isConnected;

    pState = static_cast<DisplayPortStateType*>(pContext);
    nvdcQueryDisplayInfo(dcHandle, displayHandle, &displayInfo);
    isConnected = (displayInfo.connected != 0);
    NN_CEC_INFO("%s: connected: %d\n", NN_CURRENT_FUNCTION_NAME, isConnected);
    if(s_CecDpState.flags & CecDpStateFlag_Start)
    {
        CecLowLevelInterfaceDcCallback(pState->pDcContext, isConnected);
    }
}

}

int32_t CecDisplayPortUnhookCallbacks() NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;

    if(s_CecDpState.flags & CecDpStateFlag_Init)
    {
        if(s_CecDpState.flags & CecDpStateFlag_Start)
        {
            nvcecSetEventCb(s_CecDpState.cecHandle, nullptr, nullptr);
            nvdcEventHotplug(s_CecDpState.dcHandle, nullptr, nullptr);
            rval = CecNoError;
        }
    }
    return rval;
}

int32_t CecDisplayPortGetConnectionStatus(uint32_t* pOutHpdStatus, uint32_t* pOutSinkStatus) NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;
    unsigned int    hpdStatus;
    unsigned int    sinkStatus;

    if(s_CecDpState.flags & CecDpStateFlag_Init)
    {
        rval = nvcecGetConnectionStatus(s_CecDpState.cecHandle, &hpdStatus, &sinkStatus);
    }
    if(rval == CecDisplayPortParamNoError)
    {
        *pOutHpdStatus = hpdStatus;
        *pOutSinkStatus = sinkStatus;
    }
    return rval;
}

int32_t CecDisplayPortLastTraffic(uint32_t* pDeltaUs) NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;
    int64_t currentTick;
    int64_t deltaTick;

    if(s_CecDpState.flags & CecDpStateFlag_Start)
    {
        currentTick = nn::os::GetSystemTick().GetInt64Value();
        deltaTick = currentTick - s_CecDpState.lastTick;
        *pDeltaUs = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(nn::os::Tick(deltaTick)).GetMicroSeconds());
        rval = CecNoError;
    }
    return rval;
}

int32_t CecDisplayPortInit(void* pContext) NN_NOEXCEPT
{
    int32_t rval = CecNoError;

    NN_CEC_INFO("%s: entry\n", NN_CURRENT_FUNCTION_NAME);
    if(!(s_CecDpState.flags & CecDpStateFlag_Init))
    {
        rval = CecDisplayPortInterfaceNoServiceError;
        s_CecDpState.nvMapFd = NvRm_MemmgrGetIoctlFile();
        NN_CEC_INFO("%s: map fd %d\n", NN_CURRENT_FUNCTION_NAME, s_CecDpState.nvMapFd);
        if(s_CecDpState.nvMapFd > 0)
        {
            s_CecDpState.dcHandle = nvdcOpen(s_CecDpState.nvMapFd);
            NN_CEC_INFO("%s: dc handle %X\n", NN_CURRENT_FUNCTION_NAME, s_CecDpState.dcHandle);
            if(s_CecDpState.dcHandle != NULL)
            {
                s_CecDpState.pDcContext = pContext;
                nvdcEventHotplug(s_CecDpState.dcHandle, DisplayPortInterfaceHotPlugCallback, static_cast<void*>(&s_CecDpState));
                rval = nvdcInitAsyncEvents(s_CecDpState.dcHandle);
                if(rval == CecDisplayPortParamNoError)
                {
                    s_CecDpState.flags |= CecDpStateFlag_Init;
                    rval = CecNoError;
                }
                else
                {
                    rval = CecDisplayPortInterfaceLowLevelError;
                }
            }
        }
    }
    NN_CEC_INFO("%s: leave --> %x\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
}

int32_t CecDisplayPortStart(void* pContext) NN_NOEXCEPT
{
    int32_t rval = CecNoError;

    NN_CEC_INFO("%s: entry\n", NN_CURRENT_FUNCTION_NAME);
    if((s_CecDpState.flags & CecDpStateFlag_Init) && !(s_CecDpState.flags & CecDpStateFlag_Start))
    {
        rval = CecDisplayPortInterfaceNoServiceError;
        s_CecDpState.cecHandle = nvcecOpen();
        NN_CEC_INFO("%s: cec handle %X\n", NN_CURRENT_FUNCTION_NAME, s_CecDpState.cecHandle);
        if(s_CecDpState.cecHandle != 0)
        {
            nvcecSetEventCb(s_CecDpState.cecHandle, DisplayPortInterfaceCecCallback, static_cast<void*>(&s_CecDpState));
            rval = nvcecInitAsyncEvents(s_CecDpState.cecHandle);
            if(rval == CecDisplayPortParamNoError)
            {
                s_CecDpState.pCecContext = pContext;
                s_CecDpState.flags |= CecDpStateFlag_Start;
                s_CecDpState.rxBaseTick = s_CecDpState.txBaseTick = s_CecDpState.lastTick = nn::os::GetSystemTick().GetInt64Value();
                rval = CecNoError;
            }
            else
            {
                nvcecClose(s_CecDpState.cecHandle);
                s_CecDpState.cecHandle = static_cast<nvcecHandle>(0);
            }
        }
    }
    NN_CEC_INFO("%s: leave --> %x\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
}

int32_t CecDisplayPortStop() NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;

    NN_CEC_INFO("%s: entry\n", NN_CURRENT_FUNCTION_NAME);
    if(s_CecDpState.flags & CecDpStateFlag_Start)
    {
        nvcecClose(s_CecDpState.cecHandle);
        s_CecDpState.cecHandle = static_cast<nvcecHandle>(0);
        s_CecDpState.flags &= ~CecDpStateFlag_Start;
        rval = CecNoError;
    }
    NN_CEC_INFO("%s: leave --> %x\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
}

int32_t CecDisplayPortShutdown() NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;

    if(s_CecDpState.flags & CecDpStateFlag_Init)
    {
        nvdcClose(s_CecDpState.dcHandle);
        s_CecDpState.dcHandle = nullptr;
        s_CecDpState.flags &= ~CecDpStateFlag_Init;
        rval = CecNoError;
    }
    return rval;
}

int32_t CecDisplayPortEnable(bool enable) NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;

    if(s_CecDpState.flags & CecDpStateFlag_Start)
    {
        rval = nvcecEnable(s_CecDpState.cecHandle, enable);
    }
    return rval;
}

int32_t CecDisplayPortGetPhysicalAddress(uint8_t* pPhysicalAddress) NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;

    if(s_CecDpState.flags & CecDpStateFlag_Start)
    {
        rval = nvcecGetPAddr(s_CecDpState.cecHandle, pPhysicalAddress);
    }
    return rval;
}

int32_t CecDisplayPortSetLogicalAddress(uint8_t logicalAddress) NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;

    if(s_CecDpState.flags & CecDpStateFlag_Start)
    {
        rval = nvcecSetLAddr(s_CecDpState.cecHandle, logicalAddress);
    }
    return rval;
}

int32_t CecDisplayPortReadData(uint8_t* pReadBuffer, int* pReadByteCount) NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;

    if(s_CecDpState.flags & CecDpStateFlag_Start)
    {
        rval = nvcecRead(s_CecDpState.cecHandle, pReadBuffer, pReadByteCount);
    }
    return rval;
}

int32_t CecDisplayPortTransmitData(TransmitReturnValueType* pTransmitReturnValue,
                                   uint8_t* pBuffer, uint8_t count) NN_NOEXCEPT
{
    int32_t rval = CecDisplayPortInterfaceInvalidInitError;
    int64_t currentTick;
    CecDecimalThousandsT fractDelta;

    currentTick = nn::os::GetSystemTick().GetInt64Value();
    if(s_CecDpState.flags & CecDpStateFlag_Start)
    {
        uint32_t    idx;
        int64_t     txDelta;
        int64_t     lastTrafficDelta;
        char        buffer[CecDisplayPortParamReceiveFormatBufferLength];
        char        payloadString[CecDisplayPortParamReceiveFormatBufferLength];
        int         bufferSize = CecDisplayPortParamReceiveFormatBufferLength;
        int         payloadStringSize = CecDisplayPortParamReceiveFormatBufferLength;
        int         length;

        txDelta = currentTick - s_CecDpState.txBaseTick;
        lastTrafficDelta = currentTick - s_CecDpState.lastTick;
        s_CecDpState.txBaseTick = currentTick;
        s_CecDpState.lastTick = currentTick;
        for(idx = 0, length = 0; (idx < static_cast<uint32_t>(count)) && (length < payloadStringSize); idx++)
        {
            length += ::std::snprintf(&payloadString[length], payloadStringSize - length, "%02X ", pBuffer[idx]);
        }
        NN_CEC_INFO("About to Tx:\t%s\n", payloadString);
        rval = nvcecWrite(s_CecDpState.cecHandle, pBuffer, count);
        DisplayPortInterfaceComputeFractTime(&fractDelta, lastTrafficDelta);
        length = ::std::snprintf(buffer, bufferSize, "[%08X][%6d.%03d] ",
                                 (unsigned int)(currentTick & 0x00000000FFFFFFFFLL), fractDelta.mantissa, fractDelta.fraction);
        DisplayPortInterfaceComputeFractTime(&fractDelta, txDelta);
        length += ::std::snprintf(&buffer[length], bufferSize - length, "%6d.%03d Tx:\t",
                                  fractDelta.mantissa, fractDelta.fraction);
        if(rval == CecDisplayPortParamNoError)
        {
            pTransmitReturnValue->error = CecNoError;
            pTransmitReturnValue->ackNak = TransmitAckNakValueAck;
        }
        else if(rval == CecDisplayPortParamNackOrBusy)
        {
            pTransmitReturnValue->error = CecNoError;
            pTransmitReturnValue->ackNak = TransmitAckNakValueNak;
        }
        else if(rval == CecDisplayPortParamTimeout)
        {
            pTransmitReturnValue->error = CecDisplayPortInterfaceTimeoutError;
            pTransmitReturnValue->ackNak = TransmitAckNakValueOther;
        }
        else
        {
            pTransmitReturnValue->error = CecDisplayPortInterfaceWriteError;
            pTransmitReturnValue->ackNak = TransmitAckNakValueOther;
        }
        NN_CEC_INFO("%s%s --> %d\n", buffer, payloadString, rval);
    }
    return rval;
}

}
}   // namespace cec
}   // namespace nn
