﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidWin32Hid_Private.h"
#include <nn/nn_SystemThreadDefinition.h>

namespace nn {
namespace cdhid {
namespace win32 {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result Win32HidEnum::Initialize() NN_NOEXCEPT
{
    m_Run = true;

    nn::os::CreateThread(
                        &m_Thread,
                        ThreadFunction,
                        this,
                        m_ThreadStack,
                        nn::os::StackRegionAlignment,
                        NN_SYSTEM_THREAD_PRIORITY(cdhid,Win32HidEnumThread)
                        );
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(cdhid, Win32HidEnumThread));
    nn::os::StartThread(&m_Thread);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidEnum::Finalize() NN_NOEXCEPT
{
    m_Run = false;

    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidEnum::AcquireDevice(uint32_t deviceHandle, Win32HidInterface **ppWin32HidInterface) NN_NOEXCEPT
{
    Result result;
    int index;

    result = GetInterfaceIndex(deviceHandle, &index);

    if (result.IsSuccess())
    {
        result = m_Win32HidInterface[index].ClientAcquireDevice(deviceHandle);

        if (result.IsSuccess())
        {
            *ppWin32HidInterface = &m_Win32HidInterface[index];
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidEnum::ReleaseDevice(uint32_t deviceHandle) NN_NOEXCEPT
{
    Result result;
    int index;

    result = GetInterfaceIndex(deviceHandle, &index);

    if (result.IsSuccess())
    {
        result = m_Win32HidInterface[index].ClientReleaseDevice(deviceHandle);
    }

    return result;
}

//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
void Win32HidEnum::ServiceLoop() NN_NOEXCEPT
{
    m_Hdr.Initialize();

    for (uint16_t i = 0; i < WIN32_HID_MAX_DEVICES; i++)
    {
        m_Win32HidInterface[i].Initialize(i);
    }

    while (m_Run)
    {
        EnumerateDevices();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }

    for (uint16_t i = 0; i < WIN32_HID_MAX_DEVICES; i++)
    {
        m_Win32HidInterface[i].Finalize();
    }

    m_Hdr.Finalize();
}


//////////////////////////////////////////////////////////////////////////////
void Win32HidEnum::EnumerateDevices() NN_NOEXCEPT
{
    // reset attach
    for (int i = 0; i < WIN32_HID_MAX_DEVICES; i++)
    {
        m_Win32HidInterface[i].SetIsAttached(false);
    }

    // SetupDi enumeration
    GUID hidGuid;
    HidD_GetHidGuid(&hidGuid);

    HDEVINFO deviceInfo = SetupDiGetClassDevs(&hidGuid, NULL, 0, DIGCF_DEVICEINTERFACE | DIGCF_PRESENT);;

    DWORD index = 0;

    do
    {
        SP_DEVICE_INTERFACE_DATA deviceInterfaceData;
        deviceInterfaceData.cbSize  = sizeof(SP_DEVICE_INTERFACE_DATA);

        if (SetupDiEnumDeviceInterfaces(deviceInfo, nullptr, &hidGuid, index, &deviceInterfaceData))
        {
            DWORD requiredSize = 0;

            // get required size
            SetupDiGetDeviceInterfaceDetail(deviceInfo, &deviceInterfaceData, NULL, 0, &requiredSize, NULL);

            SP_DEVICE_INTERFACE_DETAIL_DATA *pDeviceInterfaceDetailData = (SP_DEVICE_INTERFACE_DETAIL_DATA*)m_DataBuffer;
            pDeviceInterfaceDetailData->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);

            SP_DEVINFO_DATA deviceInfoData;
            deviceInfoData.cbSize = sizeof(SP_DEVINFO_DATA);

            if(SetupDiGetDeviceInterfaceDetail(deviceInfo, &deviceInterfaceData, pDeviceInterfaceDetailData, requiredSize, &requiredSize, &deviceInfoData))
            {
                if (!IsAlreadyAttached(&deviceInfoData))
                {
                    HANDLE handle = CreateFile(
                                                pDeviceInterfaceDetailData->DevicePath,
                                                GENERIC_READ | GENERIC_WRITE,
                                                FILE_SHARE_READ | FILE_SHARE_WRITE,
                                                NULL,
                                                OPEN_EXISTING,
                                                0,
                                                NULL
                                                );

                    nn::ahid::hdr::DeviceParameters hdrDeviceParameters;

                    memset(&hdrDeviceParameters, 0, sizeof(nn::ahid::hdr::DeviceParameters));

                    sprintf((char*)hdrDeviceParameters.servicePath, "hid");

                    // TODO get usage page and usage
                    //hdrDeviceParameters.usagePage   = ;
                    //hdrDeviceParameters.usageId     = ;
                    hdrDeviceParameters.busId       = nn::ahid::hdr::AhidBusIdWin32;

                    if (handle != INVALID_HANDLE_VALUE)
                    {
                        HIDD_ATTRIBUTES deviceAttributes;
                        deviceAttributes.Size = sizeof(HIDD_ATTRIBUTES);

                        if (HidD_GetAttributes(handle, &deviceAttributes))
                        {
                            hdrDeviceParameters.vendorId        = deviceAttributes.VendorID;
                            hdrDeviceParameters.productId       = deviceAttributes.ProductID;
                            hdrDeviceParameters.versionNumber   = deviceAttributes.VersionNumber;
                        }

                        memset(m_DataBuffer, 0, sizeof(m_DataBuffer));

                        if (HidD_GetManufacturerString(handle, m_DataBuffer, sizeof(m_DataBuffer)))
                        {
                            hdrDeviceParameters.manufacturer[0] = CopyWideString(reinterpret_cast<uint16_t*>(&hdrDeviceParameters.manufacturer[2]), reinterpret_cast<uint16_t*>(m_DataBuffer)) + 2;
                            hdrDeviceParameters.manufacturer[1] = nn::ahid::hdr::AhidDescriptorTypeString;
                        }

                        memset(m_DataBuffer, 0, sizeof(m_DataBuffer));

                        if (HidD_GetProductString(handle, m_DataBuffer, sizeof(m_DataBuffer)))
                        {
                            hdrDeviceParameters.product[0]      = CopyWideString(reinterpret_cast<uint16_t*>(&hdrDeviceParameters.product[2]), reinterpret_cast<uint16_t*>(m_DataBuffer)) + 2;
                            hdrDeviceParameters.product[1]      = nn::ahid::hdr::AhidDescriptorTypeString;
                        }

                        memset(m_DataBuffer, 0, sizeof(m_DataBuffer));

                        if (HidD_GetSerialNumberString(handle, m_DataBuffer, sizeof(m_DataBuffer)))
                        {
                            hdrDeviceParameters.serialNumber[0] = CopyWideString(reinterpret_cast<uint16_t*>(&hdrDeviceParameters.serialNumber[2]), reinterpret_cast<uint16_t*>(m_DataBuffer)) + 2;
                            hdrDeviceParameters.serialNumber[1] = nn::ahid::hdr::AhidDescriptorTypeString;
                        }

                        // Find a Win32HidInterface to accept this new device reference
                        for (int i = 0; i < WIN32_HID_MAX_DEVICES; i++)
                        {
                            Result result = m_Win32HidInterface[i].EnumAttachDevice(&m_Hdr, &hdrDeviceParameters);

                            if (result.IsSuccess())
                            {
                                m_Win32HidInterface[i].SetWin32Handle(handle);
                                m_Win32HidInterface[i].SetDevInst(deviceInfoData.DevInst);
                                break;
                            }
                        }
                    }
                }
            }
        }

        index++;

    } while (GetLastError() != ERROR_NO_MORE_ITEMS);

    // handle detached devices
    for (int i = 0; i < WIN32_HID_MAX_DEVICES; i++)
    {
        if (m_Win32HidInterface[i].GetDevInst() && (m_Win32HidInterface[i].GetIsAttached() == false))
        {
            m_Win32HidInterface[i].EnumDetachDevice(&m_Hdr);
            m_Win32HidInterface[i].SetDevInst(0);

            CloseHandle(m_Win32HidInterface[i].GetWin32Handle());
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
bool Win32HidEnum::IsAlreadyAttached(SP_DEVINFO_DATA* pDeviceInfoData) NN_NOEXCEPT
{
    for (int i = 0; i < WIN32_HID_MAX_DEVICES; i++)
    {
        if (m_Win32HidInterface[i].GetDevInst() == pDeviceInfoData->DevInst)
        {
            m_Win32HidInterface[i].SetIsAttached(true);
            return true;
        }
    }

    return false;
}

//////////////////////////////////////////////////////////////////////////////
uint8_t Win32HidEnum::CopyWideString(uint16_t* pDst, uint16_t* pSrc) NN_NOEXCEPT
{
    uint8_t chars = 0;

    while (*pSrc)
    {
        *pDst++ = *pSrc++;
        chars += 2;
    }

    return chars;
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidEnum::GetInterfaceIndex(int deviceHandle, int *pIndex) NN_NOEXCEPT
{
    int index = deviceHandle >> 16;

    if (index < WIN32_HID_MAX_DEVICES)
    {
        *pIndex = index;

        return ResultSuccess();
    }

    return nn::ahid::ResultInvalidHandle();
}

} // end of namespace win32
} // end of namespace cdhid
} // end of namespace nn
