﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include <nn/cdacm/cdacm.h>
#include <nn/util/util_FormatString.h>
#include "driver/cdacm_Driver.h"

namespace {
    nn::cdacm::driver::Driver s_Driver;
}


namespace nn {
namespace cdacm {

static nn::diag::LogObserverHolder g_LogObserverHolder;
static NN_USB_DMA_ALIGN char g_Buffer[4096];
static nn::os::Mutex g_Mutex(false);

bool g_LogObserverEnabled = false;


void LogObserver(const nn::diag::LogMetaData& metaInfo, const nn::diag::LogBody& logBody, void* argument)
{
    size_t bytesWritten = 0;
    uint64_t nBytesToWrite = logBody.messageBytes;
    uint64_t bytesInStage = 0;
    const char *pSrc = logBody.message;

    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    NN_UNUSED(metaInfo);
    NN_UNUSED(argument);

    do
    {
        bytesInStage = nBytesToWrite > sizeof(g_Buffer) ? sizeof(g_Buffer) : nBytesToWrite;

        std::memcpy(static_cast<void*>(g_Buffer), pSrc, bytesInStage);
        nBytesToWrite -= bytesInStage;
        pSrc += bytesInStage;

        Write(&bytesWritten, static_cast<const void *>(g_Buffer), 0, bytesInStage);
    } while (nBytesToWrite > 0);
}

Result
Initialize(nn::os::MultiWaitType **ppDeviceAvailableEvent,
    void                   *pBuffer,
    uint32_t                bufferLen) NN_NOEXCEPT
{
    return s_Driver.Initialize(ppDeviceAvailableEvent, pBuffer, bufferLen);
}

void
EnableLogObserver()
{
    nn::diag::InitializeLogObserverHolder(&g_LogObserverHolder, LogObserver, 0);
    nn::diag::RegisterLogObserver(&g_LogObserverHolder);
    g_LogObserverEnabled = true;
}

void
DisableLogObserver()
{
    g_LogObserverEnabled = false;
    nn::diag::UnregisterLogObserver(&g_LogObserverHolder);
}

Result
Finalize() NN_NOEXCEPT
{
    return s_Driver.Finalize();
}

Result
OpenHandle(nn::os::SystemEventType     **pDetachEventPtr,
    UnitProfile                  *pOutProfile,
    nn::os::MultiWaitHolderType  *holder,
    nn::cdacm::CdAcmParameters   *pParameters) NN_NOEXCEPT
{
    return s_Driver.OpenHandle(pDetachEventPtr, pOutProfile, holder, pParameters);
}

Result
CloseHandle(UnitProfile *pProfile) NN_NOEXCEPT
{
    return s_Driver.CloseHandle(pProfile);
}


Result
Read(size_t * pBytesRead, void* pInBuffer, UnitHandle handle, uint32_t length) NN_NOEXCEPT
{
    if (util::is_aligned(reinterpret_cast<uintptr_t>(pInBuffer), ::nn::usb::HwLimitDmaBufferAlignmentSize) == false)
    {
        return ResultMisalignedBuffer();
    }
    return s_Driver.Read(pBytesRead, pInBuffer, handle, length);
}

Result
Write(size_t * pBytesWritten, const void* pOutBuffer, UnitHandle handle, uint32_t length) NN_NOEXCEPT
{
    if (util::is_aligned(reinterpret_cast<uintptr_t>(pOutBuffer), ::nn::usb::HwLimitDmaBufferAlignmentSize) == false)
    {
        return ResultMisalignedBuffer();
    }
    return s_Driver.Write(pBytesWritten, pOutBuffer, handle, length);
}

Result
WriteAsync(
    uint32_t                 *pOutXferId,
    nn::os::SystemEventType **pCompletionEventPtr,
    const void               *pOutBuffer,
    UnitHandle                handle,
    uint32_t                  length) NN_NOEXCEPT
{
    if (util::is_aligned(reinterpret_cast<uintptr_t>(pOutBuffer),
        ::nn::usb::HwLimitDmaBufferAlignmentSize) == false)
    {
        return ResultMisalignedBuffer();
    }
    return s_Driver.WriteAsync(pOutXferId, pCompletionEventPtr, pOutBuffer, handle, length);
}

Result
GetWriteAsyncResult(
    uint32_t            *pOutCount,
    nn::usb::XferReport *pOutReport,
    UnitHandle           handle,
    uint32_t             count) NN_NOEXCEPT
{
    return s_Driver.GetWriteAsyncResult(pOutCount, pOutReport, handle, count);
}

Result
Print(UnitHandle handle, const char * fmt, ...) NN_NOEXCEPT
{
    int                          nBytesToWrite = 0;
    size_t                       bytesWritten;

    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    va_list args;
    va_start(args, fmt);
    nBytesToWrite = nn::util::VSNPrintf(g_Buffer, sizeof(g_Buffer), fmt, args);
    va_end(args);

    if (nBytesToWrite <= 0)
    {
        return ResultInvalidArgs();
    }

    return s_Driver.Write(&bytesWritten, static_cast<const void *>(g_Buffer), handle, nBytesToWrite);
}

}  // cdacm
}  // nn

