﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Macro.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>

#include "cal_Crc16.h"
#include "cal_KeyEncryptor.h"
#include "cal_Settings.h"

namespace nn { namespace cal {

namespace {

//!< 固有鍵による暗号化を始めたフォーマットバージョン
const uint32_t EncryptedKeyFormatVersion = 9;

//!< 秘密鍵の種類
enum class KeyType : int
{
    EccB233Device,   //!< デバイス登録用（ECC-B233）
    Rsa2048Device,   //!< デバイス登録用（RSA-2048）
    ETicket,         //!< eTicket 発行用
    Ssl,             //!< SSL クライアント用
    GameCard,        //!< ゲームカード相互認証用
    Amiibo,          //!< Amiibo 用
    AmiiboEcqvBls,   //!< Amiibo 用（ECQV-BLS）
};

::nn::Result EncryptWithUniqueKey(
        void* pOutKey,
        size_t keySize,
        void* pOutGeneration,
        size_t generationSize,
        KeyType keyType) NN_NOEXCEPT
{
    auto size = keySize + generationSize;
    ::std::unique_ptr<char []> buffer(new char[size]);
    ::std::memcpy(buffer.get(), pOutKey, keySize);
    ::std::memcpy(
            buffer.get() + keySize,
            &CalibrationInfoCommonKeyGeneration,
            generationSize);

    ::std::unique_ptr<char []> outBuffer(new char[size]);
    ::nn::spl::InitializeForManu();
    NN_UTIL_SCOPE_EXIT
    {
        ::nn::spl::Finalize();
    };

    const int keyGeneration = nn::spl::LatestKeyGeneration;
    switch(keyType)
    {
    case KeyType::EccB233Device:
        NN_RESULT_DO(::nn::spl::ReencryptDrmDeviceCertEccKey(
                    outBuffer.get(),
                    size,
                    buffer.get(),
                    size,
                    keyGeneration));
        break;
    case KeyType::Rsa2048Device:
        NN_RESULT_DO(::nn::spl::ReencryptDrmDeviceCertKey(
                    outBuffer.get(),
                    size,
                    buffer.get(),
                    size,
                    keyGeneration));
        break;
    case KeyType::ETicket:
        NN_RESULT_DO(::nn::spl::ReencryptEsDeviceKey(
                    outBuffer.get(),
                    size,
                    buffer.get(),
                    size,
                    keyGeneration));
        break;
    case KeyType::Ssl:
        NN_RESULT_DO(::nn::spl::ReencryptSslClientCertKey(
                    outBuffer.get(),
                    size,
                    buffer.get(),
                    size,
                    keyGeneration));
        break;
    case KeyType::GameCard:
        NN_RESULT_DO(::nn::spl::ReencryptGcKey(
                    outBuffer.get(),
                    size,
                    buffer.get(),
                    size,
                    keyGeneration));
        break;
    case KeyType::Amiibo:
        NN_RESULT_DO(::nn::spl::ReencryptNfpEccP160Key(
                    outBuffer.get(),
                    size,
                    buffer.get(),
                    size,
                    keyGeneration));
        break;
    case KeyType::AmiiboEcqvBls:
        NN_RESULT_DO(::nn::spl::ReencryptNfpEccBn128Key(
                    outBuffer.get(),
                    size,
                    buffer.get(),
                    size,
                    keyGeneration));
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    NN_ABORT_UNLESS_EQUAL(
            *reinterpret_cast<uint32_t*>(outBuffer.get() + keySize),
            keyGeneration);

    ::std::memcpy(pOutKey, outBuffer.get(), keySize);
    ::std::memcpy(
            pOutGeneration,
            outBuffer.get() + keySize,
            generationSize);
    NN_RESULT_SUCCESS;
}

bool IsZero(void* p, size_t size) NN_NOEXCEPT
{
    ::std::unique_ptr<char []> zeros(new char[size]);
    ::std::memset(zeros.get(), 0, size);
    return ::std::memcmp(p, zeros.get(), size) == 0;
}

bool IsEncryptableFormat(
        const CalibrationInfoHeader& header) NN_NOEXCEPT
{
    if (header.updateCount == 0)
    {
        return true;
    }

    return header.version >= EncryptedKeyFormatVersion ? true : false;
}

} // namespace

::nn::Result EncryptExtendedEccB233DeviceKey(
        CalibrationInfo* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (!IsEncryptableFormat(pOutValue->header))
    {
        NN_RESULT_SUCCESS;
    }
    auto& block = pOutValue->body.extendedEccB233DeviceKeyBlock;
    if (IsZero(&block.extendedDeviceKey, sizeof(block.extendedDeviceKey)))
    {
        NN_RESULT_SUCCESS;
    }
    if (block.encryptionKeyGeneration.IsZero())
    {
        NN_RESULT_DO(EncryptWithUniqueKey(
                    &block.extendedDeviceKey,
                    sizeof(block.extendedDeviceKey),
                    &block.encryptionKeyGeneration,
                    sizeof(block.encryptionKeyGeneration),
                    KeyType::EccB233Device));
        block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result EncryptExtendedRsa2048DeviceKey(
        CalibrationInfo* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (!IsEncryptableFormat(pOutValue->header))
    {
        NN_RESULT_SUCCESS;
    }
    auto& block = pOutValue->body.extendedRsa2048DeviceKeyBlock;
    if (IsZero(&block.extendedDeviceKey, sizeof(block.extendedDeviceKey)))
    {
        NN_RESULT_SUCCESS;
    }
    if (block.encryptionKeyGeneration.IsZero())
    {
        NN_RESULT_DO(EncryptWithUniqueKey(
                    &block.extendedDeviceKey,
                    sizeof(block.extendedDeviceKey),
                    &block.encryptionKeyGeneration,
                    sizeof(block.encryptionKeyGeneration),
                    KeyType::Rsa2048Device));
        block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result EncryptExtendedRsa2048ETicketKey(
        CalibrationInfo* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (!IsEncryptableFormat(pOutValue->header))
    {
        NN_RESULT_SUCCESS;
    }
    auto& block = pOutValue->body.extendedRsa2048ETicketKeyBlock;
    if (IsZero(&block.extendedRsa2048ETicketKey, sizeof(block.extendedRsa2048ETicketKey)))
    {
        NN_RESULT_SUCCESS;
    }
    if (block.encryptionKeyGeneration.IsZero())
    {
        NN_RESULT_DO(EncryptWithUniqueKey(
                    &block.extendedRsa2048ETicketKey,
                    sizeof(block.extendedRsa2048ETicketKey),
                    &block.encryptionKeyGeneration,
                    sizeof(block.encryptionKeyGeneration),
                    KeyType::ETicket));
        block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result EncryptExtendedSslKey(
        CalibrationInfo* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (!IsEncryptableFormat(pOutValue->header))
    {
        NN_RESULT_SUCCESS;
    }
    auto& block = pOutValue->body.extendedSslKeyBlock;
    if (IsZero(&block.extendedSslKey, sizeof(block.extendedSslKey)))
    {
        NN_RESULT_SUCCESS;
    }
    if (block.encryptionKeyGeneration.IsZero())
    {
        NN_RESULT_DO(EncryptWithUniqueKey(
                    &block.extendedSslKey,
                    sizeof(block.extendedSslKey),
                    &block.encryptionKeyGeneration,
                    sizeof(block.encryptionKeyGeneration),
                    KeyType::Ssl));
        block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result EncryptExtendedGameCardKey(
        CalibrationInfo* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (!IsEncryptableFormat(pOutValue->header))
    {
        NN_RESULT_SUCCESS;
    }
    auto& block = pOutValue->body.extendedGameCardKeyBlock;
    if (IsZero(&block.extendedGameCardKey, sizeof(block.extendedGameCardKey)))
    {
        NN_RESULT_SUCCESS;
    }
    if (block.encryptionKeyGeneration.IsZero())
    {
        NN_RESULT_DO(EncryptWithUniqueKey(
                    &block.extendedGameCardKey,
                    sizeof(block.extendedGameCardKey),
                    &block.encryptionKeyGeneration,
                    sizeof(block.encryptionKeyGeneration),
                    KeyType::GameCard));
        block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result EncryptAmiiboKey(
        CalibrationInfo* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (!IsEncryptableFormat(pOutValue->header))
    {
        NN_RESULT_SUCCESS;
    }
    auto& block = pOutValue->body.amiiboKeyBlock;
    if (IsZero(&block.amiiboKey, sizeof(block.amiiboKey)))
    {
        NN_RESULT_SUCCESS;
    }
    if (block.encryptionKeyGeneration.IsZero())
    {
        NN_RESULT_DO(EncryptWithUniqueKey(
                    &block.amiiboKey,
                    sizeof(block.amiiboKey),
                    &block.encryptionKeyGeneration,
                    sizeof(block.encryptionKeyGeneration),
                    KeyType::Amiibo));
        block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result EncryptAmiiboEcqvBlsKey(
        CalibrationInfo* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    if (!IsEncryptableFormat(pOutValue->header))
    {
        NN_RESULT_SUCCESS;
    }
    auto& block = pOutValue->body.amiiboEcqvBlsKeyBlock;
    if (IsZero(&block.amiiboEcqvBlsKey, sizeof(block.amiiboEcqvBlsKey)))
    {
        NN_RESULT_SUCCESS;
    }
    if (block.encryptionKeyGeneration.IsZero())
    {
        NN_RESULT_DO(EncryptWithUniqueKey(
                    &block.amiiboEcqvBlsKey,
                    sizeof(block.amiiboEcqvBlsKey),
                    &block.encryptionKeyGeneration,
                    sizeof(block.encryptionKeyGeneration),
                    KeyType::AmiiboEcqvBls));
        block.crc16 = GetCrc16(&block, sizeof(block) - sizeof(block.crc16));
    }
    NN_RESULT_SUCCESS;
}

}}
