﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fssystem/fs_IStorage.h>

namespace nn { namespace cal {

//!< ファイルシステム用のアロケータです。
void* AllocateForFileSystem(size_t size) NN_NOEXCEPT;

//!< ファイルシステム用のデアロケータです。
void DeallocateForFileSystem(void* addr, size_t) NN_NOEXCEPT;

//!< 生産時較正情報パーティション (RAW) のマウント名を返します。
const char* GetRawPartitionMountName() NN_NOEXCEPT;

//!< 生産時較正情報パーティション (FAT) のマウント名を返します。
const char* GetFatPartitionMountName() NN_NOEXCEPT;

//!< 生産時較正情報パーティション (FAT) のルートパス（ダミー）を返します。
const char* GetFatDummyRootPath() NN_NOEXCEPT;

//!< ファイルをコピーします。
::nn::Result CopyFile(::nn::fs::FileHandle src,
                      ::nn::fs::FileHandle dst) NN_NOEXCEPT;

//!< 新しいファイルを作成します。
::nn::Result CreateFatPartitionFile(
    ::nn::fs::FileHandle dst, void* pBuffer, size_t length) NN_NOEXCEPT;

//!< ファイルを読み込みます。
::nn::Result ReadFatPartitionFile(
    ::nn::fs::FileHandle src, void* pBuffer, size_t length) NN_NOEXCEPT;

//!< 生産時較正情報パーティション (FAT) 上のファイルを削除します。
::nn::Result DeleteFatPartitionFile(const ::std::string& value) NN_NOEXCEPT;

//!< 生産時較正情報パーティション (FAT) 上にディレクトリを作成します。
::nn::Result CreateFatPartitionDirectory(const ::std::string& value
                                         ) NN_NOEXCEPT;

//!< 生産時較正情報パーティション (FAT) 上のディレクトリを削除します。
::nn::Result DeleteFatPartitionDirectory(const ::std::string& value
                                         ) NN_NOEXCEPT;

::nn::Result IsFatPartitionDirectoryExist(
    bool& pOutValue, const ::std::string& value) NN_NOEXCEPT;

::nn::Result IsFatPartitionFileExist(
    bool& pOutValue, const ::std::string& value) NN_NOEXCEPT;

//!< デバッグモードを有効にします。
void EnableDebugMode() NN_NOEXCEPT;

//!< 専用パーティションの利用が可能か否かを表す値を返します。
bool ExistsPartition() NN_NOEXCEPT;

//!< ファイルシステムを初期化します。
void InitializeFileSystem() NN_NOEXCEPT;

//!< 生産時較正情報パーティション (RAW) を扱うクラスです。
class RawPartition final
{
    NN_DISALLOW_COPY(RawPartition);
    NN_DISALLOW_MOVE(RawPartition);

public:
    RawPartition() NN_NOEXCEPT;

    ~RawPartition() NN_NOEXCEPT;

    //!< 生産時較正情報パーティション (RAW) をマウントします。
    ::nn::Result Mount() NN_NOEXCEPT;

private:
    //!< マウントされているか否かを表す値
    bool m_IsMounted;
};

//!< 生産時較正情報パーティション (FAT) を扱うクラスです。
class FatPartition final
{
    NN_DISALLOW_COPY(FatPartition);
    NN_DISALLOW_MOVE(FatPartition);

public:
    FatPartition() NN_NOEXCEPT;

    ~FatPartition() NN_NOEXCEPT;

    //!< 生産時較正情報パーティション (FAT) をマウントします。
    ::nn::Result Mount() NN_NOEXCEPT;

private:
    //!< マウントされているか否かを表す値
    bool m_IsMounted;
};

//!< 較正情報ファイルを扱うクラスです。
class CalibrationFile final
{
    NN_DISALLOW_COPY(CalibrationFile);
    NN_DISALLOW_MOVE(CalibrationFile);

public:
    CalibrationFile() NN_NOEXCEPT;

    ~CalibrationFile() NN_NOEXCEPT;

    //!< 較正情報ファイルをオープンします。
    ::nn::Result Open() NN_NOEXCEPT;

    ::nn::Result Read(void* buffer, size_t size) NN_NOEXCEPT;

    ::nn::Result Write(const void* buffer, size_t size) NN_NOEXCEPT;

private:
    //!< ファイルがオープンされているか否かを表す値
    bool m_IsOpen;

    //!< ファイルハンドル
    ::nn::fs::FileHandle m_Handle;

    //!< ストレージ領域操作のためのインターフェース
    ::std::unique_ptr<::nn::fssystem::IStorage> m_Storage;
};

//!< ホストファイルシステム上のファイルを扱うクラスです。
class HostFsFile final
{
    NN_DISALLOW_COPY(HostFsFile);
    NN_DISALLOW_MOVE(HostFsFile);

public:
    HostFsFile() NN_NOEXCEPT;

    ~HostFsFile() NN_NOEXCEPT;

    //!< ホストファイルシステム上のファイルをオープンします。
    ::nn::Result Open(::nn::fs::FileHandle* outValue,
                      const char* path,
                      int mode) NN_NOEXCEPT;

private:
    //!< ファイルがオープンされているか否かを表す値
    bool m_IsOpen;

    //!< ファイルハンドル
    ::nn::fs::FileHandle m_Handle;
};

//!< 生産時較正情報パーティション (FAT) 上のファイルを扱うクラスです。
class FatPartitionFile final
{
    NN_DISALLOW_COPY(FatPartitionFile);
    NN_DISALLOW_MOVE(FatPartitionFile);

public:
    FatPartitionFile() NN_NOEXCEPT;

    ~FatPartitionFile() NN_NOEXCEPT;

    //!< 生産時較正情報パーティション (FAT) 上のファイルをオープンします。
    ::nn::Result Open(::nn::fs::FileHandle* outValue,
                      const char* path,
                      int mode) NN_NOEXCEPT;

private:
    //!< ファイルがオープンされているか否かを表す値
    bool m_IsOpen;

    //!< ファイルハンドル
    ::nn::fs::FileHandle m_Handle;
};

//!< 生産時較正情報パーティション (FAT) 上のディレクトリを扱うクラスです。
class FatPartitionDirectory final
{
    NN_DISALLOW_COPY(FatPartitionDirectory);
    NN_DISALLOW_MOVE(FatPartitionDirectory);

public:
    FatPartitionDirectory() NN_NOEXCEPT;

    ~FatPartitionDirectory() NN_NOEXCEPT;

    //!< 生産時較正情報パーティション (FAT) 上のディレクトリをオープンします。
    ::nn::Result Open(::nn::fs::DirectoryHandle* outValue,
                      const char* path,
                      int mode) NN_NOEXCEPT;

private:
    //!< ディレクトリがオープンされているか否かを表す値
    bool m_IsOpen;

    //!< ディレクトリハンドル
    ::nn::fs::DirectoryHandle m_Handle;
};

}}
