﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/bgtc/bgtc_Types.h>
#include <nn/bgtc/detail/bgtc_TaskImpl.h>
#include <nn/os.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <mutex>

#include "bgtc_Config.h"
#include "bgtc_LibraryMain.h"
#include "bgtc_ServiceProxy.h"


#if !NN_BGTC_ENABLE_HIPC
#include "bgtc_Instance.h"
#include "bgtc_StateControlServiceImpl.h"
#endif

#include "bgtc_Log.h"

namespace nn { namespace bgtc {

//----------------------------------------------------------------------------------
namespace
{
    class InitializeCounter
    {
    public:
        InitializeCounter()
            : m_Count(0)
        {

        }
        ~InitializeCounter()
        {
        }

        bool DoInitialize()
        {
            std::lock_guard<os::Mutex> lock(s_Mutex);

            NN_SDK_REQUIRES(m_Count >= 0);
            if (m_Count++ == 0)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        bool DoFinalize()
        {
            std::lock_guard<os::Mutex> lock(s_Mutex);

            NN_SDK_REQUIRES(m_Count >= 0);
            if (--m_Count == 0)
            {
                return true;
            }
            else
            {
                if (m_Count < 0)
                {
                    m_Count = 0;
                }
                return false;
            }
        }

        bool IsInitialized() const
        {
            return m_Count > 0;
        }

    private:
        int32_t m_Count;
        static os::Mutex s_Mutex;
    };

    os::Mutex InitializeCounter::s_Mutex(false);

    nn::os::SystemEvent s_SystemEventForState;

    InitializeCounter s_CounterForTaskControlService;
    InitializeCounter s_CounterForStateControlService;

    nn::sf::SharedPointer<IStateControlService> s_pStateControlService = nullptr;

    detail::TaskImpl s_TaskImpl;
}


Result Initialize() NN_NOEXCEPT
{
    if (s_CounterForTaskControlService.DoInitialize())
    {
        return s_TaskImpl.Initialize();
    }
    else
    {
        return ResultSuccess();
    }
}

void Finalize() NN_NOEXCEPT
{
    if (s_CounterForTaskControlService.DoFinalize())
    {
        s_TaskImpl.Finalize();
    }
}

bool IsInitialized() NN_NOEXCEPT
{
    return s_TaskImpl.IsInitialized();
}

Result NotifyTaskStarting() NN_NOEXCEPT
{
    return s_TaskImpl.NotifyStarting();
}

void NotifyTaskFinished() NN_NOEXCEPT
{
    s_TaskImpl.NotifyFinished();
}

nn::os::SystemEvent& GetTriggerEvent() NN_NOEXCEPT
{
    return s_TaskImpl.GetTriggerEvent();
}

bool IsInHalfAwake()
{
    return s_TaskImpl.IsInHalfAwake();
}

bool IsInFullAwake()
{
    return s_TaskImpl.IsInFullAwake();
}

void ScheduleTask(Interval intervalSeconds) NN_NOEXCEPT
{
    s_TaskImpl.Schedule(intervalSeconds);
}

void ScheduleTaskUnsafe(Interval intervalSeconds) NN_NOEXCEPT
{
    s_TaskImpl.ScheduleUnsafe(intervalSeconds);
}

void SchedulePeriodicTask(Interval intervalSecondsFirst, Interval intervalSecondsPeriodic) NN_NOEXCEPT
{
    s_TaskImpl.SchedulePeriodic(intervalSecondsFirst, intervalSecondsPeriodic);
}

Result GetScheduledTaskInterval(Interval* pOutSeconds) NN_NOEXCEPT
{
    return s_TaskImpl.GetScheduledInterval(pOutSeconds);
}

Result UnscheduleTask() NN_NOEXCEPT
{
    return s_TaskImpl.Unschedule();
}

nn::os::SystemEvent& GetScheduleEvent() NN_NOEXCEPT
{
    return s_TaskImpl.GetScheduleEvent();
}

OperationMode GetOperationMode() NN_NOEXCEPT
{
    return s_TaskImpl.GetOperationMode();
}

bool WillDisconnectNetworkWhenEnteringSleep() NN_NOEXCEPT
{
    return s_TaskImpl.WillDisconnectNetworkWhenEnteringSleep();
}

bool WillStayHalfAwakeInsteadSleep() NN_NOEXCEPT
{
    return s_TaskImpl.WillStayHalfAwakeInsteadSleep();
}

void SetTaskClientName(const char* pName) NN_NOEXCEPT
{
    s_TaskImpl.SetClientName(pName);
}

//----------------------------------------------------------------------------------

IStateControlService& GetStateControlService() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(s_pStateControlService != nullptr, "bgtc::InitializeStateControl() should be called before using this API.\n");
    return *s_pStateControlService.Get();
}

Result InitializeStateControl() NN_NOEXCEPT
{
    if (s_CounterForStateControlService.DoInitialize())
    {
        Result result;

#if NN_BGTC_ENABLE_HIPC
        const char* pPortName = PortNames[PortIndex_StateControlService];

        result = sf::CreateHipcProxyByName<IStateControlService, Allocator::Policy>(&s_pStateControlService, pPortName);

        if (result.IsFailure())
        {
            return result;
        }
#else
        typedef nn::sf::ObjectFactory<Allocator::Policy> Factory;

        NN_BGTC_WARN("Windows build is not supported. It may not work correctly.\n");
        s_pStateControlService = Factory::CreateSharedEmplaced<IStateControlService, StateControlServiceImpl>();

        if (!s_pStateControlService)
        {
            return ResultUnexpected();
        }
#endif

        result = AttachEventByService<IStateControlService, &IStateControlService::GetStateChangedEvent>(
            GetStateControlService(),
            s_SystemEventForState, os::EventClearMode_AutoClear
        );
        if (result.IsFailure())
        {
            s_pStateControlService = nullptr;
            return result;
        }
    }
    return ResultSuccess();
}

void FinalizeStateControl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(s_pStateControlService);

    if (s_CounterForStateControlService.DoFinalize())
    {
        s_SystemEventForState.~SystemEvent();
        s_pStateControlService = nullptr;
    }
}

State GetState() NN_NOEXCEPT
{
    int32_t state;
    sf::Out<int32_t> out(&state);
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetStateControlService().GetState(out)
    );
    return static_cast<State>(state);
}

os::SystemEvent& GetStateChangedEvent() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(s_CounterForStateControlService.IsInitialized());
    return s_SystemEventForState;
}

void NotifyEnteringHalfAwake() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetStateControlService().NotifyEnteringHalfAwake()
    );
}
void NotifyLeavingHalfAwake() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetStateControlService().NotifyLeavingHalfAwake()
    );
}

void SetIsUsingSleepUnsupportedDevices(bool bIsUsing) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetStateControlService().SetIsUsingSleepUnsupportedDevices(bIsUsing)
    );
}

}}  // namespace nn::npns

