﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_BitTypes.h>
#include <nn/util/util_BitPack.h>
#include <nn/bconfig/bconfig_Api.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fssystem/fs_IStorage.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace bconfig {

namespace {

struct DevelopmentFlags
{
    typedef util::BitPack8::Field<0,                                1, bool>  Unused10;
    typedef util::BitPack8::Field<Unused10::Next,                   1, bool>  EnableDevelopmentFunction; //!< 開発用機能有効
    typedef util::BitPack8::Field<EnableDevelopmentFunction::Next,  6, Bit8>  Unused11;
};

struct KernelFlags1
{
    typedef util::BitPack8::Field<0,                                     1, bool>  EnableNonZeroFillMemory;          //!< メモリの非ゼロ値フィルを有効化
    typedef util::BitPack8::Field<EnableNonZeroFillMemory::Next,         1, bool>  EnableUserExceptionHandler;       //!< ユーザランド例外ハンドラを有効化
    typedef util::BitPack8::Field<EnableUserExceptionHandler::Next,      1, bool>  EnablePerformanceMonitoringUnit;  //!< PMU を有効化
    typedef util::BitPack8::Field<EnablePerformanceMonitoringUnit::Next, 5, Bit8>  Unused12;
};

struct KernelFlags0
{
    typedef util::BitPack8::Field<0,                                1, bool>  CallShowErrorOnPanic;
    typedef util::BitPack8::Field<CallShowErrorOnPanic::Next,       7, Bit8>  Unused20;
};

struct Default0Byte4
{
    typedef util::BitPack8::Field<0,                                1, bool>  EnableTscInitialValue;
    typedef util::BitPack8::Field<EnableTscInitialValue::Next,      7, Bit8>  Unused;
};

}

// 読み書きできるサイズは SectorSize の倍数のみ
static_assert(sizeof(BootConfig) == sdmmc::SectorSize, "");

void LoadBootConfig(BootConfig* pOutBootConfig) NN_NOEXCEPT
{
    int64_t partitionSize;
    std::unique_ptr<fssystem::IStorage> m_Storage;

    NN_ABORT_UNLESS(pOutBootConfig);

    // Part1 を読み込む
    auto result = fs::OpenBisPartition(&m_Storage, fs::BisPartitionId::BootConfigAndPackage2Part1);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = m_Storage->GetSize(&partitionSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ABORT_UNLESS(partitionSize > sizeof(BootConfig));

    result = m_Storage->Read(0, pOutBootConfig, sizeof(BootConfig));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void SaveBootConfig(const BootConfig* bootConfig) NN_NOEXCEPT
{
    int64_t partitionSize;
    std::unique_ptr<fssystem::IStorage> m_Storage;

    NN_ABORT_UNLESS(bootConfig);

    // Part1 に書き込む
    auto result = fs::OpenBisPartition(&m_Storage, fs::BisPartitionId::BootConfigAndPackage2Part1);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = m_Storage->GetSize(&partitionSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ABORT_UNLESS(partitionSize > sizeof(BootConfig));

    result = m_Storage->Write(0, bootConfig, sizeof(BootConfig));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//
// Memory mode
//
void SetMemoryMode(BootConfig* bootConfig, Bit8 memoryMode) NN_NOEXCEPT
{
    bootConfig->reservedDefault0[3] = memoryMode;
}

Bit8 GetMemoryMode(const BootConfig* bootConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT(bootConfig);
    return bootConfig->reservedDefault0[3];
}

//
// EnableNonZeroFillMemory
//
void SetEnableNonZeroFillMemory(BootConfig* bootConfig, bool enabled) NN_NOEXCEPT
{
    util::BitPack8 kernelFlags1 = { static_cast<Bit8>(bootConfig->reservedDefault1[1]) };
    kernelFlags1.Set<KernelFlags1::EnableNonZeroFillMemory>(enabled ? 1 : 0);
    bootConfig->reservedDefault1[1] = kernelFlags1.storage;
}

bool GetEnableNonZeroFillMemory(const BootConfig* bootConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT(bootConfig);
    const util::BitPack8 kernelFlags1 = { static_cast<Bit8>(bootConfig->reservedDefault1[1]) };
    return kernelFlags1.Get<KernelFlags1::EnableNonZeroFillMemory>();
}

//
// EnableUserExceptionHandler
//
void SetEnableUserExceptionHandler(BootConfig* bootConfig, bool enabled) NN_NOEXCEPT
{
    util::BitPack8 kernelFlags1 = { static_cast<Bit8>(bootConfig->reservedDefault1[1]) };
    kernelFlags1.Set<KernelFlags1::EnableUserExceptionHandler>(enabled ? 1 : 0);
    bootConfig->reservedDefault1[1] = kernelFlags1.storage;
}

bool GetEnableUserExceptionHandler(const BootConfig* bootConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT(bootConfig);
    const util::BitPack8 kernelFlags1 = { static_cast<Bit8>(bootConfig->reservedDefault1[1]) };
    return kernelFlags1.Get<KernelFlags1::EnableUserExceptionHandler>();
}

//
// EnablePerformanceMonitoringUnit
//
void SetPerformanceMonitoringUnit(BootConfig* bootConfig, bool enabled) NN_NOEXCEPT
{
    util::BitPack8 kernelFlags1 = { static_cast<Bit8>(bootConfig->reservedDefault1[1]) };
    kernelFlags1.Set<KernelFlags1::EnablePerformanceMonitoringUnit>(enabled ? 1 : 0);
    bootConfig->reservedDefault1[1] = kernelFlags1.storage;
}

bool GetPerformanceMonitoringUnit(const BootConfig* bootConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT(bootConfig);
    const util::BitPack8 kernelFlags1 = { static_cast<Bit8>(bootConfig->reservedDefault1[1]) };
    return kernelFlags1.Get<KernelFlags1::EnablePerformanceMonitoringUnit>();
}

//
// CallShowErrorOnPanic
//
void SetCallShowErrorOnPanic(BootConfig* bootConfig, bool enabled) NN_NOEXCEPT
{
    util::BitPack8 kernelFlags0 = { static_cast<Bit8>(bootConfig->reservedDefault0[1]) };
    kernelFlags0.Set<KernelFlags0::CallShowErrorOnPanic>(enabled ? 1 : 0);
    bootConfig->reservedDefault0[1] = kernelFlags0.storage;
}

bool GetCallShowErrorOnPanic(const BootConfig* bootConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT(bootConfig);
    const util::BitPack8 kernelFlags0 = { static_cast<Bit8>(bootConfig->reservedDefault0[1]) };
    return kernelFlags0.Get<KernelFlags0::CallShowErrorOnPanic>();
}

//
// TSC (Tick) Initial value
//
void SetInitialTscValue(BootConfig* bootConfig, Bit64 value) NN_NOEXCEPT
{
    // value が 0 なら「TSC の初期値有効」フラグを落とす
    util::BitPack8 flags = { static_cast<Bit8>(bootConfig->reservedDefault0[4]) };
    flags.Set<Default0Byte4::EnableTscInitialValue>(value ? 1 : 0);
    bootConfig->reservedDefault0[4] = flags.storage;

    value = value & ((1ull << 55) - 1);

    std::memcpy(bootConfig->tscInitialValue, &value, sizeof(value));
}

Bit64 GetInitialTscValue(const BootConfig* bootConfig) NN_NOEXCEPT
{
    Bit64 value;

    NN_SDK_ASSERT(bootConfig);
    const util::BitPack8 flags = { static_cast<Bit8>(bootConfig->reservedDefault0[4]) };

    std::memcpy(&value, bootConfig->tscInitialValue, sizeof(value));

    return flags.Get<Default0Byte4::EnableTscInitialValue>() ? value : 0;
}

}} // namespace nn::bconfig
