﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Version.h>

#include <nn/bcat/service/bcat_ArchiveDownloader.h>
#include <nn/bcat/service/bcat_ArchiveDecryptionKeyGenerator.h>
#include <nn/bcat/service/bcat_ArchiveVerifier.h>
#include <nn/bcat/detail/service/core/bcat_FileSystem.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_ResultPrivate.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <nn/bcat/detail/service/util/bcat_HttpErrorHandler.h>
#include <cstdio>
#include <cstdlib>
#include <cctype>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>

#define CR 0x0D // NOLINT(preprocessor/const)
#define LF 0x0A // NOLINT(preprocessor/const)

#if defined (NN_BUILD_CONFIG_SPEC_GENERIC)

    // Generic Windows
    #define UA_PLATFORM "5797bc97-d32c-48ad-b47e-3847bda0db7c"

#elif defined(NN_BUILD_CONFIG_SPEC_NX)

    #if defined (NN_BUILD_CONFIG_OS_WIN)
        // NX Windows
        #define UA_PLATFORM "159b8d57-d7af-4ce6-823e-c38d1d661918"
    #elif defined(NN_BUILD_CONFIG_OS_HORIZON)
        // NX Horizon
        #define UA_PLATFORM "789f928b-138e-4b2f-afeb-1acae821d897"
    #else
        #error "Unsupported OS"
    #endif

#else

    #error "Unsupported Spec"

#endif

#define CURL_EASY_DO(exp) \
    do                                                              \
    {                                                               \
        CURLcode code_ = (exp);                                     \
                                                                    \
        if (code_ != CURLE_OK)                                      \
        {                                                           \
            return detail::service::util::HandleHttpError(code_);   \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

#define CURL_SLIST_APPEND(headers, header) \
    do                                                                  \
    {                                                                   \
        curl_slist* headers_ = curl_slist_append(headers, header);      \
                                                                        \
        NN_RESULT_THROW_UNLESS(headers_, ResultHttpErrorOutOfMemory()); \
        headers = headers_;                                             \
    }                                                                   \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace bcat { namespace service {

namespace
{
    void InitializeMeta(ArchiveMetadata* meta) NN_NOEXCEPT
    {
        ArchiveMetadata initial =
        {
            {'m', 'e', 't', 'a'}, 1, {}, 0, {""}, {{'b', 'c', 'a', 't'}, 1}
        };

        std::memcpy(meta, &initial, sizeof (initial));
    }

    nn::Result VerifyArchiveHeader(const ArchiveHeader& header) NN_NOEXCEPT
    {
        if (!(header.magic[0] == 'b' && header.magic[1] == 'c' && header.magic[2] == 'a' && header.magic[3] == 't'))
        {
            NN_RESULT_THROW(ResultDataVerificationFailed());
        }
        if (header.version != 1)
        {
            NN_RESULT_THROW(ResultUnsupportedFormatDetected());
        }
        if (header.encryptionType > EncryptionType_AesCtr256)
        {
            NN_RESULT_THROW(ResultUnsupportedFormatDetected());
        }
        if (header.signatureType > SignatureType_RsaPssSha256)
        {
            NN_RESULT_THROW(ResultUnsupportedFormatDetected());
        }
        if (header.saltIndex > ArchiveDecryptionKeyGenerator::SaltIndexMax)
        {
            NN_RESULT_THROW(ResultUnsupportedFormatDetected());
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result VerifyArchiveMeta(const ArchiveMetadata& meta) NN_NOEXCEPT
    {
        if (!(meta.magic[0] == 'm' && meta.magic[1] == 'e' && meta.magic[2] == 't' && meta.magic[3] == 'a'))
        {
            NN_RESULT_THROW(ResultDataVerificationFailed());
        }
        if (meta.version != 1)
        {
            NN_RESULT_THROW(ResultDataVerificationFailed());
        }

        int eTagLength = nn::util::Strnlen(meta.eTag.value, sizeof (meta.eTag.value));

        if (eTagLength == 0 || static_cast<size_t>(eTagLength) == sizeof (meta.eTag.value))
        {
            NN_RESULT_THROW(ResultDataVerificationFailed());
        }
        for (size_t i = 0; meta.eTag.value[i] != '\0'; i++)
        {
            char c = meta.eTag.value[i];

            // 使用文字の確認程度にとどめる。
            // W/"0000000000000000-0000000000000000"
            if (!(std::isxdigit(c) || c == '"' || c == '-' || c == 'W' || c == '/'))
            {
                NN_RESULT_THROW(ResultDataVerificationFailed());
            }
        }

        NN_RESULT_DO(VerifyArchiveHeader(meta.header));

        NN_RESULT_SUCCESS;
    }

    void GetUserAgent(char* buffer, size_t size) NN_NOEXCEPT
    {
        nn::util::SNPrintf(buffer, size, "libcurl (nnBcat; %s; SDK %d.%d.%d.%d)",
            UA_PLATFORM, NN_SDK_VERSION_MAJOR, NN_SDK_VERSION_MINOR, NN_SDK_VERSION_MICRO, NN_SDK_VERSION_RELSTEP);
    }

    void MakeIncompletePath(char* path, size_t size, const char* dataPath) NN_NOEXCEPT
    {
        nn::util::SNPrintf(path, size, "%s.incomplete", dataPath);
    }
}

ArchiveDownloader::ArchiveDownloader() NN_NOEXCEPT
{
    Reset();
}

ArchiveDownloader::~ArchiveDownloader() NN_NOEXCEPT
{
}

void ArchiveDownloader::Reset() NN_NOEXCEPT
{
    m_AppId = nn::ApplicationId::GetInvalidId();
    m_Passphrase = nullptr;
    m_ProgressCallback = nullptr;
    m_ProgressCallbackParam = nullptr;

    std::memset(&m_FileAccessParam, 0, sizeof (m_FileAccessParam));
    std::memset(&m_RequestParam, 0, sizeof (m_RequestParam));
    std::memset(&m_ResponseParam, 0, sizeof (m_ResponseParam));
    std::memset(&m_DecodeContext, 0, sizeof (m_DecodeContext));

    m_DecodeContext.downloadBuffer = m_DownloadBuffer;
    m_DecodeContext.downloadBufferSize = sizeof (m_DownloadBuffer);

    m_IsMetaDownloaded = false;

    m_pCancelEvent = nullptr;
    m_LastError = nn::ResultSuccess();

    InitializeMeta(&m_Meta);
}

void ArchiveDownloader::SetPath(const char* dataPath, const char* metaPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(dataPath);
    NN_SDK_REQUIRES_NOT_NULL(metaPath);

    m_FileAccessParam.dataPath = dataPath;
    m_FileAccessParam.metaPath = metaPath;

    for (int i = 0; i < NN_ARRAY_SIZE(m_FileAccessParam.mountName) - 1; i++)
    {
        if (dataPath[i] == ':')
        {
            NN_SDK_REQUIRES(nn::util::Strncmp(dataPath, metaPath, i + 1) == 0);

            m_FileAccessParam.mountName[i] = '\0';
            break;
        }
        m_FileAccessParam.mountName[i] = dataPath[i];
    }
}

void ArchiveDownloader::SetPassphrase(const nn::ApplicationId& appId, const char* passphrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(passphrase);

    m_AppId = appId;
    m_Passphrase = passphrase;
}

void ArchiveDownloader::SetETagForIfNoneMatch(const char* eTag) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(eTag);

    m_RequestParam.eTagForIfNoneMatch = eTag;
}

void ArchiveDownloader::SetJournalSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(size == 0 || size >= JournalSizeMin);

    if (size > 0)
    {
        size -= MetaDataJournalSize;
    }

    m_FileAccessParam.journal = size;
}

void ArchiveDownloader::SetProgressCallback(ProgressCallback callback, void* param) NN_NOEXCEPT
{
    m_ProgressCallback = callback;
    m_ProgressCallbackParam = param;
}

void ArchiveDownloader::SetFileWorkBuffer(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_EQUAL(size % DownloadBufferSize, 0u);

    m_DecodeContext.fileBuffer = reinterpret_cast<Bit8*>(buffer);
    m_DecodeContext.fileBufferSize = size;
}

nn::Result ArchiveDownloader::Download(const char* url, nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(url);

    m_pCancelEvent = pCancelEvent;

    NN_RESULT_DO(Open());

    NN_UTIL_SCOPE_EXIT
    {
        if (m_FileAccessParam.isFileOpened)
        {
            nn::fs::CloseFile(m_FileAccessParam.handle);
            m_FileAccessParam.isFileOpened = false;
        }
    };

    CURL* curl = curl_easy_init();

    NN_RESULT_THROW_UNLESS(curl, ResultHttpErrorFailedInit());

    curl_slist* headers = nullptr;

    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curl);

        if (headers)
        {
            curl_slist_free_all(headers);
        }
    };

    ArchiveDecoder decoder;

    // デコーダーの生存期間を関数実行中だけにする。
    m_DecodeContext.pDecoder = &decoder;

    NN_UTIL_SCOPE_EXIT
    {
        m_DecodeContext.pDecoder = nullptr;
    };

    NN_RESULT_DO(Preprocess(&headers, curl, url));

    nn::Result result = Perform(curl);

    if (m_LastError.IsSuccess() && result.IsFailure())
    {
        m_LastError = result;
    }

    result = Postprocess();

    if (m_LastError.IsSuccess() && result.IsFailure())
    {
        m_LastError = result;
    }

    return m_LastError;
}

const ETag& ArchiveDownloader::GetETag() const NN_NOEXCEPT
{
    return m_Meta.eTag;
}

nn::Result ArchiveDownloader::Open() NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadMeta())
        NN_RESULT_CATCH_ALL
        {
            char path[256] = {};
            MakeIncompletePath(path, sizeof (path), m_FileAccessParam.dataPath);

            nn::fs::DeleteFile(path);
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(OpenData());

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::OpenData() NN_NOEXCEPT
{
    bool isCreateNew = false;

    char path[256] = {};
    MakeIncompletePath(path, sizeof (path), m_FileAccessParam.dataPath);

    NN_RESULT_TRY(nn::fs::OpenFile(&m_FileAccessParam.handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(detail::service::core::FileSystem::CreateFile(path, 0));
            NN_RESULT_DO(nn::fs::OpenFile(&m_FileAccessParam.handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

            isCreateNew = true;
        }
    NN_RESULT_END_TRY;

    if (isCreateNew)
    {
        m_ResponseParam.downloaded = 0;
    }
    else
    {
        int64_t downloaded = 0;

        if (nn::fs::GetFileSize(&downloaded, m_FileAccessParam.handle).IsFailure() || (downloaded % ArchiveBlockSize) != 0)
        {
            nn::fs::CloseFile(m_FileAccessParam.handle);

            NN_RESULT_DO(nn::fs::DeleteFile(path));
            NN_RESULT_DO(detail::service::core::FileSystem::CreateFile(path, 0));
            NN_RESULT_DO(nn::fs::OpenFile(&m_FileAccessParam.handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

            downloaded = 0;
        }

        m_ResponseParam.downloaded = downloaded;
    }

    m_FileAccessParam.isFileOpened = true;

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::LoadMeta() NN_NOEXCEPT
{
    nn::fs::FileHandle handle = {};

    bool deleteOnReturn = false;

    NN_RESULT_DO(nn::fs::OpenFile(&handle, m_FileAccessParam.metaPath, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);

        if (deleteOnReturn)
        {
            InitializeMeta(&m_Meta);
            nn::fs::DeleteFile(m_FileAccessParam.metaPath);
        }
    };

    NN_RESULT_TRY(nn::fs::ReadFile(handle, 0, &m_Meta, sizeof (m_Meta)))
        NN_RESULT_CATCH_ALL
        {
            deleteOnReturn = true;
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_TRY(VerifyArchiveMeta(m_Meta))
        NN_RESULT_CATCH_ALL
        {
            deleteOnReturn = true;
            NN_RESULT_THROW(ResultSaveVerificationFailed());
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::SaveMeta() NN_NOEXCEPT
{
    nn::fs::FileHandle handle = {};

    bool deleteOnReturn = false;

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, m_FileAccessParam.metaPath, nn::fs::OpenMode_Write))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(detail::service::core::FileSystem::CreateFile(m_FileAccessParam.metaPath, sizeof (m_Meta)));
            NN_RESULT_DO(nn::fs::OpenFile(&handle, m_FileAccessParam.metaPath, nn::fs::OpenMode_Write));
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);

        if (deleteOnReturn)
        {
            nn::fs::DeleteFile(m_FileAccessParam.metaPath);
        }
    };

    NN_RESULT_TRY(nn::fs::WriteFile(handle, 0, &m_Meta, sizeof (m_Meta), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)))
        NN_RESULT_CATCH_ALL
        {
            deleteOnReturn = true;
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::VerifyData() NN_NOEXCEPT
{
    char path[256] = {};
    MakeIncompletePath(path, sizeof (path), m_FileAccessParam.dataPath);

    bool result = false;
    NN_RESULT_DO(ArchiveVerifier::VerifyFile(&result, path, m_Meta.header, m_DecodeContext.fileBuffer, m_DecodeContext.fileBufferSize));

    NN_RESULT_THROW_UNLESS(result, ResultDataVerificationFailed());

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::Preprocess(curl_slist** outHeaders, CURL* curl, const char* url) NN_NOEXCEPT
{
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_URL, url));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, SslCtxFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_CTX_DATA, this));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, HttpHeaderFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HEADERDATA, this));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, HttpWriteFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEDATA, this));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_XFERINFOFUNCTION, HttpProgressCallback));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_XFERINFODATA, this));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 1));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_LIMIT, 1));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_TIME, 30));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 30));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0));

    // CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_VERBOSE, 1));

    if (m_ResponseParam.downloaded)
    {
        char header[64] = {};
        nn::util::SNPrintf(header, sizeof (header), "If-Range: %s", m_Meta.eTag.value);

        CURL_SLIST_APPEND(*outHeaders, header);

        m_RequestParam.rangeBytes = sizeof (m_Meta.header) + m_ResponseParam.downloaded;
        nn::util::SNPrintf(header, sizeof (header), "Range: bytes=%llu-", m_RequestParam.rangeBytes);

        m_ResponseParam.rangeBytes = m_RequestParam.rangeBytes;

        CURL_SLIST_APPEND(*outHeaders, header);
    }
    else if (m_RequestParam.eTagForIfNoneMatch && m_RequestParam.eTagForIfNoneMatch[0] != '\0')
    {
        char header[128] = {};
        nn::util::SNPrintf(header, sizeof (header), "If-None-Match: %s", m_RequestParam.eTagForIfNoneMatch);

        CURL_SLIST_APPEND(*outHeaders, header);
    }

    {
        nn::settings::system::SerialNumber serialNumber = {};
        nn::settings::system::GetSerialNumber(&serialNumber);

        char header[128] = {};
        nn::util::SNPrintf(header, sizeof (header), "X-Nintendo-Serial-Number: %s", serialNumber.string);

        CURL_SLIST_APPEND(*outHeaders, header);
    }

    char ua[128] = {};
    GetUserAgent(ua, sizeof (ua));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_USERAGENT, ua));

    if (*outHeaders)
    {
        CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HTTPHEADER, *outHeaders));
    }

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::Perform(CURL* curl) NN_NOEXCEPT
{
    if (m_pCancelEvent && m_pCancelEvent->TryWait())
    {
        NN_RESULT_THROW(ResultCanceledByUser());
    }

    CURLM* multi = curl_multi_init();

    if (!multi)
    {
        NN_RESULT_THROW(ResultHttpErrorFailedInit());
    }

    curl_multi_add_handle(multi, curl);

    NN_UTIL_SCOPE_EXIT
    {
        curl_multi_remove_handle(multi, curl);
        curl_multi_cleanup(multi);
    };

    int stillRunning = 0;

    while (NN_STATIC_CONDITION(true))
    {
        int rval = 0;
        CURLMcode code = curl_multi_wait(multi, nullptr, 0, 10, &rval);

        if (code != CURLM_OK)
        {
            if (code == CURLM_OUT_OF_MEMORY)
            {
                NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
            }
            else
            {
                NN_RESULT_THROW(ResultHttpErrorBadFunctionArgument());
            }
        }

        if (rval == -1)
        {
            NN_RESULT_THROW(ResultHttpErrorRecvError());
        }

        curl_multi_perform(multi, &stillRunning);

        if (stillRunning == 0)
        {
            break;
        }

        if (m_pCancelEvent && m_pCancelEvent->TryWait())
        {
            NN_RESULT_THROW(ResultCanceledByUser());
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    if (stillRunning == 0)
    {
        CURLMsg* msg = nullptr;

        do
        {
            int n = 0;
            msg = curl_multi_info_read(multi, &n);

            if (msg && (msg->msg == CURLMSG_DONE))
            {
                return detail::service::util::HandleHttpError(msg->data.result);
            }
        }
        while (msg);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::Postprocess() NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        if (m_FileAccessParam.isFileOpened && m_ResponseParam.downloaded > 0)
        {
            nn::fs::FlushFile(m_FileAccessParam.handle);
        }
    };

    if (m_DecodeContext.sizeInFileBuffer > 0)
    {
        NN_RESULT_DO(FlushJournalIfFull(m_DecodeContext.sizeInFileBuffer));

        NN_RESULT_DO(nn::fs::WriteFile(m_FileAccessParam.handle, m_ResponseParam.downloaded,
            m_DecodeContext.fileBuffer, m_DecodeContext.sizeInFileBuffer, nn::fs::WriteOption()));

        m_ResponseParam.downloaded += m_DecodeContext.sizeInFileBuffer;
        m_FileAccessParam.journalWritten += m_DecodeContext.sizeInFileBuffer;
    }

    if (m_LastError.IsSuccess())
    {
        // ダウンロード完了時にデコードしきれていないデータが存在すればそれも書き込む。
        if (m_DecodeContext.sizeInDownloadBuffer > 0)
        {
            m_DecodeContext.pDecoder->Decode(m_DecodeContext.fileBuffer,
                m_DecodeContext.downloadBuffer, m_DecodeContext.sizeInDownloadBuffer);

            NN_RESULT_DO(FlushJournalIfFull(m_DecodeContext.sizeInDownloadBuffer));

            NN_RESULT_DO(nn::fs::WriteFile(m_FileAccessParam.handle, m_ResponseParam.downloaded,
                m_DecodeContext.fileBuffer, m_DecodeContext.sizeInDownloadBuffer, nn::fs::WriteOption()));

            m_ResponseParam.downloaded += m_DecodeContext.sizeInDownloadBuffer;
            m_FileAccessParam.journalWritten += m_DecodeContext.sizeInDownloadBuffer;
        }
    }

    if (m_ResponseParam.downloaded > 0)
    {
        NN_RESULT_DO(nn::fs::FlushFile(m_FileAccessParam.handle));
    }

    // 416 ステータスはすべて受信済みの時に Range リクエストを行った場合に発生する。
    if (m_LastError.IsSuccess() || ResultServerError416::Includes(m_LastError))
    {
        nn::fs::CloseFile(m_FileAccessParam.handle);
        m_FileAccessParam.isFileOpened = false;

        char path[256] = {};
        MakeIncompletePath(path, sizeof (path), m_FileAccessParam.dataPath);

        NN_RESULT_TRY(VerifyData())
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_DO(nn::fs::DeleteFile(path));
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_TRY(nn::fs::DeleteFile(m_FileAccessParam.dataPath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO(nn::fs::RenameFile(path, m_FileAccessParam.dataPath));

        m_LastError = nn::ResultSuccess();
    }

    if (m_IsMetaDownloaded)
    {
        NN_RESULT_DO(SaveMeta());
    }

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::FlushJournalIfFull(size_t writeSize) NN_NOEXCEPT
{
    if (m_FileAccessParam.journal > 0 &&
        m_FileAccessParam.journalWritten + static_cast<int64_t>(writeSize) >= m_FileAccessParam.journal)
    {
        char path[256] = {};
        MakeIncompletePath(path, sizeof (path), m_FileAccessParam.dataPath);

        NN_DETAIL_BCAT_INFO("[bcat] Flush journal(%s) ...\n", path);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::FlushFile(m_FileAccessParam.handle));
        nn::fs::CloseFile(m_FileAccessParam.handle);

        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::service::core::FileSystem::Commit(m_FileAccessParam.mountName));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&m_FileAccessParam.handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

        m_FileAccessParam.journalWritten = 0;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::HeaderCallback(const char* label, const char* value) NN_NOEXCEPT
{
    if (nn::util::Strncmp(label, "ETag", sizeof ("ETag")) == 0)
    {
        nn::util::Strlcpy(m_Meta.eTag.value, value, sizeof (m_Meta.eTag.value));
    }
    else if (m_ResponseParam.statusCode == 200 && nn::util::Strncmp(label, "Content-Length", sizeof ("Content-Length")) == 0)
    {
        m_Meta.contentLength = std::strtoll(value, nullptr, 10);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::HeaderEndCallback() NN_NOEXCEPT
{
    // 新規ダウンロード
    if (m_ResponseParam.statusCode == 200)
    {
        NN_RESULT_DO(nn::fs::SetFileSize(m_FileAccessParam.handle, 0));

        m_ResponseParam.downloaded = 0;
        m_ResponseParam.rangeBytes = 0;
    }
    // レジュームダウンロード
    else if (m_ResponseParam.statusCode == 206)
    {
        m_DecodeContext.pDecoder->Initialize(m_Meta.header, m_Passphrase, m_AppId,
            static_cast<size_t>(m_ResponseParam.downloaded) / ArchiveBlockSize);

        m_IsMetaDownloaded = true;
    }
    else
    {
        return detail::service::util::HandleHttpStatusCode(m_ResponseParam.statusCode);
    }

    NN_RESULT_SUCCESS;
}

nn::Result ArchiveDownloader::ResponseBodyCallback(const char* buffer, size_t size) NN_NOEXCEPT
{
    if (!m_IsMetaDownloaded)
    {
        size_t remainSize = sizeof (m_Meta.header) - m_DecodeContext.sizeInDownloadBuffer;

        size_t copySize = (remainSize < size) ? remainSize : size;

        std::memcpy(&m_DecodeContext.downloadBuffer[m_DecodeContext.sizeInDownloadBuffer], buffer, copySize);
        m_DecodeContext.sizeInDownloadBuffer += copySize;

        if (m_DecodeContext.sizeInDownloadBuffer == sizeof (m_Meta.header))
        {
            std::memcpy(&m_Meta.header, m_DecodeContext.downloadBuffer, m_DecodeContext.sizeInDownloadBuffer);

            NN_RESULT_DO(VerifyArchiveHeader(m_Meta.header));

            m_DecodeContext.pDecoder->Initialize(m_Meta.header, m_Passphrase, m_AppId);

            m_DecodeContext.sizeInDownloadBuffer = 0;
            m_IsMetaDownloaded = true;
        }

        buffer += copySize;
        size -= copySize;
    }

    if (size == 0)
    {
        NN_RESULT_SUCCESS;
    }

    while (size > 0)
    {
        size_t remainSize = m_DecodeContext.downloadBufferSize - m_DecodeContext.sizeInDownloadBuffer;

        size_t copySize = (remainSize < size) ? remainSize : size;

        std::memcpy(&m_DecodeContext.downloadBuffer[m_DecodeContext.sizeInDownloadBuffer], buffer, copySize);
        m_DecodeContext.sizeInDownloadBuffer += copySize;

        if (m_DecodeContext.sizeInDownloadBuffer == m_DecodeContext.downloadBufferSize)
        {
            m_DecodeContext.pDecoder->Decode(&m_DecodeContext.fileBuffer[m_DecodeContext.sizeInFileBuffer],
                m_DecodeContext.downloadBuffer, m_DecodeContext.downloadBufferSize);

            m_DecodeContext.sizeInFileBuffer += m_DecodeContext.sizeInDownloadBuffer;

            if (m_DecodeContext.sizeInFileBuffer == m_DecodeContext.fileBufferSize)
            {
                NN_RESULT_DO(FlushJournalIfFull(m_DecodeContext.sizeInFileBuffer));

                NN_RESULT_DO(nn::fs::WriteFile(m_FileAccessParam.handle, m_ResponseParam.downloaded,
                    m_DecodeContext.fileBuffer, m_DecodeContext.sizeInFileBuffer, nn::fs::WriteOption()));

                m_ResponseParam.downloaded += m_DecodeContext.sizeInFileBuffer;
                m_FileAccessParam.journalWritten += m_DecodeContext.sizeInFileBuffer;

                m_DecodeContext.sizeInFileBuffer = 0;
            }

            m_DecodeContext.sizeInDownloadBuffer = 0;
        }

        buffer += copySize;
        size -= copySize;
    }

    NN_RESULT_SUCCESS;
}

CURLcode ArchiveDownloader::SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT
{
    nn::ssl::Context* context = reinterpret_cast<nn::ssl::Context*>(ssl);

    if (context->Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
    {
        return CURLE_ABORTED_BY_CALLBACK;
    }

    return CURLE_OK;
}

size_t ArchiveDownloader::HttpHeaderFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    ArchiveDownloader* downloader = static_cast<ArchiveDownloader*>(param);

    size_t bufferSize = size * count;

    if (bufferSize >= sizeof ("HTTP/X.Y SSS ") - 1 && nn::util::Strncmp(buffer, "HTTP/", 5) == 0)
    {
        int majorVersion = 0;
        int minorVersion = 0;
        int statusCode = 0;

        char space = '\0';

        int result = std::sscanf(buffer, "HTTP/%1d.%1d %03d%c", &majorVersion, &minorVersion, &statusCode, &space);

        if (result == 4 && space == ' ')
        {
            downloader->m_ResponseParam.statusCode = statusCode;
        }

        return bufferSize;
    }

    // 1xx 系のステータスコードは内部処理に影響を与えるものであるため、アプリに影響を与えないようにする。
    if (downloader->m_ResponseParam.statusCode / 100 == 1)
    {
        return bufferSize;
    }

    if (bufferSize == 2 && buffer[0] == CR && buffer[1] == LF)
    {
        downloader->m_LastError = downloader->HeaderEndCallback();

        if (downloader->m_LastError.IsFailure())
        {
            return 0;
        }

        return bufferSize;
    }

    char* end = buffer + bufferSize;

    char* label = buffer;
    char* colon = reinterpret_cast<char*>(std::memchr(buffer, ':', bufferSize));

    if (!colon)
    {
        return bufferSize;
    }

    char* value = colon + 1;

    if (value == end)
    {
        return bufferSize;
    }

    while ((*value == ' ' || *value == '\t') && value != end)
    {
        value++;
    }

    size_t remainSize = end - value;

    // 必ず改行コード（CRLF）は存在するはず。
    if (remainSize < 2)
    {
        return bufferSize;
    }

    char* cr = reinterpret_cast<char*>(std::memchr(value, CR, remainSize));

    if (!cr)
    {
        return bufferSize;
    }

    *colon = '\0';
    *cr = '\0';

    downloader->m_LastError = downloader->HeaderCallback(label, value);

    *colon = ':';
    *cr = CR;

    if (downloader->m_LastError.IsFailure())
    {
        return 0;
    }

    return bufferSize;
}

size_t ArchiveDownloader::HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    ArchiveDownloader* downloader = static_cast<ArchiveDownloader*>(param);

    size_t bufferSize = size * count;

    downloader->m_LastError = downloader->ResponseBodyCallback(buffer, bufferSize);

    if (downloader->m_LastError.IsFailure())
    {
        return 0;
    }

    return bufferSize;
}

int ArchiveDownloader::HttpProgressCallback(void* param, curl_off_t dlTotal, curl_off_t dlNow, curl_off_t ulTotal, curl_off_t ulNow) NN_NOEXCEPT
{
    NN_UNUSED(ulTotal);
    NN_UNUSED(ulNow);

    ArchiveDownloader* downloader = static_cast<ArchiveDownloader*>(param);

    if (downloader->m_ProgressCallback)
    {
        // レジュームダウンロード時は、ダウンロード済み分を加算しておく。
        if (!downloader->m_ProgressCallback(downloader->m_ResponseParam.rangeBytes + static_cast<int64_t>(dlNow),
            downloader->m_ResponseParam.rangeBytes + static_cast<int64_t>(dlTotal), downloader->m_ProgressCallbackParam))
        {
            return 1;
        }
    }

    return 0;
}

}}}
