﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_FileSystem.h>
#include <nn/bcat/detail/service/core/bcat_FileSystemConfig.h>
#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/util/util_BitUtil.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    struct StorageParam
    {
        const char* mountName;
        nn::fs::SystemSaveDataId saveDataId;
        int64_t storageSize;
        int64_t storageJournalSize;
    };

    const StorageParam StorageParams[] =
    {
        {NN_DETAIL_BCAT_SYSTEM_MOUNT_NAME, 0x80000000000000C1, SystemStorageSize, SystemStorageJournalSize},
        {NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME, 0x80000000000000C2, DownloadStorageSize, DownloadStorageJournalSize}
    };
}

namespace
{
    nn::os::SdkMutexType g_Mutexes[] = {NN_OS_SDK_MUTEX_INITIALIZER(), NN_OS_SDK_MUTEX_INITIALIZER()};

    bool g_IsTestMode = false;
}

NN_STATIC_ASSERT(NN_ARRAY_SIZE(StorageParams) == NN_ARRAY_SIZE(g_Mutexes));

namespace
{
    nn::Result MountSystemSaveData(const char* name, nn::fs::SystemSaveDataId id,
        int64_t size, int64_t journalSize, bool createIfNotExists) NN_NOEXCEPT
    {
        NN_RESULT_TRY(nn::fs::MountSystemSaveData(name, id))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                if (createIfNotExists)
                {
                    NN_RESULT_DO(nn::fs::CreateSystemSaveData(id, size, journalSize, 0));
                    NN_RESULT_DO(nn::fs::MountSystemSaveData(name, id));

                    NN_DETAIL_BCAT_INFO("[bcat] A new system save data (%s) is created.\n", name);
                }
                else
                {
                    NN_RESULT_RETHROW;
                }
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    int GetStorageIndex(const char* mountName) NN_NOEXCEPT
    {
        for (int i = 0; i < NN_ARRAY_SIZE(StorageParams); i++)
        {
            if (nn::util::Strncmp(mountName, StorageParams[i].mountName, nn::fs::MountNameLengthMax + 1) == 0)
            {
                return i;
            }
        }

        NN_ABORT("");

        return -1;
    }
}

void FileSystem::EnableTestMode() NN_NOEXCEPT
{
    g_IsTestMode = true;
}

nn::fs::AbortSpecifier FileSystem::ResultHandlerForHandlingDataCorruption(nn::Result result) NN_NOEXCEPT
{
    if (nn::fs::ResultDataCorrupted::Includes(result))
    {
        return nn::fs::AbortSpecifier::ReturnResult;
    }

    return nn::fs::AbortSpecifier::Default;
}

nn::Result FileSystem::Mount(const char* mountName, bool createIfNotExists) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        NN_RESULT_SUCCESS;
    }

    int index = GetStorageIndex(mountName);

    g_Mutexes[index].Lock();

    NN_RESULT_TRY(MountSystemSaveData(StorageParams[index].mountName, StorageParams[index].saveDataId,
        StorageParams[index].storageSize, StorageParams[index].storageJournalSize, createIfNotExists))
        NN_RESULT_CATCH_ALL
        {
            g_Mutexes[index].Unlock();
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

void FileSystem::Unmount(const char* mountName) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        return;
    }

    int index = GetStorageIndex(mountName);

    nn::fs::Unmount(StorageParams[index].mountName);

    g_Mutexes[index].Unlock();
}

nn::Result FileSystem::Commit(const char* mountName) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        NN_RESULT_SUCCESS;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CommitSaveData(mountName));

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::GetFreeSpaceSize(int64_t* outSize, const char* mountName) NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        *outSize = StorageParams[GetStorageIndex(mountName)].storageSize;
    }
    else
    {
        char root[nn::fs::MountNameLengthMax + 3] = {};
        nn::util::SNPrintf(root, sizeof (root), "%s:/", mountName);

        NN_RESULT_DO(nn::fs::GetFreeSpaceSize(outSize, root));
    }

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::CreateDirectoryRecursively(const char* path, bool isFilePath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    char currentPath[nn::fs::EntryNameLengthMax + 1] = {};
    bool root = true;

    char c = path[0];

    for (int i = 0; path[i] != '\0'; i++)
    {
        c = path[i];

        if (c == '/')
        {
            if (!root)
            {
                NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
                    NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                    {
                    }
                NN_RESULT_END_TRY;
            }
            else
            {
                root = false;
            }
        }

        currentPath[i] = c;
    }

    if (!root && !isFilePath && c != '/')
    {
        NN_RESULT_TRY(nn::fs::CreateDirectory(currentPath))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    return nn::fs::DeleteDirectoryRecursively(path);
}

nn::Result FileSystem::CreateFile(const char* path, int64_t size, bool ignoreIfExists) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    NN_RESULT_TRY(nn::fs::CreateFile(path, size))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(CreateDirectoryRecursively(path, true));
            NN_RESULT_DO(nn::fs::CreateFile(path, size));
        }
        NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
        {
            if (!ignoreIfExists)
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(currentPath);
    NN_SDK_REQUIRES_NOT_NULL(newPath);

    NN_RESULT_TRY(nn::fs::RenameFile(currentPath, newPath))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(CreateDirectoryRecursively(newPath, true));
            NN_RESULT_DO(nn::fs::RenameFile(currentPath, newPath));
        }
        NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
        {
            NN_RESULT_DO(nn::fs::DeleteFile(newPath));
            NN_RESULT_DO(nn::fs::RenameFile(currentPath, newPath));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result FileSystem::GetFileSize(int64_t* outSize, const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(path);

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::GetFileSize(outSize, handle));

    NN_RESULT_SUCCESS;
}

bool FileSystem::Exists(const char* path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);

    nn::fs::DirectoryEntryType entryType = {};

    return nn::fs::GetEntryType(&entryType, path).IsSuccess();
}

nn::Result FileSystem::CopyFileWithBuffer(JournalInfo* outJournalInfo, const char* source, const char* dest,
    JournalInfo journalInfo, void* buffer, size_t bufferSize, nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outJournalInfo);
    NN_SDK_REQUIRES_NOT_NULL(source);
    NN_SDK_REQUIRES_NOT_NULL(dest);
    NN_SDK_REQUIRES_EQUAL(journalInfo.size % BlockSize, 0);
    NN_SDK_REQUIRES_GREATER(journalInfo.size, static_cast<size_t>(BlockSize));
    NN_SDK_REQUIRES_GREATER(journalInfo.size, journalInfo.written);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_EQUAL(bufferSize % BlockSize, 0);
    NN_SDK_REQUIRES_GREATER(bufferSize, static_cast<size_t>(BlockSize));

    char mountName[nn::fs::MountNameLengthMax + 1] = {};

    for (int i = 0; i < NN_ARRAY_SIZE(mountName) - 1; i++)
    {
        if (dest[i] == ':')
        {
            break;
        }
        mountName[i] = dest[i];
    }

    nn::fs::FileHandle sourceHandle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&sourceHandle, source, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(sourceHandle);
    };

    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, sourceHandle));

    NN_RESULT_TRY(nn::fs::DeleteFile(dest))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(CreateFile(dest, fileSize));

    nn::fs::FileHandle destHandle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&destHandle, dest, nn::fs::OpenMode_Write));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(destHandle);
    };

    int64_t offset = 0;

    while (offset < fileSize && !(pCancelEvent && pCancelEvent->TryWait()))
    {
        size_t copyBufferSize = std::min(static_cast<size_t>(journalInfo.size - journalInfo.written), bufferSize);

        NN_SDK_ASSERT(copyBufferSize > 0 && (copyBufferSize % BlockSize) == 0);

        size_t read = 0;
        NN_RESULT_DO(nn::fs::ReadFile(&read, sourceHandle, offset, buffer, copyBufferSize));

        if (read == 0)
        {
            break;
        }

        NN_RESULT_DO(nn::fs::WriteFile(destHandle, offset, buffer, read, nn::fs::WriteOption::MakeValue(0)));

        offset += static_cast<int64_t>(read);

        journalInfo.written += nn::util::align_up(read, BlockSize);

        NN_SDK_ASSERT(journalInfo.written <= journalInfo.size);

        if (journalInfo.written == journalInfo.size)
        {
            NN_DETAIL_BCAT_INFO("[bcat] Flush journal(%s) ...\n", dest);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::FlushFile(destHandle));
            nn::fs::CloseFile(destHandle);

            NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(mountName));

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&destHandle, dest, nn::fs::OpenMode_Write));
            journalInfo.written = 0;
        }
    }

    NN_RESULT_DO(nn::fs::FlushFile(destHandle));

    NN_RESULT_THROW_UNLESS(!(pCancelEvent && pCancelEvent->TryWait()), ResultCanceledByUser());

    *outJournalInfo = journalInfo;

    NN_RESULT_SUCCESS;
}

int64_t FileSystem::CalculateFileSizeOnStorage(int64_t size) NN_NOEXCEPT
{
    return nn::util::align_up(size, BlockSize);
}

}}}}}
