﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_DirectorySynchronizer.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryListReader.h>
#include <nn/bcat/detail/service/msgpack/bcat_FileInputStream.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    void MakeDownloadFilePath(char* path, size_t size, nn::ApplicationId appId, const char* dirName, const char* fileName) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/directories/%s/files/%s",
            appId.value, dirName, fileName);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }
}

DirectorySynchronizer::DirectorySynchronizer() NN_NOEXCEPT :
    m_Mutex(true),
    m_AppId(nn::ApplicationId::GetInvalidId()),
    m_pCancelEvent(nullptr),
    m_FileCount(0)
{
}

void DirectorySynchronizer::SetCancelEvent(nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCancelEvent);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_pCancelEvent = pCancelEvent;
}

nn::Result DirectorySynchronizer::Sync(nn::ApplicationId appId, const DirectoryMeta& dirMeta, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_BCAT_INFO("[bcat] DirectorySynchronizer::Sync(%016llx, %s) ...\n", appId.value, dirMeta.name.value);

    m_AppId = appId;
    m_DirMeta = dirMeta;

    NN_RESULT_DO(StepSetup(buffer, bufferSize));
    NN_RESULT_DO(CreateTransaction());

    // 容量を確保するために、削除→更新→追加の順で処理すること。

    NN_RESULT_DO(StepRemoveRemovedFiles());
    NN_RESULT_DO(StepRemoveOverwrittenFiles());

    NN_RESULT_TRY(StepCopyFiles(buffer, bufferSize))
        NN_RESULT_CATCH(ResultCanceledByUser)
        {
            NN_RESULT_DO(RemoveUnstableDirectory());
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(nn::fs::ResultUsableSpaceNotEnough)
        {
            NN_RESULT_DO(RemoveUnstableDirectory());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_TRY(StepCommit())
        NN_RESULT_CATCH(nn::fs::ResultUsableSpaceNotEnough)
        {
            NN_RESULT_DO(RemoveUnstableDirectory());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::Recover(nn::ApplicationId appId, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_AppId = appId;

    NN_RESULT_TRY(LoadTransaction())
        NN_RESULT_CATCH(ResultSaveVerificationFailed)
        {
            NN_DETAIL_BCAT_INFO("[bcat] The transaction file was corrupted. Do nothing.\n");
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_DETAIL_BCAT_INFO("[bcat] Do nothing.\n");
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_DETAIL_BCAT_INFO("[bcat] DirectorySynchronizer::Recover(%016llx, %s) ...\n", m_AppId.value, m_DirMeta.name.value);

    NN_RESULT_DO(StepSetup(buffer, bufferSize));

    // 容量を確保するために、削除→更新→追加の順で処理すること。

    NN_RESULT_DO(StepRemoveRemovedFiles());
    NN_RESULT_DO(StepRemoveOverwrittenFiles());

    NN_RESULT_TRY(StepCopyFiles(buffer, bufferSize))
        NN_RESULT_CATCH(ResultCanceledByUser)
        {
            NN_RESULT_DO(RemoveUnstableDirectory());
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(RemoveUnstableDirectory());
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH(nn::fs::ResultUsableSpaceNotEnough)
        {
            NN_RESULT_DO(RemoveUnstableDirectory());
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_TRY(StepCommit())
        NN_RESULT_CATCH(nn::fs::ResultUsableSpaceNotEnough)
        {
            NN_RESULT_DO(RemoveUnstableDirectory());
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

bool DirectorySynchronizer::IsTransactionCreated(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    char path[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeTransactionPath(path, sizeof (path), appId);

    return FileSystem::Exists(path);
}

nn::Result DirectorySynchronizer::StepSetup(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectorySynchronizer::StepSetup ...\n");

    char listPath[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeListPath(listPath, sizeof (listPath), m_AppId);

    detail::service::msgpack::FileInputStream stream;

    stream.SetBuffer(buffer, bufferSize);

    NN_RESULT_DO(stream.Open(listPath));

    DeliveryListReader reader;

    reader.SetupForFileListing(&m_FileCount, m_Files, NN_ARRAY_SIZE(m_Files), m_DirMeta.name.value);

    NN_RESULT_DO(reader.Read(nullptr, stream));

    stream.Close();

    NN_RESULT_DO(FileMetaView::LoadFromDeliveryCacheStorage(&m_FileMetaView, m_AppId, m_DirMeta.name.value));
    NN_RESULT_DO(m_FileDiffView.Make(m_FileMetaView, m_Files, m_FileCount));

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::StepRemoveRemovedFiles() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectorySynchronizer::StepRemoveRemovedFiles ...\n");

    int indexes[DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    m_FileDiffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), DiffType_Remove);

    if (count == 0)
    {
        NN_DETAIL_BCAT_INFO("[bcat] Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    for (int i = 0; i < count; i++)
    {
        FileMeta entry = {};
        NN_RESULT_DO(m_FileMetaView.GetEntry(&entry, indexes[i]));

        NN_DETAIL_BCAT_INFO("[bcat] Remove file(%s/%s) ...\n", m_DirMeta.name.value, entry.name.value);

        char path[160] = {};
        DeliveryCacheStorageManager::GetInstance().MakeFilePath(path, sizeof (path),
            m_AppId, m_DirMeta.name.value, entry.name.value);

        NN_RESULT_TRY(nn::fs::DeleteFile(path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;
    }

    // 後ろから削除することで、削除するインデックスがずれることを回避する。
    for (int i = count - 1; i >= 0; i--)
    {
        NN_RESULT_DO(m_FileMetaView.Remove(indexes[i]));
    }

    NN_RESULT_DO(m_FileDiffView.Make(m_FileMetaView, m_Files, m_FileCount));

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::StepRemoveOverwrittenFiles() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectorySynchronizer::StepRemoveOverwrittenFiles ...\n");

    int indexes[DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    m_FileDiffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), DiffType_Update);

    if (count == 0)
    {
        NN_DETAIL_BCAT_INFO("[bcat] Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    // ファイルの更新順によってストレージの空き容量が不足することがないよう、先に容量を空けておく。
    for (int i = 0; i < count; i++)
    {
        FileMeta entry = {};
        NN_RESULT_DO(m_FileMetaView.GetEntry(&entry, indexes[i]));

        NN_DETAIL_BCAT_INFO("[bcat] Remove file(%s/%s) ...\n", m_DirMeta.name.value, entry.name.value);

        char path[160] = {};
        DeliveryCacheStorageManager::GetInstance().MakeFilePath(path, sizeof (path),
            m_AppId, m_DirMeta.name.value, entry.name.value);

        NN_RESULT_TRY(nn::fs::DeleteFile(path))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::StepCopyFiles(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectorySynchronizer::StepCopyFiles ...\n");

    NN_DETAIL_BCAT_INFO("[bcat] -- DirectorySynchronizer::CopyFiles(Update) ...\n");
    NN_RESULT_DO(CopyFiles(DiffType_Update, buffer, bufferSize));

    NN_DETAIL_BCAT_INFO("[bcat] -- DirectorySynchronizer::CopyFiles(Add) ...\n");
    NN_RESULT_DO(CopyFiles(DiffType_Add, buffer, bufferSize));

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::StepCommit() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectorySynchronizer::StepCommit ...\n");

    NN_RESULT_DO(FileMetaView::SaveToDeliveryCacheStorage(m_FileMetaView, m_AppId, m_DirMeta.name.value, false));

    NN_RESULT_DO(DirectoryMetaView::LoadFromDeliveryCacheStorage(&m_DirectoryMetaView, m_AppId));
    NN_RESULT_DO(m_DirectoryMetaView.AddOrUpdate(m_DirMeta));
    NN_RESULT_DO(DirectoryMetaView::SaveToDeliveryCacheStorage(m_DirectoryMetaView, m_AppId, false));

    NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(m_AppId));

    NN_RESULT_DO(DeleteTransaction());

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::CopyFiles(DiffType diffType, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    int indexes[DeliveryCacheFileCountMaxPerDirectory] = {};
    int count = 0;

    m_FileDiffView.GetIndexListFromRemote(&count, indexes, NN_ARRAY_SIZE(indexes), diffType);

    if (count == 0)
    {
        NN_DETAIL_BCAT_INFO("[bcat] Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    // ジャーナルサイズは管理テーブル分を引いておく。
    // 内訳：
    //  ディレクトリ: 2 ブロック (3 エントリー追加)
    //  ファイル: 12 ブロック (最大 2000 エントリーの中から最大 100 エントリーを書き換えるため、2000 / 170[entries/block] ≒ 12)
    //  マージン: 2 ブロック
    FileSystem::JournalInfo journalInfo = {0, DeliveryCacheStorageManager::StorageJournalSize - BlockSize * 16};

    for (int i = 0; i < count; i++)
    {
        const ListFile& entry = m_Files[indexes[i]];

        NN_DETAIL_BCAT_INFO("[bcat] Copy file(%s/%s) ...\n", m_DirMeta.name.value, entry.name.value);

        char srcPath[160] = {};
        MakeDownloadFilePath(srcPath, sizeof (srcPath), m_AppId, m_DirMeta.name.value, entry.name.value);

        char destPath[160] = {};
        DeliveryCacheStorageManager::GetInstance().MakeFilePath(destPath, sizeof (destPath),
            m_AppId, m_DirMeta.name.value, entry.name.value);

        NN_RESULT_DO(FileSystem::CopyFileWithBuffer(&journalInfo, srcPath, destPath, journalInfo, buffer, bufferSize, m_pCancelEvent));

        FileMeta meta = {};

        meta.name = entry.name;
        meta.dataId = entry.dataId;
        meta.size = entry.size;
        meta.digest = entry.digest;

        NN_RESULT_DO(m_FileMetaView.AddOrUpdate(meta));
    }

    // 未コミットのファイルがあればコミットする。
    if (journalInfo.written > 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(m_AppId));
    }

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::RemoveUnstableDirectory() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DirectorySynchronizer::RemoveUnstableDirectory ...\n");

    char path[160] = {};
    DeliveryCacheStorageManager::GetInstance().MakeDirectoryPath(path, sizeof (path), m_AppId, m_DirMeta.name.value);

    NN_RESULT_TRY(nn::fs::DeleteDirectoryRecursively(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(DirectoryMetaView::LoadFromDeliveryCacheStorage(&m_DirectoryMetaView, m_AppId));
    NN_RESULT_DO(m_DirectoryMetaView.Remove(m_DirMeta.name.value));
    NN_RESULT_DO(DirectoryMetaView::SaveToDeliveryCacheStorage(m_DirectoryMetaView, m_AppId, false));

    NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(m_AppId));

    NN_RESULT_DO(DeleteTransaction());

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::LoadTransaction() NN_NOEXCEPT
{
    char path[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeTransactionPath(path, sizeof (path), m_AppId);

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    DirectoryMeta dirMeta = {};

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, &dirMeta, sizeof (dirMeta)));

    NN_RESULT_THROW_UNLESS(dirMeta.name.IsValid(), ResultSaveVerificationFailed());

    m_DirMeta = dirMeta;

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::CreateTransaction() NN_NOEXCEPT
{
    char path[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeTransactionPath(path, sizeof (path), m_AppId);

    {
        nn::fs::FileHandle handle = {};

        NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (m_DirMeta)));
                NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &m_DirMeta, sizeof (m_DirMeta),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(m_AppId));

    NN_RESULT_SUCCESS;
}

nn::Result DirectorySynchronizer::DeleteTransaction() NN_NOEXCEPT
{
    char path[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeTransactionPath(path, sizeof (path), m_AppId);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(m_AppId));

    NN_RESULT_SUCCESS;
}

bool DirectorySynchronizer::IsCanceled() NN_NOEXCEPT
{
    return (m_pCancelEvent && m_pCancelEvent->TryWait());
}

}}}}}
