﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_DeliveryListDownloader.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    const char* Fqdn = "bcat-list-%.cdn.nintendo.net";
}

namespace
{
    struct ProgressCallbackParam
    {
        nn::nifm::NetworkConnection* pConnection;
        nn::os::Event* pCancelEvent;
    };
}

namespace
{
    bool ProgressCallback(int64_t now, int64_t total, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(total);

        ProgressCallbackParam* cp = reinterpret_cast<ProgressCallbackParam*>(param);

        return (cp->pConnection->IsAvailable() && !(cp->pCancelEvent && cp->pCancelEvent->TryWait()));
    }

    void MakeDownloadListPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/list.msgpack", appId.value);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }

    void MakeDownloadListMetaPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/list.msgpack.bafmeta", appId.value);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }
}

DeliveryListDownloader::DeliveryListDownloader() NN_NOEXCEPT :
    m_Mutex(true),
    m_AppId(nn::ApplicationId::GetInvalidId()),
    m_pCancelEvent(nullptr)
{
}

void DeliveryListDownloader::SetCancelEvent(nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCancelEvent);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_pCancelEvent = pCancelEvent;
}

nn::Result DeliveryListDownloader::Download(nn::ApplicationId appId, const char* passphrase, nn::nifm::NetworkConnection* pConnection,
    bool enableProviderSpecificDelivery, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(passphrase);
    NN_SDK_REQUIRES_NOT_NULL(pConnection);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_BCAT_INFO("[bcat] DeliveryListDownloader::Download(%016llx) ...\n", appId.value);

    m_AppId = appId;

    nn::util::Strlcpy(m_Passphrase, passphrase, sizeof (m_Passphrase));

    NN_RESULT_DO(StepDownload(pConnection, enableProviderSpecificDelivery, buffer, bufferSize));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryListDownloader::StepDownload(nn::nifm::NetworkConnection* pConnection,
    bool enableProviderSpecificDelivery, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryListDownloader::StepDownload ...\n");

    nn::nifm::NetworkProfileData profile = {};

    NN_RESULT_TRY(nn::nifm::GetCurrentNetworkProfile(&profile))
        NN_RESULT_CATCH_ALL
        {
            // 現在の接続設定が取得できない場合、切断扱いとする。
            NN_RESULT_THROW(ResultInternetRequestCanceled());
        }
    NN_RESULT_END_TRY;

    char queryString[64] = {};

    if (profile.networkProfileType == nn::nifm::NetworkProfileType_SsidList)
    {
        NN_DETAIL_BCAT_INFO("[bcat] SSID list connection (%s)\n", profile.name);

        if (enableProviderSpecificDelivery)
        {
            // u=<uuid>
            int offset = nn::util::Strlcpy(queryString, "u=", sizeof (queryString));

            profile.id.ToString(&queryString[offset], sizeof (queryString) - offset);
        }
    }

    TopicId topicId = {};
    ParameterConverter::Convert(&topicId, m_AppId);

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url), "https://%s/api/nx/v1/list/%s%s%s",
        Fqdn, topicId.value, queryString[0] ? "?" : "", queryString);

    char dlListPath[64] = {};
    MakeDownloadListPath(dlListPath, sizeof (dlListPath), m_AppId);

    char dlListMetaPath[64] = {};
    MakeDownloadListMetaPath(dlListMetaPath, sizeof (dlListMetaPath), m_AppId);

    nn::bcat::service::ETag eTag = {};

    NN_RESULT_DO(LoadETag(&eTag));

    ProgressCallbackParam callbackParam = {pConnection, m_pCancelEvent};

    m_ArchiveDownloader.Reset();
    m_ArchiveDownloader.SetPath(dlListPath, dlListMetaPath);
    m_ArchiveDownloader.SetPassphrase(m_AppId, m_Passphrase);
    m_ArchiveDownloader.SetETagForIfNoneMatch(eTag.value);
    m_ArchiveDownloader.SetProgressCallback(ProgressCallback, &callbackParam);
    m_ArchiveDownloader.SetJournalSize(DownloadStorageJournalSize);
    m_ArchiveDownloader.SetFileWorkBuffer(buffer, bufferSize);

    NN_DETAIL_BCAT_INFO("[bcat] Url = %s\n", url);

    if (eTag.value[0])
    {
        NN_DETAIL_BCAT_INFO("[bcat] ETag = %s\n", eTag.value);
    }

    NN_RESULT_TRY(m_ArchiveDownloader.Download(url, m_pCancelEvent))
        NN_RESULT_CATCH(ResultServerError304)
        {
            NN_DETAIL_BCAT_INFO("[bcat] The list is not modified. (status 304)\n");
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH(ResultServerError404)
        {
            NN_DETAIL_BCAT_WARN("[bcat] The list is not found. (status 404)\n");
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(ResultServerError410)
        {
            NN_DETAIL_BCAT_WARN("[bcat] Service expired. (status 410)\n");
            NN_RESULT_THROW(ResultPlatformServiceExpired());
        }
        NN_RESULT_CATCH(ResultDataVerificationFailed)
        {
            NN_DETAIL_BCAT_WARN("[bcat] The list verification failed.\n");
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH(nn::fs::ResultDataCorrupted)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceledByUser());
            NN_RESULT_THROW_UNLESS(pConnection->IsAvailable(), ResultInternetRequestCanceled());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));

    NN_RESULT_DO(SaveListAndETag(buffer, bufferSize));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryListDownloader::LoadETag(nn::bcat::service::ETag* out) NN_NOEXCEPT
{
    char eTagPath[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeETagPath(eTagPath, sizeof (eTagPath), m_AppId);

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, eTagPath, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            out->value[0] = '\0';
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    size_t size;
    NN_RESULT_DO(nn::fs::ReadFile(&size, handle, 0, out->value, sizeof (out->value) - 1));

    out->value[size] = '\0';

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryListDownloader::SaveListAndETag(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    char srcListPath[64] = {};
    MakeDownloadListPath(srcListPath, sizeof (srcListPath), m_AppId);

    char destListPath[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeListPath(destListPath, sizeof (destListPath), m_AppId);

    // ジャーナルサイズは管理テーブル分を引いておく。
    // 内訳：
    //  ファイル: 2 ブロック (2 エントリー書き換え)
    FileSystem::JournalInfo journalInfo = {0, DeliveryCacheStorageManager::StorageJournalSize - BlockSize * 2};

    NN_RESULT_DO(FileSystem::CopyFileWithBuffer(&journalInfo, srcListPath, destListPath, journalInfo, buffer, bufferSize));

    if (journalInfo.written > 0)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(m_AppId));
    }

    const nn::bcat::service::ETag& eTag = m_ArchiveDownloader.GetETag();

    char eTagPath[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeETagPath(eTagPath, sizeof (eTagPath), m_AppId);

    int size = nn::util::Strnlen(eTag.value, sizeof (eTag.value));

    if (size == 0)
    {
        NN_RESULT_TRY(nn::fs::DeleteFile(eTagPath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;
    }
    else
    {
        nn::fs::FileHandle handle = {};

        NN_RESULT_TRY(nn::fs::OpenFile(&handle, eTagPath, nn::fs::OpenMode_Write))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(FileSystem::CreateFile(eTagPath, size));
                NN_RESULT_DO(nn::fs::OpenFile(&handle, eTagPath, nn::fs::OpenMode_Write));
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::SetFileSize(handle, size));

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, eTag.value, size,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(m_AppId));

    NN_RESULT_SUCCESS;
}

bool DeliveryListDownloader::IsCanceled() NN_NOEXCEPT
{
    return (m_pCancelEvent && m_pCancelEvent->TryWait());
}

}}}}}
