﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/bcat_DeliveryCacheFileService.generic.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheStorageService.generic.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <mutex>
#include <nn/util/util_FormatString.h>

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        if (size == 0 || size > maxSize || value[size - 1] != '\0') \
        {                                                           \
            NN_RESULT_THROW(ResultInvalidArgument());               \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace bcat { namespace detail { namespace service {

DeliveryCacheFileServiceForGeneric::DeliveryCacheFileServiceForGeneric(nn::sf::SharedPointer<DeliveryCacheStorageServiceForGeneric>&& parent) NN_NOEXCEPT :
    m_Mutex(true),
    m_Parent(std::move(parent)),
    m_IsOpened(false)
{
}

DeliveryCacheFileServiceForGeneric::~DeliveryCacheFileServiceForGeneric() NN_NOEXCEPT
{
    if (m_IsOpened)
    {
        nn::fs::CloseFile(m_Handle);
    }
}

nn::Result DeliveryCacheFileServiceForGeneric::Open(const nn::bcat::DirectoryName& dirName, const nn::bcat::FileName& fileName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(dirName.IsValid(), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(fileName.IsValid(), ResultInvalidArgument());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(!m_IsOpened, ResultAlreadyOpened());

    char path[80] = {};
    nn::util::SNPrintf(path, sizeof (path), "bcat-dc:/%s/%s", dirName.value, fileName.value);

    NN_RESULT_TRY(nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultNotFound());
        }
    NN_RESULT_END_TRY;

    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheFileServiceForGeneric::Read(nn::sf::Out<std::uint64_t> outSize, std::int64_t offset, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    size_t actualSize = 0;

    NN_RESULT_DO(nn::fs::ReadFile(&actualSize, m_Handle, offset, buffer.GetPointerUnsafe(), buffer.GetSize()));

    *outSize = static_cast<uint64_t>(actualSize);

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheFileServiceForGeneric::GetSize(nn::sf::Out<std::int64_t> outSize) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    NN_RESULT_DO(nn::fs::GetFileSize(outSize.GetPointer(), m_Handle));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheFileServiceForGeneric::GetDigest(nn::sf::Out<nn::bcat::Digest> outDigest) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    // ダイジェスト値はサポートしない。
    std::memset(outDigest->value, 0, sizeof (outDigest->value));

    NN_RESULT_SUCCESS;
}

}}}}
