﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/bcat_BcatService.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheProgressService.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryCacheStorageManager.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryCacheProgressManager.h>
#include <nn/bcat/detail/service/core/bcat_BackgroundWorkerThread.h>
#include <nn/bcat/detail/service/core/bcat_PassphraseManager.h>
#include <nn/bcat/detail/service/core/bcat_PushNotificationLogManager.h>
#include <nn/bcat/detail/service/core/bcat_SessionManager.h>
#include <nn/bcat/detail/service/core/bcat_TaskManager.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/arp/arp_Api.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        if (size == 0 || size > maxSize || value[size - 1] != '\0') \
        {                                                           \
            NN_RESULT_THROW(ResultInvalidArgument());               \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace bcat { namespace detail { namespace service {

BcatService::BcatService(const Capability& capability) NN_NOEXCEPT :
    m_Capability(capability),
    m_AppId(nn::ApplicationId::GetInvalidId()),
    m_AppVersion(0),
    m_IsResumeRequired(false)
{
    detail::service::core::SessionManager::GetInstance().Register(&m_SessionId);
}

BcatService::BcatService(nn::ApplicationId appId, uint32_t appVersion, const Capability& capability) NN_NOEXCEPT :
    m_Capability(capability),
    m_AppId(appId),
    m_AppVersion(appVersion),
    m_IsResumeRequired(true)
{
    detail::service::core::SessionManager::GetInstance().Register(&m_SessionId);

    detail::service::core::BackgroundWorkerThread::RegisterBackgroundTask(m_AppId, m_AppVersion);

    // アプリケーション起動中にバックグラウンドタスクが実行されないよう、登録直後にサスペンド状態にする。
    detail::service::core::BackgroundWorkerThread::SuspendBackgroundTask(m_AppId);
}

BcatService::~BcatService() NN_NOEXCEPT
{
    if (m_AppId != nn::ApplicationId::GetInvalidId())
    {
        detail::service::core::PassphraseManager::GetInstance().RemoveTemporary(m_AppId);
    }
    if (m_IsResumeRequired)
    {
        // コンストラクタでサスペンド状態にしたので、アプリケーション終了時に再開する。
        detail::service::core::BackgroundWorkerThread::ResumeBackgroundTask(m_AppId);
    }

    // セッションに紐付くタスクが実行キューに存在しないことを保証するため、先に削除する。
    detail::service::core::SessionManager::GetInstance().Unregister(m_SessionId);

    detail::service::core::BackgroundWorkerThread::CancelTask(m_SessionId);
}

nn::Result BcatService::RequestSyncDeliveryCache(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheProgressService>> outService) NN_NOEXCEPT
{
    // アプリケーション情報が取得できないクライアントからの要求はエラーにする。
    NN_RESULT_THROW_UNLESS(m_AppId != nn::ApplicationId::GetInvalidId(), ResultInvalidOperation());

    TaskId taskId = InvalidTaskId;

    NN_RESULT_DO(detail::service::core::SessionManager::GetInstance().AddForegroundTask(&taskId, m_SessionId, m_AppId, m_AppVersion));
    detail::service::core::BackgroundWorkerThread::CancelCurrentBackgroundTask();

    NN_RESULT_DO(CreateDeliveryCacheProgressService(outService, taskId));

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::RequestSyncDeliveryCacheWithDirectoryName(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheProgressService>> outService, const nn::bcat::DirectoryName& dirName) NN_NOEXCEPT
{
    // アプリケーション情報が取得できないクライアントからの要求はエラーにする。
    NN_RESULT_THROW_UNLESS(m_AppId != nn::ApplicationId::GetInvalidId(), ResultInvalidOperation());

    NN_RESULT_THROW_UNLESS(dirName.IsValid(), ResultInvalidArgument());

    TaskId taskId = InvalidTaskId;

    NN_RESULT_DO(detail::service::core::SessionManager::GetInstance().AddForegroundTask(&taskId, m_SessionId, m_AppId, m_AppVersion, dirName.value));
    detail::service::core::BackgroundWorkerThread::CancelCurrentBackgroundTask();

    NN_RESULT_DO(CreateDeliveryCacheProgressService(outService, taskId));

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::CancelSyncDeliveryCacheRequest() NN_NOEXCEPT
{
    // セッションに紐付くタスクが実行キューに存在しないことを保証するため、先にキャンセルする。
    detail::service::core::SessionManager::GetInstance().CancelTask(m_SessionId);

    detail::service::core::BackgroundWorkerThread::CancelTask(m_SessionId);

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::RequestSyncDeliveryCacheWithApplicationId(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheProgressService>> outService, nn::ApplicationId appId, std::uint32_t appVersion) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_System);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    TaskId taskId = InvalidTaskId;

    NN_RESULT_DO(detail::service::core::SessionManager::GetInstance().AddForegroundTask(&taskId, m_SessionId, appId, appVersion));
    detail::service::core::BackgroundWorkerThread::CancelCurrentBackgroundTask();

    NN_RESULT_DO(CreateDeliveryCacheProgressService(outService, taskId));

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::RequestSyncDeliveryCacheWithApplicationIdAndDirectoryName(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheProgressService>> outService, nn::ApplicationId appId, std::uint32_t appVersion, const nn::bcat::DirectoryName& dirName) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_System);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(dirName.IsValid(), ResultInvalidArgument());

    TaskId taskId = InvalidTaskId;

    NN_RESULT_DO(detail::service::core::SessionManager::GetInstance().AddForegroundTask(&taskId, m_SessionId, appId, appVersion, dirName.value));
    detail::service::core::BackgroundWorkerThread::CancelCurrentBackgroundTask();

    NN_RESULT_DO(CreateDeliveryCacheProgressService(outService, taskId));

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::SetPassphrase(nn::ApplicationId appId, const nn::sf::InArray<char>& passphrase) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());
    CHECK_IPC_STRING(passphrase, PassphraseLengthMax + 1);

    NN_RESULT_DO(detail::service::core::DeliveryCacheStorageManager::GetInstance().RequestSavePassphrase(appId, passphrase.GetData()));

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::RegisterBackgroundDeliveryTask(nn::ApplicationId appId, std::uint32_t appVersion) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    detail::service::core::BackgroundWorkerThread::RegisterBackgroundTask(appId, appVersion);

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::UnregisterBackgroundDeliveryTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    detail::service::core::BackgroundWorkerThread::UnregisterBackgroundTask(appId);

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::BlockDeliveryTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    detail::service::core::BackgroundWorkerThread::BlockTask(appId);

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::UnblockDeliveryTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    detail::service::core::BackgroundWorkerThread::UnblockTask(appId);

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::EnumerateBackgroundDeliveryTask(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::bcat::TaskInfo>& outInfos) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    int actualCount = 0;

    detail::service::core::TaskManager::GetInstance().GetInfoList(&actualCount,
        outInfos.GetData(), static_cast<int>(outInfos.GetLength()));

    nn::ApplicationId appId = {};

    if (detail::service::core::BackgroundWorkerThread::GetCurrentBackgroundTaskApplicationId(&appId))
    {
        // TORIAEZU: 「現在実行中のタスク取得 API」を作るまでの繋ぎ。
        for (int i = 0; i < actualCount; i++)
        {
            if (appId == outInfos[i].appId)
            {
                outInfos[i].status = TaskStatus_Running;
                break;
            }
        }
    }

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::GetDeliveryList(nn::sf::Out<std::uint64_t> outSize, const nn::sf::OutBuffer& buffer, nn::ApplicationId appId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    NN_RESULT_DO(detail::service::core::DeliveryCacheStorageManager::GetInstance().Mount(appId));

    NN_UTIL_SCOPE_EXIT
    {
        detail::service::core::DeliveryCacheStorageManager::GetInstance().Unmount(appId);
    };

    char path[64] = {};
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeListPath(path, sizeof (path), appId);

    {
        nn::fs::FileHandle handle = {};

        NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultNotFound());
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        size_t actualSize = 0;

        NN_RESULT_DO(nn::fs::ReadFile(&actualSize, handle, 0, buffer.GetPointerUnsafe(), buffer.GetSize()));

        *outSize = static_cast<uint64_t>(actualSize);
    }

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::ClearDeliveryCacheStorage(nn::ApplicationId appId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    NN_RESULT_THROW_UNLESS(appId != nn::ApplicationId::GetInvalidId(), ResultInvalidArgument());

    NN_RESULT_DO(detail::service::core::DeliveryCacheStorageManager::GetInstance().Mount(appId));

    NN_UTIL_SCOPE_EXIT
    {
        detail::service::core::DeliveryCacheStorageManager::GetInstance().Unmount(appId);
    };

    // アプリケーションを起動しなくてもクリア後に即時同期要求ができるようにするため、パスフレーズのみ残しておく。

    char path[64] = {};

    // bcat-dc:/directories/*
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeRootDirectoryPath(path, sizeof (path), appId);

    NN_RESULT_TRY(nn::fs::DeleteDirectoryRecursively(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    // bcat-dc:/list.msgpack
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeListPath(path, sizeof (path), appId);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    // bcat-dc:/etag.bin
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeETagPath(path, sizeof (path), appId);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    // bcat-dc:/directories.meta
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeDirectoryMetaPath(path, sizeof (path), appId);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    // bcat-dc:/index.lock
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeTransactionPath(path, sizeof (path), appId);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    // bcat-dc:/na_required
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeNintendoAccountRequiredPath(path, sizeof (path), appId);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::service::core::DeliveryCacheStorageManager::GetInstance().Commit(appId));

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::GetPushNotificationLog(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::bcat::PushNotificationLog>& outLogs) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    int actualCount = 0;

    detail::service::core::PushNotificationLogManager::GetInstance().GetList(&actualCount,
        outLogs.GetData(), static_cast<int>(outLogs.GetLength()));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result BcatService::CreateDeliveryCacheProgressService(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheProgressService>> outService, TaskId taskId) NN_NOEXCEPT
{
    auto& allocator = ServiceMemoryManager::GetInstance().GetDeliveryCacheProgressServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IDeliveryCacheProgressService, DeliveryCacheProgressService>(&allocator, m_SessionId, taskId, m_Capability);

    if (!p)
    {
        detail::service::core::DeliveryCacheProgressManager::GetInstance().Unregister(taskId);

        detail::service::core::SessionManager::GetInstance().RemoveTask(m_SessionId, taskId);

        NN_RESULT_THROW(ResultOutOfSessionResource());
    }

    outService.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

}}}}
