﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/bcat_ShimLibraryGlobal.h>
#include <nn/bcat/bcat_Result.h>
#include <mutex>

namespace nn { namespace bcat { namespace detail {

ShimLibraryGlobal::ShimLibraryGlobal() NN_NOEXCEPT :
    m_Session(nullptr),
    m_DeliveryCacheStorage(nullptr),
    m_DeliveryCacheFileCount(0),
    m_DeliveryCacheDirectoryCount(0)
{
}

void ShimLibraryGlobal::Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }
}

detail::ipc::IBcatService* ShimLibraryGlobal::GetSession() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }

    return m_Session;
}

nn::Result ShimLibraryGlobal::MountDeliveryCacheStorage() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }

    NN_RESULT_THROW_UNLESS(!m_DeliveryCacheStorage, ResultAlreadyMounted());

    NN_RESULT_DO(detail::ipc::CreateDeliveryCacheStorageService(&m_DeliveryCacheStorage));

    NN_RESULT_SUCCESS;
}

nn::Result ShimLibraryGlobal::MountDeliveryCacheStorage(nn::ApplicationId appId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Session)
    {
        CreateIpcSession();
    }

    NN_RESULT_THROW_UNLESS(!m_DeliveryCacheStorage, ResultAlreadyMounted());

    NN_RESULT_DO(detail::ipc::CreateDeliveryCacheStorageService(&m_DeliveryCacheStorage, appId));

    NN_RESULT_SUCCESS;
}

void ShimLibraryGlobal::UnmountDeliveryCacheStorage() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_ABORT_UNLESS_NOT_NULL(m_DeliveryCacheStorage);

    NN_ABORT_UNLESS(m_DeliveryCacheFileCount == 0);
    NN_ABORT_UNLESS(m_DeliveryCacheDirectoryCount == 0);

    nn::sf::ReleaseSharedObject(m_DeliveryCacheStorage);
    m_DeliveryCacheStorage = nullptr;
}

bool ShimLibraryGlobal::IsDeliveryCacheStorageMounted() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return !!m_DeliveryCacheStorage;
}

nn::Result ShimLibraryGlobal::CreateFileService(detail::ipc::IDeliveryCacheFileService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_DeliveryCacheStorage, ResultNotMounted());

    nn::sf::SharedPointer<detail::ipc::IDeliveryCacheFileService> service = nullptr;

    NN_RESULT_DO(m_DeliveryCacheStorage->CreateFileService(&service));

    *outService = service.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result ShimLibraryGlobal::CreateDirectoryService(detail::ipc::IDeliveryCacheDirectoryService** outService) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_DeliveryCacheStorage, ResultNotMounted());

    nn::sf::SharedPointer<detail::ipc::IDeliveryCacheDirectoryService> service = nullptr;

    NN_RESULT_DO(m_DeliveryCacheStorage->CreateDirectoryService(&service));

    *outService = service.Detach();

    NN_RESULT_SUCCESS;
}

void ShimLibraryGlobal::IncrementDeliveryCacheFileCount() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_DeliveryCacheFileCount++;
}

void ShimLibraryGlobal::DecrementDeliveryCacheFileCount() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_SDK_ASSERT_GREATER(m_DeliveryCacheFileCount, 0);

    m_DeliveryCacheFileCount--;
}

void ShimLibraryGlobal::IncrementDeliveryCacheDirectoryCount() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_DeliveryCacheDirectoryCount++;
}

void ShimLibraryGlobal::DecrementDeliveryCacheDirectoryCount() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_SDK_ASSERT_GREATER(m_DeliveryCacheDirectoryCount, 0);

    m_DeliveryCacheDirectoryCount--;
}

nn::Result ShimLibraryGlobal::EnumerateDeliveryCacheDirectory(int* outCount, DirectoryName* outDirectoryNames, int count) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_ABORT_UNLESS_NOT_NULL(m_DeliveryCacheStorage);

    int32_t actualCount = 0;

    NN_RESULT_DO(m_DeliveryCacheStorage->EnumerateDeliveryCacheDirectory(&actualCount,
        nn::sf::OutArray<DirectoryName>(outDirectoryNames, count)));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

void ShimLibraryGlobal::CreateIpcSession() NN_NOEXCEPT
{
    detail::ipc::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::CreateBcatService(&m_Session));
}

}}}
