﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/bcat_ApiAdmin.h>
#include <nn/bcat/detail/bcat_ErrorHandler.h>
#include <nn/bcat/detail/bcat_ShimLibraryGlobal.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace bcat {

namespace
{
    nn::Result ResolveDeliveryCacheStorageSaveDataId(nn::fs::SaveDataId* outSaveDataId, nn::ApplicationId appId) NN_NOEXCEPT
    {
        nn::fs::SaveDataFilter filter = nn::fs::SaveDataFilter::Make(appId.value, nn::fs::SaveDataType::Bcat, util::nullopt, util::nullopt, util::nullopt);
        nn::fs::SaveDataInfo info;
        NN_RESULT_TRY(nn::fs::FindSaveDataWithFilter(&info, nn::fs::SaveDataSpaceId::User, filter))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_THROW(ResultStorageNotFound());
        }
        NN_RESULT_END_TRY

        *outSaveDataId = info.saveDataId;
        NN_RESULT_SUCCESS;
    }
}

nn::Result MountDeliveryCacheStorage(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    NN_DETAIL_BCAT_IPC_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().MountDeliveryCacheStorage(appId));

    NN_RESULT_SUCCESS;
}

nn::Result DeleteDeliveryCacheStorage(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_DETAIL_BCAT_IPC_RESULT_DO(session->BlockDeliveryTask(appId));

    NN_UTIL_SCOPE_EXIT
    {
        session->UnblockDeliveryTask(appId);
    };

    nn::fs::SaveDataId saveDataId = 0;
    NN_RESULT_DO(ResolveDeliveryCacheStorageSaveDataId(&saveDataId, appId));

    NN_RESULT_DO(nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::User, saveDataId));

    // セーブデータの削除を確認したら、バックグラウンド同期タスクの登録も解除する。
    NN_DETAIL_BCAT_IPC_RESULT_DO(session->UnregisterBackgroundDeliveryTask(appId));

    NN_RESULT_SUCCESS;
}

nn::Result RequestSyncDeliveryCache(DeliveryCacheProgress* outProgress, nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProgress);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    NN_ABORT_UNLESS(!detail::ShimLibraryGlobal::GetInstance().IsDeliveryCacheStorageMounted(),
        "[bcat] Delivery cache storage is mounted.");

    NN_RESULT_THROW_UNLESS(nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId()), ResultInternetRequestNotAccepted());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    nn::sf::SharedPointer<detail::ipc::IDeliveryCacheProgressService> service = nullptr;

    NN_DETAIL_BCAT_IPC_RESULT_DO(session->RequestSyncDeliveryCacheWithApplicationId(&service, appId, appVersion));

    outProgress->Attach(service.Detach());

    NN_RESULT_SUCCESS;
}

nn::Result RequestSyncDeliveryCache(DeliveryCacheProgress* outProgress, nn::ApplicationId appId, uint32_t appVersion,
    const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProgress);
    NN_SDK_REQUIRES_NOT_NULL(dirName);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    DirectoryName d = {};

    NN_RESULT_THROW_UNLESS(nn::util::Strlcpy(d.value, dirName, sizeof (d.value)) < sizeof (d.value), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(d.IsValid(), ResultInvalidArgument());

    NN_ABORT_UNLESS(!detail::ShimLibraryGlobal::GetInstance().IsDeliveryCacheStorageMounted(),
        "[bcat] Delivery cache storage is mounted.");

    NN_RESULT_THROW_UNLESS(nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId()), ResultInternetRequestNotAccepted());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    nn::sf::SharedPointer<detail::ipc::IDeliveryCacheProgressService> service = nullptr;

    NN_DETAIL_BCAT_IPC_RESULT_DO(session->RequestSyncDeliveryCacheWithApplicationIdAndDirectoryName(&service, appId, appVersion, d));

    outProgress->Attach(service.Detach());

    NN_RESULT_SUCCESS;
}

nn::Result SetPassphrase(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(passphrase);

    int length = nn::util::Strnlen(passphrase, PassphraseLengthMax + 1);

    NN_SDK_REQUIRES_MINMAX(length, 1, PassphraseLengthMax);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_DETAIL_BCAT_IPC_RESULT_DO(session->SetPassphrase(appId, nn::sf::InArray<char>(passphrase, static_cast<size_t>(length) + 1)));

    NN_RESULT_SUCCESS;
}

nn::Result RegisterBackgroundDeliveryTask(nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_DETAIL_BCAT_IPC_RESULT_DO(session->RegisterBackgroundDeliveryTask(appId, appVersion));

    NN_RESULT_SUCCESS;
}

nn::Result UnregisterBackgroundDeliveryTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_DETAIL_BCAT_IPC_RESULT_DO(session->UnregisterBackgroundDeliveryTask(appId));

    NN_RESULT_SUCCESS;
}

}}
