﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/audioctrl/audioctrl_PlayReport.h>

namespace nn { namespace audioctrl { namespace server { namespace detail {

// プレイレポート処理を管理するコントローラ
//
// ボリューム情報や出力先情報などの変更通知を受け取り、適切な頻度でプレイレポート向け情報の更新を行う機能を提供します。
// 例えば、ボリュームボタンの連打やヘッドホンジャックの微妙な位置での挿抜を繰り返された場合でも、
// 最後の状態の情報のみをプレイレポート向け情報として取得することが出来ます。
//
// Notify...() を呼び出して値の更新を通知すると、値はコミット保留状態として保存され、コミットタイマーが開始されます。
// コミットタイマーがシグナル状態になった時に、Commit() を呼び出すとコミット保留状態の値がコミット確定状態の値に反映されます。
// 外部からはコミット確定状態の値のみが取得可能です。
// また、更新が発生した場合は対応するイベントがシグナル状態になります。
class PlayReportController
{
    NN_DISALLOW_COPY( PlayReportController );
    NN_DISALLOW_MOVE( PlayReportController );

private:
    // コミット済みプレイレポート用音声出力先情報
    PlayReportAudioOutputTarget m_CommittedOutputTarget;

    // コミット済みプレイレポート用音量情報
    PlayReportAudioVolumeData   m_CommittedVolumeData;

    // 未コミットプレイレポート用音声出力先情報
    PlayReportAudioOutputTarget m_UncommittedOutputTarget;

    // 未コミットプレイレポート用音量情報
    PlayReportAudioVolume   m_UncommittedSpeakerVolume;
    PlayReportAudioVolume   m_UncommittedHeadphoneVolume;
    PlayReportAudioVolume   m_UncommittedUsbOutputDeviceVolume;
    bool m_UncommittedIsHeadphonePowerLimitted;

    // 一回以上通知されたかどうかのフラグ
    bool m_IsSpeakerVolumeNotified;
    bool m_IsHeadphoneVolumeNotified;
    bool m_IsHeadphonePowerLimitedNotified;
    bool m_IsUsbOutputDeviceVolumeNotified;
    bool m_IsOutputTargetNotified;

    // 通知イベントの保留の解除を行ったことがあるか
    bool m_IsReleasedHoldedEvents;

    // コミット時にシグナルするタイマーイベント
    nn::os::TimerEvent m_CommitTimer;

    // 何秒以上値が変化しなければコミットとするかの時間
    nn::TimeSpan m_CommitDelayTime;

    // 音声出力先情報の更新時にシグナルするシステムイベント
    nn::os::SystemEvent m_OutputTargetUpdateEvent;

    // 音量情報の更新時にシグナルするシステムイベント
    nn::os::SystemEvent m_VolumeUpdateEvent;

    // ミューテクス
    nn::os::Mutex m_Mutex;

    // 初期化済みフラグ
    bool m_IsInitialized;

    /**
    * @brief       コミットタイマーを開始します。
    */
    void StartCommitTimer() NN_NOEXCEPT;

    /**
    * @brief       与えられた 2 個の PlayReportAudioVolume が同一かどうかを返します。
    * @return      同一かどうか
    */
    static bool ComparePlayReportAudioVolumes(const PlayReportAudioVolume& volumeA, const PlayReportAudioVolume& volumeB) NN_NOEXCEPT;

    /**
    * @brief       更新イベントをシグナルすることを保留する必要があるかどうかを返します。
    * @return      更新イベントをシグナルすることを保留する必要があるか
    */
    bool ShouldHoldSignalUpdateEvent() const NN_NOEXCEPT;

public:
    /**
    * @brief       コンストラクタ
    */
    PlayReportController() NN_NOEXCEPT;

    /**
    * @brief       プレイレポートコントローラの初期化
    * @param[in]   commitDelayTime 何秒以上値が変化しなければコミットタイマーをシグナル状態にするかの時間
    * @pre
    *  - プレイレポートコントローラは未初期化
    * @post
    *  - プレイレポートコントローラは初期化済み
    *  - プレイレポートコントローラから取得できる全ての値の更新を検知するイベントは非シグナル状態
    *  - プレイレポートコントローラから取得できる全てのタイマーイベントは非シグナル状態かつ停止状態
    *  - コミット保留中・コミット済みの値は不定値
    * @details
    * 本 API を呼び出した後、Notify...() で値を設定した後、Commit() を呼び出して確定済み初期値を設定してください。
    * 設定せずにプレイレポート用の情報を取得する API を呼び出した場合、不定値が返ります。
    * コントローラは初期化カウンタを持たないので、多重に初期化することはできません。
    * 本 API はウェイク時の呼び出しは不要です。
    * 本 API はスレッドセーフです。
    */
    void Initialize(nn::TimeSpan commitDelayTime) NN_NOEXCEPT;

    /**
    * @brief       プレイレポートコントローラの終了
    * @pre         プレイレポートコントローラは初期化済み
    * @post        コミットタイマーは停止・非シグナル状態になります。
    * @details
    * 本 API 呼び出し時にコミットしていない情報を持っていても、コミット処理を内部で行うようなことはしません。
    * 本 API はスリープ時に呼び出しは不要です。
    * 本 API はスレッドセーフです。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief       初期化済みかどうかを返します。
    * @return      初期化済みか
    * @details
    * 本 API はスレッドセーフです。
    */
    bool IsInitialized() NN_NOEXCEPT;

    /**
    * @brief       プレイレポート用音声出力先情報を返します
    * @return      プレイレポート用音声出力先情報　
    * @pre
    *  - プレイレポートコントローラは初期化済み
    *  - 確定状態の値が存在する
    * @details
    * 本 API はスレッドセーフです。
    */
    PlayReportAudioOutputTarget GetAudioOutputTarget() NN_NOEXCEPT;

    /**
    * @brief       プレイレポート用音声情報情報を返します
    * @param[out]  pOutData プレイレポート用音声情報情報　
    * @pre
    *  - プレイレポートコントローラは初期化済み
    *  - 確定状態の値が存在する
    * @details
    * 本 API はスレッドセーフです。
    */
    void GetAudioVolumeData(PlayReportAudioVolumeData* pOutData) NN_NOEXCEPT;

    /**
    * @brief       音声出力先情報の更新時にシグナルするイベントの ReadableHandle を返します。
    * @param[out]  outHandle イベントの ReadableHandle
    * @pre         プレイレポートコントローラは初期化済み
    * @details
    * イベントは Commit() 呼び出し時にシグナル状態になります。
    * 本 API はスレッドセーフです。
    */
    void AcquireAudioOutputDeviceUpdateEvent(nn::os::NativeHandle* outHandle) NN_NOEXCEPT;

    /**
    * @brief       音量情報の更新時にシグナルするイベントの ReadableHandle を返します。
    * @param[out]  outHandle イベントの ReadableHandle
    * @pre         プレイレポートコントローラは初期化済み
    * @details
    * イベントは Commit() 呼び出し時にシグナル状態になります。
    * 本 API はスレッドセーフです。
    */
    void AcquireAudioVolumeUpdateEvent(nn::os::NativeHandle* outHandle) NN_NOEXCEPT;

    /**
    * @brief       音声出力先が変更されたことをプレイレポートコントローラに通知します。
    * @param[in]   target プレイレポート用音声出力先情報　
    * @pre         プレイレポートコントローラは初期化済み
    * @post
    *  - コミットタイマーが開始される
    *  - コミット保留中の値が更新される
    * @details
    * 本 API を呼び出すと値が変更されたかどうかに関わらずコミットタイマーが開始されます。
    * 値が変更されたかどうかの判定は Commit() 内で行われます。
    * 本 API はスレッドセーフです。
    */
    void NotifyOutputDeviceUpdate(PlayReportAudioOutputTarget target) NN_NOEXCEPT;

    /**
    * @brief       音量情報 (音量値・ミュート) が変更されたことをプレイレポートコントローラに通知します。
    * @param[in]   target プレイレポート用音声出力先情報　
    * @param[in]   volume プレイレポート用音量情報 (音量値・ミュート)
    * @pre         プレイレポートコントローラは初期化済み
    * @post
    *  - コミットタイマーが開始される
    *  - コミット保留中の値が更新される
    * @details
    * 本 API を呼び出すと値が変更されたかどうかに関わらずコミットタイマーが開始されます。
    * 値が変更されたかどうかの判定は Commit() 内で行われます。
    * 本 API はスレッドセーフです。
    */
    void NotifyAudioVolumeUpdateForVolume(PlayReportAudioOutputTarget target, PlayReportAudioVolume volume) NN_NOEXCEPT;

    /**
    * @brief       音量情報 (ヘッドホン出力制限) が変更されたことをプレイレポートコントローラに通知します。
    * @param[in]   target プレイレポート用音声出力先情報　
    * @param[in]   isHeadphonePowerLimited ヘッドホン出力制限情報
    * @pre         プレイレポートコントローラは初期化済み
    * @post
    *  - コミットタイマーが開始される
    *  - コミット保留中の値が更新される
    * @details
    * 本 API を呼び出すと値が変更されたかどうかに関わらずコミットタイマーが開始されます。
    * 値が変更されたかどうかの判定は Commit() 内で行われます。
    * 本 API はスレッドセーフです。
    */
    void NotifyAudioVolumeUpdateForHeadphonePowerLimit(bool isHeadphonePowerLimited) NN_NOEXCEPT;

    /**
    * @brief       コミット時にシグナルするタイマーを取得します。
    * @return      コミット時にシグナルするタイマー
    * @pre         プレイレポートコントローラは初期化済み
    * @details
    * 本 API はスレッドセーフです。
    */
    nn::os::TimerEvent& GetCommitTimer() NN_NOEXCEPT;

    /**
    * @brief       コミット時の処理を実行します。
    * @pre         プレイレポートコントローラは初期化済み
    * @post
    * - コミット保留中の値がコミット済みの値に反映される
    * - コミットタイマーのシグナル状態はクリア状態
    * - 値が更新された場合、対応するイベントはシグナル状態
    *   ただし、全ての値が一度以上コミットされていない状態ではシグナル状態になりません。
    *   一つでもイベントがシグナル状態になると、他の値も含めて取得されプレイレポートとして保存されてしまうためです。
    *   この場合、更新に伴うシグナル状態への遷移は保留となり、全ての値がコミット状態になったときにシグナル状態になります。
    * @details
    * 初期化時に初期値を設定する時、もしくは、GetCommitTimer() で取得したコミットタイマーがシグナル状態になった時に 本 API を呼び出してください。
    * 本 API はスレッドセーフです。
    */
    void Commit() NN_NOEXCEPT;
};

}}}}  // namespace nn::audioctrl::server::detail




