﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/ovln/format/ovln_AudioMessage.h>
#include "audioctrl_OverlaySenderController.h"

namespace nn { namespace audioctrl { namespace server { namespace detail {

void OverlaySenderController::Initialize() NN_NOEXCEPT
{
    // オーバレイ通知ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());

    // オーバレイ通知ライブラリの Sender の初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderForValue(&m_VolumeChangeSender));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderForValue(&m_MuteChangeSender));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderForValue(&m_AudioOutputTargetChangeSender));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderForValue(&m_HeadphoneWarningNoticeSender));
}

void OverlaySenderController::Finalize() NN_NOEXCEPT
{
    // オーバレイ通知ライブラリの Sender の終了
    nn::ovln::FinalizeSender(&m_VolumeChangeSender);
    nn::ovln::FinalizeSender(&m_MuteChangeSender);
    nn::ovln::FinalizeSender(&m_AudioOutputTargetChangeSender);
    nn::ovln::FinalizeSender(&m_HeadphoneWarningNoticeSender);

    // オーバレイ通知ライブラリの終了
    nn::ovln::FinalizeSenderLibraryForOverlay();
}

nn::ovln::SenderForOverlayType& OverlaySenderController::GetVolumeChangeMessageSender(VolumeChangeReason reason) NN_NOEXCEPT
{
    switch(reason)
    {
    case VolumeChangeReason::VolumeChanged:
        return m_VolumeChangeSender;
    case VolumeChangeReason::MuteChanged:
        return m_MuteChangeSender;
    case VolumeChangeReason::OutputTargetChanged:
        return m_HeadphoneWarningNoticeSender;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::ovln::format::AudioOutputTarget OverlaySenderController::ConvertToAudioOutputTarget(AudioTarget target) const NN_NOEXCEPT
{
    switch(target)
    {
    case nn::audioctrl::AudioTarget_Speaker:
        return nn::ovln::format::AudioOutputTarget_Speaker;
    case nn::audioctrl::AudioTarget_Headphone:
        return nn::ovln::format::AudioOutputTarget_HeadphoneMicJack;
    case nn::audioctrl::AudioTarget_Tv:
        return nn::ovln::format::AudioOutputTarget_Hdmi;
    case nn::audioctrl::AudioTarget_UsbOutputDevice:
        return nn::ovln::format::AudioOutputTarget_UsbOutputDevice;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void OverlaySenderController::SendVolumeChangeMessage(int volume, bool mute, AudioTarget target, VolumeChangeReason reason, VolumeChangeDetailReasonFlagSet detailReason) NN_NOEXCEPT
{
    nn::ovln::format::AudioVolumeChangedData data = {};
    data.volume = static_cast<int8_t>(volume);
    data.mute = mute;
    data.audioOutputTarget = static_cast<uint8_t>(ConvertToAudioOutputTarget(target));
    data.reason.Set<nn::ovln::format::AudioVolumeChangeReasonFlag::upButtonPressed>(detailReason.Test<VolumeChangeDetailReasonFlag::upButtonPressed>());
    data.reason.Set<nn::ovln::format::AudioVolumeChangeReasonFlag::downButtonPressed>(detailReason.Test<VolumeChangeDetailReasonFlag::downButtonPressed>());
    data.reason.Set<nn::ovln::format::AudioVolumeChangeReasonFlag::upAndDownButtonPressed>(detailReason.Test<VolumeChangeDetailReasonFlag::upAndDownButtonPressed>());
    data.reason.Set<nn::ovln::format::AudioVolumeChangeReasonFlag::headphoneMicJackPlugged>(detailReason.Test<VolumeChangeDetailReasonFlag::headphoneMicJackPlugged>());
    data.reason.Set<nn::ovln::format::AudioVolumeChangeReasonFlag::headphoneMicJackUnplugged>(detailReason.Test<VolumeChangeDetailReasonFlag::headphoneMicJackUnplugged>());
    data.reason.Set<nn::ovln::format::AudioVolumeChangeReasonFlag::usbOutputDeviceConnected>(detailReason.Test<VolumeChangeDetailReasonFlag::usbOutputDeviceConnected>());
    data.reason.Set<nn::ovln::format::AudioVolumeChangeReasonFlag::usbOutputDeviceDisconnected>(detailReason.Test<VolumeChangeDetailReasonFlag::usbOutputDeviceDisconnected>());

    nn::ovln::Message message = {};
    message.tag = nn::ovln::format::AudioVolumeChangedDataTag;
    message.dataSize = sizeof(data);
    std::memcpy(&message.data, &data, sizeof(data));

    auto sender = GetVolumeChangeMessageSender(reason);
    nn::ovln::Send(&sender, message);
}

void OverlaySenderController::SendHeadphoneVolumeWarningNoticeMessage() NN_NOEXCEPT
{
    nn::ovln::Message message = {};
    message.tag = nn::ovln::format::AudioHeadphoneVolumeWarningDataTag;
    nn::ovln::Send(&m_HeadphoneWarningNoticeSender, message);
}

}}}}  // namespace nn::audioctrl::server::detail
